/*
 * Copyright (C) 1992 by CERN/CN/SW/DC
 * All rights reserved
 */

#ifndef lint
static char sccsid[] = "@(#)disk.c	3.11 04/04/95 CERN-SW/DC Fabrizio Cane";
#endif /* not lint */

#include <stdio.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#if defined( cray ) || defined( apollo ) || defined( sgi ) || ( defined( _AIX ) && defined( _IBMR2 ) )
#include <sys/statfs.h>
#endif

#if defined (SOLARIS)
#include <sys/statvfs.h>
#endif

#if defined ( SOLARIS )
#include <string.h>
#endif
#if (defined( _AIX ) && defined( _IBMESA )) || ( defined(__osf__) && defined(__alpha) )
#include <sys/mount.h>
#endif
#if defined( sun ) || defined( hpux )
#include <sys/vfs.h>
#endif
#if defined( ultrix )
#include <sys/param.h>
#include <sys/mount.h>
#endif
#ifdef RFIO
#include <rfio.h>
#endif /* RFIO */
#if defined min
/*
 * This terrible hack, because some of the next includes define
 * struct elem as "min"
 */
#undef min
#endif /* min */
#include <shift_types.h>
#include <strutil.h>
#include <parser.h>
#include <table.h>
#include <dpmutil.h>

#define PATHSLASH		'/'

/*
 *  Create on behalf of root a directory and set the mode and owner
 *
 *  INPUT
 *	path		complete nfs-pathname of the directory to be created
 *	mode		directory mode
 *	uid,gid		directory owner and group id
 *
 *  RETURN
 *	1		directory successfully created
 *	0		directory exists
 *	-1		an error occured
 */
int create_dirname(path,mode,uid,gid)
char *path;
int mode;
int uid,gid;
{
  int mask,reply;

 /*
  *  Create the directory
  */
    mask = umask(0);
    reply = mkdir(path,mode);
    umask(mask);

 /*
  *  Check the mkdir() reply ( umask() does not change errno )
  */
    if ( reply < 0 )
	switch ( errno ) {
	    case EEXIST :
		return(0);
	    default :
		vperror(1,"mkdir(%s,%o)",path,mode);
		return(-1);
	}

 /*
  *  Set the directory owner and group id
  */
    if ( chown(path,uid,gid) < 0 ) {
	vperror(1,"chown(%s,%d,%d)",path,uid,gid);
	return(-1);
    }

    return(1);
}

/*
 *  Create all missing directories that are components of the file path
 *
 *  INPUT
 *	file		complete nfs-pathname of the file
 *	gridx		index in the nfs-pathname of the group directory
 *	clientuid	uid of the user calling sfrm
 *	pathuid		uid of the user whose home directory is in the nfs-pathname
 *	pathgid		gid of the users specified by pathuid and clientuid
 *
 *  RETURN
 *	1		directories successfully created
 *	-1		an error occured
 */
int create_dirpath(file,gridx,clientuid,pathuid,pathgid)
char *file;
int gridx;
int clientuid;
int pathuid,pathgid;
{
  char *dir;
  int mode = 0755; /* rwxr-xr-x */
  int rootuid = 0;

 /*
  *  Create the group name directory
  */
    dir = file+gridx+1;
    if ( (dir = (char*)strchr(dir,PATHSLASH)) == NULL ) exception(eformat);
    *dir = EOS;
    if ( create_dirname(file,mode,rootuid,pathgid) < 0 ) exception(ecreate);
    *dir = PATHSLASH;

 /*
  *  Create the user name directory
  */
    if ( (dir = (char*)strchr(dir+1,PATHSLASH)) == NULL ) exception(eformat);
    *dir = EOS;
    if ( create_dirname(file,mode,pathuid,pathgid) < 0 ) exception(ecreate);
    *dir = PATHSLASH;

 /*
  *  Create the user's subdirectories
  */
    while ( (dir = (char*)strchr(dir+1,PATHSLASH)) != NULL ) {
	*dir = EOS;
	if ( create_dirname(file,mode,clientuid,pathgid) < 0 ) exception(ecreate);
	*dir = PATHSLASH;
    }

    return(1);

handle_exception(eformat) : vperror(1,"internal: wrong path format in create_dirpath()");
handle_exception(ecreate) : if ( dir ) *dir = PATHSLASH;
			    return(-1);
}

/*
 *  Create a file on disk on behalf of the client
 *
 *  INPUT
 *	file		complete nfs-pathname of the file to be created
 *	gridx		index in the nfs-pathname of the group directory
 *	clientuid	uid of the user calling sfrm
 *	clientgid	gid of the user specified by clientuid
 *	pathuid		uid of the user whose home directory is in the nfs-pathname
 *	pathgid		gid of the user specified by pathuid
 *
 *  RETURN
 *	1		file successfully created
 *	0		file exists
 *	-1		an error occured
 */
int create_file(file,gridx,clientuid,clientgid,pathuid,pathgid)
char *file;
int gridx;
int clientuid,clientgid;
int pathuid,pathgid;
{
  int flags = O_WRONLY | O_CREAT | O_EXCL ;
  int mode = 0777; /* rwxrwxrwx */
  boolean subdir;
  int fd;

 /*
  *  Check the client group be correct
  */
    if ( clientgid != pathgid ) {
	vperror(0,"client group doesn't match user group");
	return(-1);
    }

 /*
  *  Create the file ( and the sub-directories if needed ) 
  */
    do {
	if ( (fd = open(file,flags,mode)) < 0 && (subdir = (errno == ENOENT)) &&
		(subdir = create_dirpath(file,gridx,clientuid,pathuid,pathgid)) < 0 )
	    return(-1);
    } while ( fd < 0 && subdir );

    if ( fd < 0 )
	switch ( errno ) {
	 /*
	  *  File exists ( because of O_EXCL and O_CREAT flags )
	  */
	    case EEXIST :
		return(0);
	 /*
	  *  open() failure
	  */
	    default :
		vperror(1,"open(%s,%x,%o)",file,flags,mode);
		return(-1);
	}

 /*
  *  Set the file owner and group id
  */
    if ( chown(file,clientuid,clientgid) < 0 ) {
	vperror(1,"chown(%s,%d,%d)",file,clientuid,clientgid);
	return(-1);
    }

 /*
  *  Close the file successfully created
  */
    close(fd);

    return(1);
}

/*
 *  Remove a file on disk on behalf of root
 *
 *  INPUT
 *	nfspath		the complete nfs-pathname of the file to be removed
 *
 *  RETURN
 *	1		file removed
 *	0		file does NOT exist
 *	-1		an error occured
 */
int remove_file(nfspath)
char *nfspath;
{
    if ( unlink(nfspath) < 0 )
	switch ( errno ) {
	    case ENOENT :
			return(0);
	    case ESTALE :
	    case ETIMEDOUT :
	    default :
			return(-1);
	}

    return(1);
}

/*
 *  Check on behalf of root that a given file be on disk
 *
 *  INPUT
 *	nfspath		the complete nfs-pathname of the file
 *
 *  OUTPUT
 *	buffer		contains the information about the existing file
 *
 *  RETURN
 *	1		the file exists
 *	0		the file does NOT exist
 *	-1		an error occured
 */
int is_file_there(nfspath,buffer)
char *nfspath;
struct stat *buffer;
{
    if ( lstat(nfspath,buffer) < 0 )
	switch ( errno ) {
	    case ENOENT :
			return(0);
	    case ESTALE :
	    case ETIMEDOUT :
	    default :
			return(-1);
	}

    return(1);
}

#define ROOTPATH	"/"

/*
 *  Check that a shift file system be locally mounted
 *
 *  INPUT
 *	path		file system path ( different from ROOTPATH )
 *	buffer		information about a file or directory in the file system
 *
 *  RETURN
 *	1		the file system is locally mounted
 *	0		the file system is locally unmounted
 *	-1		an error occured
 *
 *  DESCRIPTION
 *	It assumes the shift file systems be locally mounted on ROOTPATH subdirectories.
 *	Thus it compares the device numbers of the ROOTPATH and the input file system.
 */
int is_device_mounted(path,buffer)
char *path;
struct stat *buffer;
{
  static dev_t root_dev 	= -1 ;
  static struct stat local;

 /*
  *  Get the root device number
  */
    if ( root_dev < 0 ) {
	stat(ROOTPATH,&local);
	root_dev = local.st_dev;
    }

 /*
  *  Get the device number of the input file system
  */
    if ( buffer == NULL )
	if ( stat(path,&local) < 0 )
	    switch ( errno ) {
		case ENOENT :
		case ENOTDIR :
			return(0);
		default :
			return(-1);
	    }
	else
	    return( local.st_dev != root_dev );
    else
	return( buffer->st_dev != root_dev );
}

/*
 *  Check a file system be accessible and locally mounted
 *
 *  INPUT
 *	path		file system path
 *	buffer		information about a file or directory in the file system path
 *	errno		errno set by a previous system call 
 *
 *  RETURN
 *	FSMNTD		file system is locally mounted and accessible
 *	FSUMNTD		file system in locally unmounted
 *	FSUNACC		file system is not remotely accessible
 *	FSERROR		other file system failure
 */
int is_filesystem(path,buffer,lerrno)
char *path;
struct stat *buffer;
int lerrno;
{
  for(;;)
    switch ( lerrno ) {
	case ETIMEDOUT :	/* no answer from the remote host 	*/
	case ESTALE :	 	/* remote host is down 			*/
				/* file system not remotely mounted 	*/
		return(FSUNACC);

	case ENOENT :		/* no such file or directory		*/
	case ENOTDIR :		/* a component is not a directory	*/
	case 0 :		/* no error				*/
		switch ( is_device_mounted(path,buffer) ) {
		    case 1  : return(FSMNTD);
#ifndef RFIO
		    case 0  : return(FSUMNTD);
#else
		    case 0  : return(FSMNTD);
#endif
		    case -1 : lerrno = errno; continue;
		}
	default :		/* other errors				*/
				/* I/O error while reading or writing	*/
	    return(FSERROR);
    }
}

/*
 *  Print an error message related to the is_filesystem() reply
 *
 *  INPUT
 *	path		file system path
 *	reply		is_filesystem()'s reply
 *	errno		is_filesystem()'s errno parameter
 */
void filesystem_error(path,reply,lerrno)
char *path;
int reply;
int lerrno;
{
    switch ( reply ) {
	case FSUMNTD :
		vperror(0,"%s: unmounted",path);
		break;
	case FSUNACC :
	case FSERROR :
		vperror(0,"%s: %s",path,strerror(lerrno));
		break;
    }
}

/*
 *  Get the statistic space about a file system
 *
 *     SYSTEM		       STATUS
 *
 *	SGI			ok ( fixed a bug for the nfs file system )
 *	HP-UX			ok
 *	IBM (RS6000)		ok
 *	APOLLO (DN10k)		ok
 *	SUN 			ok
 *	SUN (ipx)		ERROR for file system bigger than 2 Giga
 *	ULTRIX 			ERROR for file system bigger than 2 Giga
 *
 *  INPUT
 *	path		file system path
 *
 *  OUTPUT
 *	total		total blocks
 *	avail		free blocks
 *	bsize		block size
 *
 *  RETURN
 *	1		successful completion
 *	-1		statfs() failure
 */
int getfsystat(path,total,avail,bsize,device)
char *path;
long *total,*avail,*bsize;
int *device;
{
#if !defined(RFIO)

#if defined( ultrix )
  struct fs_data fsbuffer;
#else

#if !defined(SOLARIS)
  static struct statfs fsbuffer;
#else
  static struct statvfs fsbuffer;
#endif

#endif
  static struct stat buffer;

 /*
  *  Call statfs()
  */
    if ( 
#if defined(__osf__) && defined(__alpha)
	 statfs(path,&fsbuffer,(int)sizeof(struct statfs))
#endif
#if defined( cray ) || defined( apollo ) || defined( sgi ) 
	 statfs(path,&fsbuffer,(int)sizeof(struct statfs),(int)0)
#endif
#if ( defined( sun ) && !defined(SOLARIS) )|| defined( ultrix ) || defined( _AIX ) || defined( hpux )
	 statfs(path,&fsbuffer)
#endif
#if defined(SOLARIS)
	 statvfs(path, &fsbuffer) 
#endif
#if defined( ultrix )
			      < 1 )
#else
			      < 0 )
#endif
				    return(-1);

 /*
  *  Get the output parameters
  */
#if defined( ultrix )
    *bsize = (long)1024;
    *total = (long)fsbuffer.fd_req.btot;
    *avail = (long)fsbuffer.fd_req.bfreen;
#else
#if defined( sgi )
    *bsize = (long)512; 					/*  SGI bug 	*/
#else
#if ( defined(__osf__) && defined ( __alpha) )
    *bsize = (long)fsbuffer.f_fsize ;
#else
#if defined (SOLARIS)
    *bsize = (long)fsbuffer.f_frsize ;
#else
    *bsize = (long)fsbuffer.f_bsize;
#endif
#endif
#endif
    *total = (long)fsbuffer.f_blocks;
#if defined( cray ) || defined( apollo ) || defined( sgi ) || defined(SOLARIS)
    *avail = (long)fsbuffer.f_bfree;
#endif
#if defined( sun ) || defined( hpux ) || defined( _AIX ) || ( defined(__osf__) && defined ( __alpha) )
    *avail = (long)fsbuffer.f_bavail;
#endif
#endif

    if ( device ) {
	if ( stat(path,&buffer) < 0 )
	    return(-1);
	*device = buffer.st_dev;
    }
#else /* RFIO */
	/*
	 * Using rfio call with rfio structure
	 */
	struct rfstatfs statfsb ;
	int rcode ;

	rcode = rfio_statfs(path , &statfsb ) ;
	if ( rcode < 0 )
		return -1 ;
	*avail = statfsb.freeblks ;
	*total = statfsb.totblks ;
	*bsize = statfsb.bsize ;
	
#endif /* RFIO */

    return(1);
}
