/*
 * Copyright (C) 1990,1991,1992 by CERN/CN/SW/DC
 * All rights reserved
 */

#ifndef lint
static char sccsid[] = "@(#)dosfgc.c	3.4 10/07/92 CERN-SW/DC Fabrizio Cane";
#endif /* not lint */

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <ctype.h>
#include <string.h>
#ifndef apollo
#include <malloc.h>
#endif
#include <time.h>
#include <math.h>
#include <varargs.h>
#include <shift_types.h>
#include <strutil.h>
#include <scandir.h>
#include <parser.h>
#include <table.h>
#include <dpmutil.h>
#include <sfgc.h>
#include <dosfgc.h>

int	gcerrno	= 0 ;

gc_list_t	GcList;		    /*  Garbage Collector List			*/

/*
 *
 */
typedef struct statistics_t {
    float	total;
    int		nofiles;
    int		biggest;
    path_t  	biggestname;
    time_t	oldest;
    path_t	oldestname;
} statistics_t ;

statistics_t 	statistics;

float		total_device;

#define LISTSIZE		GcList.space

#define OLDFILE(creatime)	\
	( GcList.youngest != NULL  &&  creatime < GcList.youngest->creatime )

#define SAFERATE		10.0
#define SAFESPACE(space)	(space + (((float)space)*SAFERATE/100.0))

boolean	gc_test		= FALSE;    /* '-t' option - do not delete files 	   */
boolean gc_inquiry	= FALSE;    /* '-i' option - inquiry to delete files	   */ 
boolean gc_recursion	= FALSE;    /* '-r' option - scan subdirectories	   */
byte gc_timetouse	= MTIME;    /* '-A' '-M' option - access modification time */ 
double	gc_factor	= 1.0;	    /* factor used in the weigth() function	   */
int	gc_removedir	= 0;	    /* remove empty directories control flag	   */
boolean gc_quiet	= FALSE;    /* '-q' option - do not some outputs	   */ 
boolean gc_statistics	= FALSE;    /* '-S' option - statistic 			   */

/*
 *  Initialize the GC List structure
 */
int init_list()
{
	memset((char*)&GcList,0,sizeof(gc_list_t));
}

/*
 *  Initialize the GC statistics
 */
int init_statistics()
{
	statistics.total = 0;
	statistics.nofiles = 0;
	statistics.biggest = -1;
	statistics.biggestname[0] = EOS;
	statistics.oldest = 0;
	statistics.oldestname[0] = EOS;
}

/*
 *  Print the GC statistics
 */
int print_statistics()
{
    gc_printf("Total      :  %s ",ftoanu(statistics.total));
    if ( total_device > 0.0 )
	gc_printf("(%s%%)\n",ftoanu(statistics.total*100.0/total_device));
    else
	gc_printf("(error)\n");
    gc_printf("Files	   :  %d\n",statistics.nofiles);
    if ( statistics.nofiles > 0 ) {
	gc_printf("Max        :  %s  ->  %s\n",
				ftoanu((float)statistics.biggest),
				statistics.biggestname);
	gc_printf("Average    :  %s\n",ftoanu(statistics.total/(float)statistics.nofiles));
	gc_printf("Oldest     :  %s  ->  %s\n",
				ctimeln(&statistics.oldest),statistics.oldestname);
    }
}

/*
 *  Reduce the GC List if there are too many files (i.e. too much space)
 *  The 'toobig list' is not touched by this routine ( files bigger than the 
 *  maximum specified size are always deleted although their size contribute
 *  to the total space in the GC List )
 */
int resize_list(spacetofree)
float  spacetofree;
{
  struct gc_element_t *element;
  int counter 			= 0 ;

    while ( GcList.youngest != NULL  &&
		(GcList.space - GcList.youngest->size) >= SAFESPACE(spacetofree) ) {

	counter++;

	LISTSIZE -= GcList.youngest -> size;
	element = GcList.youngest;

	if ( GcList.youngest -> older != NULL ) {
		GcList.youngest -> older -> younger = NULL;
		GcList.youngest = GcList.youngest -> older;
	}
	else {
		GcList.youngest = NULL;
		GcList.oldest = NULL;
	}

	free(element);
    }

    return(counter);
}

/*
 *
 */
double weight(ptime,size)
time_t	ptime;
int	size;
{
  /*
   *   func (1)
   *
   *   f(t,s,k) = t * ( 1 - k/sqrt(s) ) 
   *
   *   where k is a real in [0,1] and s is an integer in [1,n]
   *
   *   func(2)
   *
   *   f(t,s,k) = t - log((size/1024)**k)*86400
   *
   *   where k is an real in [0,n]  ( if k=0 then f(t,s,k) = t )
   *
   */

  if ( size < 1024 ) size = 1024;

 /*  
  *  return( (float)ptime * ( 1 + gc_factor/(float)sqrt((float)size/1024.0) ) );  
  */

  return( (double)ptime - log( pow((double)size/1024.0,(double)gc_factor) ) * 86400.0 );
}

/*
 *
 */
boolean isrightplace(element,creatime,size)
struct gc_element_t	*element;
time_t			creatime;
int			size;
{
#ifdef NOSORTFUNC 
	return( !( element->creatime < creatime  ||
			( element->creatime==creatime  &&  element->size<=size ) ) );
#else
	return( weight(element->creatime,element->size) > weight(creatime,size) );
#endif
}

/*
 *  Insert a new file in the GC List. If toobig is TRUE then the file is inserted 
 *  in the 'toobig list' otherwise in the main list. The files in the first list
 *  are always physically deleted while the files in the latter are dynamically
 *  inserted ( via this routine ) or extracted ( by resize_list() ) when more space
 *  than what was required has been selected.
 */
int insert_file(path,size,creatime,linktype,toobig)
char	 *path;
int	 size;
time_t	 creatime;
char	 linktype;
boolean	 toobig;
{
  struct gc_element_t	*position;
  struct gc_element_t	*element;
  struct stat buffer;

   element = toobig ? GcList.tbgoldest : GcList.oldest ;

  if ( element != NULL ) {

 /*
  *  Search the correct position to insert the new element
  *  If the time creation matches then smaller files are inserted first
  */
    while ( element != NULL  &&  !isrightplace(element,creatime,size ) )

	element = element->younger;


    if ( element == NULL ) {
     /*
      *  Insert a new element at the queue of the list
      */
	element = (struct gc_element_t*)malloc(sizeof(struct gc_element_t));

	element->younger = NULL;
	element->older = toobig ? GcList.tbgyoungest : GcList.youngest;
	element->older->younger = element;
	if ( toobig )
		GcList.tbgyoungest = element ;
	else
		GcList.youngest = element ;
    }

    else {
    /*
     *  Insert a new element in the list 
     *  ( in front of the list or in the middle )
     */

	position = element;

	element = (struct gc_element_t*)malloc(sizeof(struct gc_element_t)); 

	if ( position->older == NULL ) {
		element->younger = position;
		position->older = element ; 
		element->older = NULL;
		if ( toobig )
			GcList.tbgoldest = element;
		else
			GcList.oldest = element;
	}
	else {
		element->younger = position;
		element->older = position -> older;
		position->older = element;
		element->older->younger = element;
	}
    }

  }

  else {
 /*
  *  Insert the first element in the empty list
  */
	if ( toobig )
	  	GcList.tbgoldest = GcList.tbgyoungest = element = 
			(struct gc_element_t*)malloc(sizeof(struct gc_element_t));
	else
	  	GcList.oldest = GcList.youngest = element = 
			(struct gc_element_t*)malloc(sizeof(struct gc_element_t));

  	element -> older = NULL;
  	element -> younger = NULL;
  }

  strcpy(element -> path , path);
  element -> size = size;
  element -> creatime = creatime;
  element -> linktype = linktype;
  if ( stat(path,&buffer) < 0 ) {
	element -> uid = -1;
	element -> gid = -1;
  }
  else {
  	element -> uid = buffer.st_uid;
  	element -> gid = buffer.st_gid;
  }

  LISTSIZE += size;
  return(1);
}

/*
 *  Scan all the directories given in input ( dirpath, nodir )
 *  and build the GC List using the insert_file() function.
 */
int build_list(timeold,spacetofree,maxsize,load,mountpath,dirpath,nodir,select)
time_t	  timeold;
float 	  spacetofree;
int	  maxsize;
boolean	  load;
path_t	  *mountpath;
path_t	  *dirpath;
int	  nodir;
boolean   (*select)();
{
  path_t	path,nfspath;
  int		size;
  time_t	creatime;
  char		linktype;
  boolean	toobig,tooold;
  int		reply;

    while ( nodir-- ) {

	if ( (reply=gc_open_dir(mountpath[nodir],dirpath[nodir],select)) > 0 ) {

	  while ( gc_next_file(path,&size,&creatime,&linktype) > 0 ) {

		strcpy(nfspath,dirpath[nodir]);
		strcat(nfspath,"/");
		strcat(nfspath,path);

	      /*
	       *  If < the GC List is not big enough >  or
	       *     < the file is older then the specified time > or
	       *     < load is specified > or
	       *     < a max size is specified and the file is bigger >
	       *  then insert the new file in the GC List
	       */
		toobig = size > maxsize  &&  maxsize > 0 ;
		tooold = creatime < timeold  ||  timeold == 1 ;
  
		if ( LISTSIZE < spacetofree || tooold || load || toobig )

			insert_file(nfspath,size,creatime,linktype,toobig);

		else

	      /*
	       *  If the GC List is big enough then insert the new file unless its
	       *  time creation is bigger the all files in the list
	       */ 
		if ( OLDFILE(creatime) ) {
			insert_file(nfspath,size,creatime,linktype,toobig);
			resize_list(spacetofree);
		}

	      /*
	       *  Update statistic data ( total space , number of files , biggest file )
	       */
		statistics.total += (float)size;
		statistics.nofiles ++ ;
		if ( statistics.biggest < 0 || size > statistics.biggest ) {
			statistics.biggest = size;
			strcpy(statistics.biggestname,nfspath);
		}
		if ( statistics.oldest == 0 || creatime < statistics.oldest ) {
			statistics.oldest = creatime;
			strcpy(statistics.oldestname,nfspath);
		}
 
	  } /* while  -  gc_next_file() succeed */

	  gcerrno = scanerrno;

	  if ( scanerrno != EOF ) return( -1 );

	  gc_close_dir();

	} /* if  -  gc_open_dir() succeed */

	else
	if ( reply < 0 ) {
		gcerrno = scanerrno;
		fprintf(stderr,"Cannot open directory %s\n",dirpath[nodir]);
		return( -1 );
	}

    } /* while  -  list of directories is not empty */

    return(1);
}

/*
 *
 */
int check_space(timeold,spacetofree,rate,left,maxsize)
time_t	timeold;
float 	*spacetofree;
float   rate;
boolean left;
int	maxsize;
{

 /*
  *  If the time option has been specified then no other options can be used
  */
    if ( timeold > 0 ) {
	if ( left ) {
		vperror(0,"Left option (-left) cannot be used with time option (-T)");
		return( -1 );
	}
	if ( *spacetofree >= 0 ) {
		vperror(0,"Space option (-S) cannot be used with time option (-T)");
		return( -1 );
	}
	if ( rate >= 0 ) {
		vperror(0,"Rate option (-R) cannot be used with time option (-T)");
		return( -1 );
	}
    }
    else {
      /*
       *  The option -S and -R cannot be used together
       */
	if ( *spacetofree >= 0  &&  rate >= 0 ) {
		vperror(0,"Space option (-S) cannot be used with rate option (-R)");
		return( -1 );
	}

      /*
       *  If the -Left option is used with the -S option then the effective space
       *  to deallocate is what exceed the space specified in the -S option
       */
	if ( *spacetofree >= 0  &&  left )
		*spacetofree = LISTSIZE - *spacetofree ;

      /*
      **  If rate is specified then compute the effective space to deallocate 
      **  using the size of the GC List 
      */
	if ( rate >= 0 )
		if ( left )
			*spacetofree = LISTSIZE/100.0*(100.0-rate);
		else
			*spacetofree = LISTSIZE/100.0*rate;
    }
    return(1);
}

/*
 *  Delete the files contained in the GC List 
 *  The options '-test' and '-inquiry' are used here
 */
int garbage(timeold,spacetofree,filelist)
time_t	     timeold;
float 	     *spacetofree;
gc_element_t **filelist;
{
  struct gc_element_t	*tofree,*element = GcList.oldest ;
  boolean		delete;
  char			ch;
  int			reply;
  int			ReturnCode;
  char			*astime;
  time_t		ntime;
  float			space 	 	 = *spacetofree ;
  user_t		user;
  group_t		group;

	ReturnCode = (element != NULL && (*spacetofree > 0 || timeold > 0) ) 
				? EXGCOK : EXGCNONE ;

	while ( element != NULL  &&  ( *spacetofree > 0  ||  timeold > 0 ) ) {

	    astime = (char*)ctimeln(&element->creatime);

	    get_userbyid(user,element->uid,NULL);
	    get_groupbyid(group,element->gid);


	    if ( gc_test || gc_inquiry ) {
			printf("%s %10s %6s %10d  %4s  %s ",
					astime,
					user,group,
					element->size,
					linkname[element->linktype],
					element->path);
			if ( !gc_inquiry )   newline;
	    }


	    if ( gc_inquiry ) {
			printf(" Delete (Y/N) ? ");
			do
				ch = uppercase(getchar());
			while ( ch != 'Y'  &&  ch != 'N' );
			delete = ( ch == 'Y' );
	    }
	    else
			delete = TRUE;


	    if ( delete ) {
		if ( gc_test || (reply=unlink(element->path)) >= 0 )
				*spacetofree -= element -> size;

		if ( !gc_test )  
		    if ( reply < 0 ) {
			fprintf(stderr,"CAN'T DELETE    ");
			fprintf(stderr,"%s %10s %6s %10d  %4s  %s\n",
						astime,
						user,group,
						element->size,
						linkname[element->linktype],
						element->path);
		    }
		    else {
		        if ( !gc_inquiry ) 
				printf("%s %10s %6s %10d  %4s  %s\n",
						astime,
						user,group,
						element->size,
						linkname[element->linktype],
						element->path);
		    }
 /*
		mr_IO(mr_USER1,"%s %10s %6s %10d  %4s  %s %s\n",
					astime,
					user,group,
					element->size,element->path,
					linkname[element->linktype],
					reply < 0 ? "FAILED" : "" );
*/
	    }

 	    if ( /*gc_test  ||*/  !delete  ||  reply < 0  ||  filelist == NULL ) {
		    tofree = element;
		    element = element -> younger;	/* next element		*/
		    free(tofree);			/* deallocate memory 	*/
	    }
	    else {
		    tofree = element;
		    element = tofree -> younger;
		    tofree -> older = *filelist;
		    if ( *filelist != NULL )
				tofree -> older -> younger = tofree;
		    *filelist = tofree;
	    }

	}

	GcList.oldest = element;
	if ( element == NULL ) {
		GcList.youngest = NULL;
		LISTSIZE = 0;
	}

	time(&ntime);
	gc_printf("\tTOTAL FILE SELECTED\n%s %28s\n\n",
				ctimeln(&ntime),ftoanu(space-(*spacetofree)));

      /*
       *  remove the hard/soft link list
       */
	remove_link_info();

	return( ReturnCode );
}

/*
 *  Delete files inserted in the 'toobig list'
 *  The options '-test' is used here
 */
int garbage_toobig(spacetofree,maxsize,filelist)
float 	     *spacetofree;
int	     maxsize;
gc_element_t **filelist;
{
  struct gc_element_t	*tofree,*element = GcList.tbgoldest ;
  float			space 		 = *spacetofree ;
  char			*astime;
  time_t		ntime;
  user_t		user;
  group_t		group;
  int			reply;

	if ( maxsize <= 0 ) return( 1 );

	while ( element != NULL ) {

	    astime = (char*)ctimeln(&element->creatime);

	    get_userbyid(user,element->uid,NULL);
	    get_groupbyid(group,element->gid);

	    if ( gc_test )
			printf("%s %10s %6s %10d  %4s  %s\n",
					astime,
					user,group,
					element->size,
					linkname[element->linktype],
					element->path);

	    if ( gc_test || (reply = unlink(element->path)) >= 0 )
			*spacetofree -= element -> size;

	    if ( !gc_test )
		if ( reply < 0 ) {
			fprintf(stderr,"CAN'T DELETE    ");
			fprintf(stderr,"%s %10s %6s %10d  %4s  %s\n",
						astime,
						user,group,
						element->size,
						linkname[element->linktype],
						element->path);
		}
		else {
			printf("%s %10s %6s %10d  %4s  %s\n",
						astime,
						user,group,
						element->size,
						linkname[element->linktype],
						element->path);
		}
/*
	    mr_IO(mr_USER1,"%s %10s %6s %10d  %4s  %s %s\n",
					astime,
					user,group,
					element->size,element->path,
					linkname[element->linktype],
					reply < 0 ? "FAILED" : "" );
*/
	    if ( /*gc_test  ||*/  reply < 0  ||  filelist == NULL ) {
		    tofree = element;
		    element = element -> younger;	/* next element		*/
		    free(tofree);			/* deallocate memory 	*/
	    }
	    else {
		    tofree = element;
		    element = tofree -> younger;
		    tofree -> older = *filelist;
		    if ( *filelist != NULL )
				tofree -> older -> younger = tofree;
		    *filelist = tofree;
	    }

	}

	time(&ntime);
	gc_printf("\tTOTAL FILE BIGGER THAN %s\n",itoanmu(maxsize));
	gc_printf("%s %28s\n\n",ctimeln(&ntime),ftoanu(space-(*spacetofree)));

	return( 1 );
}

/*
 *  Garbage the directories given in input ( dirpath, nodir ) in order to
 *  free 'spacetofree' bytes
 */
int dosfgc(timeold,spacetofree,rate,maxsize,left,mountpath,dirpath,nodir,select,filelist)
time_t	     timeold;
float	     *spacetofree;
float	     rate;
int	     maxsize;
boolean      left;
path_t	     *mountpath;
path_t	     *dirpath;
int	     nodir;
boolean      (*select)();
gc_element_t **filelist;
{
  float    space;
  boolean  load;
  int	   reply;

    if ( timeold==0  &&  *spacetofree<0  &&  rate<0  &&  maxsize<0 ) {
	vperror(0,"no option used to specify how to garbage ( -s / -r / -t / -B )");
	return(-1);
    }

    init_list();

    if ( gc_statistics )
		init_statistics();

    load = rate > 0 || left ;

    if ( filelist != NULL )
		*filelist = NULL ;

     reply=build_list(timeold,*spacetofree,maxsize,load,mountpath,dirpath,nodir,select);
	if ( reply < 0 )
		return( reply );

	if ( (reply=check_space(timeold,spacetofree,rate,left,maxsize)) < 0 )
		return( reply );

	space = *spacetofree;

	if ( gc_statistics )
		print_statistics();

	garbage_toobig(spacetofree,maxsize,filelist);

	reply = garbage(timeold,spacetofree,filelist);

	if ( space > 0  &&  *spacetofree == space ) {
/*
		mr_IO(mr_ERROR,"garbage() couldn't delete any file");
*/
		return(-1);
	}

    return(reply);
}


/**************************************************************************************/
/*	I N T E R F A C E   T O   T H E   S C A N D I R   U T I L I T I E S	      */
/**************************************************************************************/

int	*gc_scandir;		/* scandir structure */

/*
 *  Create the 'scandir' structure used later by gc_next_file() and gc_close_dir()
 */
int gc_open_dir(mountpath,dirpath,select)
char    *mountpath;
char  	*dirpath;
boolean (*select)();
{
  int reply;
    gc_scandir = (int*)scandir_open(mountpath,dirpath,-1,select,&reply);
    return( gc_scandir == NULL ? reply : 1 );
}

/*
 *  Close and deallocate the 'scandir' structure
 */
int gc_close_dir()
{
	return( scandir_close(gc_scandir) );
}

/*
 *  Get the next file to be processed by the garbage collector using the scandir
 *  structure previously created.
 *
 *  This routine recognizes the hard and soft link and processes them ( a list 
 *  of links is kept )
 */
int gc_next_file(path,size,creatime,linktype)
char	*path;
int	*size;
int	*creatime;
char	*linktype;
{
  int			puid;
  static struct stat	buffer,lbuffer;
  int			nlink;


	if ( scandir_next(gc_scandir,path,&buffer,&lbuffer,&puid) < 0 )
		return( -1 );

	switch ( ( *linktype = get_link_info(&buffer,&lbuffer,&nlink) ) ) {

	  case  NOLINK :
	    *creatime = gc_timetouse == MTIME ? buffer.st_mtime : buffer.st_atime ;
	    *size = buffer.st_size;
	    break;

	  case  SOFTLINK :
	    *creatime = gc_timetouse == MTIME ? lbuffer.st_mtime : lbuffer.st_atime ;
	    *size = lbuffer.st_size;
	    break;

	  case  HARDLINK :
	    *creatime = gc_timetouse == MTIME ? buffer.st_mtime : buffer.st_atime ;
	    *size = ( nlink == 1 ) ? buffer.st_size : 0 ;
	    break;

	  case -1 :
	    return( -1 );

	  default :
	    vperror(0,"get_link_info() returned unexpected value %d",*linktype);
	    return( -1 );

	} /* switch */

	return( 1 );

}

/*
 *  
 */
boolean gc_select_dir(dirpath,buffer,lbuffer,rmdirpath)
path_t		dirpath;
struct stat	*buffer;
struct stat	*lbuffer;
path_t		rmdirpath;
{
  int  nlink;
  int  tlink;

  switch ( ( tlink = get_link_info(buffer,lbuffer,&nlink) ) ) {

    case NOLINK :
    case HARDLINK :
	if ( rmdirpath != NULL  &&  !gc_test )
		if ( rm_emptydir(dirpath,-1,strlen(dirpath)-strlen(rmdirpath)) < 0 )
			fprintf(stderr,"can't remove directories %s,%s\n",
								dirpath,rmdirpath);
	return( 1 );

    case SOFTLINK :
	return( 0 );

    default :
	vperror(0,"get_link_info() returned unexpected value %d",tlink);
	return( -1 );
  }

}

/*
 *  
 */
boolean gc_select_files(fspath,cfpath,buffer,lbuffer,dir,level,rpath,puid)
char		*fspath;
char		*cfpath;
struct stat	*buffer;
struct stat	*lbuffer;
boolean		dir;
int		level;
char		*rpath;
int		*puid;
{
  boolean  reply;
  char	   *rmdirpath;

  if ( gc_recursion )

    reply = select_002(fspath,cfpath,buffer,lbuffer,dir,level,rpath,puid) ;

  else

    reply = select_001(fspath,cfpath,buffer,lbuffer,dir,level,rpath,puid) ;

  if ( dir  &&  reply ) {
	rmdirpath = ( gc_removedir == -1 ) ? NULL : rpath ;
	level = 0;
	while ( rmdirpath  &&  level++ < gc_removedir   &&  
			(rmdirpath=strchr(rmdirpath,'/')) != NULL ) 
		rmdirpath++ ;
	return( gc_select_dir(cfpath,buffer,lbuffer,rmdirpath) );
  }

  return(reply);
}

/*
 *
 */
boolean gc_select_fsys(fspath,cfpath,buffer,lbuffer,dir,level,rpath,puid)
char		*fspath;
char		*cfpath;
struct stat	*buffer;
struct stat	*lbuffer;
boolean		dir;
int		level;
char		*rpath;	
int		*puid;
{
  boolean  reply;
  char	   *rmdirpath;

  reply = select_110(fspath,cfpath,buffer,lbuffer,dir,level,rpath,puid) ;

  if ( dir  &&  reply ) {
	if ( (rmdirpath=strchr(rpath,'/')) && 
		(rmdirpath=strchr(rmdirpath+1,'/')) )
			rmdirpath++ ;
	return( gc_select_dir(cfpath,buffer,lbuffer,rmdirpath) );
  }

  return(reply);
}

/*
 *
 */
boolean gc_select_fsysgrp(fspath,cfpath,buffer,lbuffer,dir,level,rpath,puid)
char		*fspath;
char		*cfpath;
struct stat	*buffer;
struct stat	*lbuffer;
boolean		dir;
int		level;
char		*rpath;	
int		*puid;
{
  boolean  reply;
  char	   *rmdirpath;

  reply = select_210(fspath,cfpath,buffer,lbuffer,dir,level,rpath,puid) ;

  if ( dir  &&  reply ) {
	if ( (rmdirpath=strchr(rpath,'/')) )
		rmdirpath++;
	return( gc_select_dir(cfpath,buffer,lbuffer,rmdirpath) );
  }

  return(reply);
}
