/*
 * Copyright (C) 1992 by CERN/CN/SW/DC
 * All rights reserved
 */

#ifndef lint
static char sccsid[] = "@(#)dosfget.c	3.11 01/20/94 CERN-SW/DC Fabrizio Cane";
#endif /* not lint */

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <shift_types.h>
#include <strutil.h>
#include <parser.h>
#include <table.h>
#include <dpmutil.h>

/*
 *  Do sfget
 */
int dosfget(pool,pathuid,clientuid,look,table,remove,size,filename,atomic,msg)
char *pool;
int pathuid,clientuid;
boolean look,table,remove;
long size;
char *filename;
boolean atomic;
char *msg;
{
  extern fsys_t *fsys_table;            /*  pointer to the file system table	*/
  extern pool_t *pool_table;            /*  pointer to the pool table		*/
  user_t client,user;
  int clientgid,pathgid;
  int nofs;
  static poolfs_t *poolfs	= NULL ;
  static scanp_t *scan1		= NULL ;
  static int length1		= 0 ;
#if defined( CREATMP )
  static scanp_t *scan2		= NULL ;
  static int length2		= 0 ;
#endif
  char *file;
  int loops 			= 0;
  int index;
  int idx,fsys,scan;
  int reply;
  int maxsize,himaxsize;
  int pidx,sizeflag;

 /*
  *  Get the group ids
  */
    if ( getgroupids(clientuid,&clientgid,client,pathuid,&pathgid,user) < 0 )
	return(EXSYS);

 /*
  *  Build the file system list
  */
    if ( (reply = getnfspaths(pool,pathuid,-1,&nofs,&poolfs)) != 1 ) return(reply);

#if defined( CREATMP )
 /*
  *  The atomic mode is supported unless more than one pool is specified
  */
    if ( atomic )
	for (idx=1; idx<nofs; idx++)
	    if ( poolfs[idx].pidx != poolfs[0].pidx ) {
		atomic = 0;
		break;
	    }
#endif

 /*
  *  Loop
  */
    for (;;) {

    if ( loops++ > MAXLOOPS ) {
	vperror(0,"Internal: too many loops");
	return(EXINT);
    }

 /*
  *  Scan the shift disk pool
  */
    if ( scandiskpool(1,clientuid,pathuid,pathgid,filename,nofs,poolfs,&scan1,atomic,&length1) < 0 ) 
	return(EXSYS);

 /*
  *  Look for the shift file
  */
    switch ( (reply = lookforfile(nofs,scan1,NULL,&remove,look,client,msg)) ) {
	case 1      : setgid(clientgid);
		      setuid(clientuid);
		      return(EXFND);
	case 2      : remove = FALSE ;
		      break;
	case 3      : continue;
	case EXPERM : return(EXPERM);
	default     : vperror(0,"Internal: unexpected lookforfile(1) reply (%d)",reply);
		      return(EXINT);
    }

#if !defined( CREATMP )

    if ( atomic ) {
	vperror(0,"atomic mode selected but not available");
	return(EXCONF);
    }

#else

    if ( atomic ) {

 /*
  *  Look for the tmp files
  */
    for (idx=0; idx<nofs; idx++)
	if ( scan1[idx].tmp > 0 ) break;

 /*
  *  If at least one tmp file was found then read all tmps unless look is selected
  */
    if ( idx < nofs && !look )
readtmps :
	switch ( (reply = read_tmps(nofs,poolfs,scan1,NULL,&fsys,&scan)) ) {
	    case 3     : if ( (reply = getnfspaths(pool,pathuid,-1,&nofs,&poolfs)) != 1 )
			     return(reply);
	    case 2     : remove_tmps(nofs,scan1,NULL,TRUE);
			 continue;
	    case 1     : goto creation;
	    case EXSYS : return(EXSYS);
	    case EXINT : return(EXINT);
	    default    : vperror(0,"Internal: unexpected read_tmps(1) reply (%d)",reply);
			 return(EXINT);
	}

   } /* atomic */

#endif

 /*
  *  If the look option is selected then the file cannot be created
  */
    if ( look ) {
	vperror(0,"File <%s> not found in pool <%s>",filename,pool);
	return(EXNFND);
    }

    pidx = -1;
    sizeflag = 0;
    for (idx=0; idx<nofs; idx++) {

	if ( pidx == poolfs[idx].pidx )
	    continue;
	else
	    pidx = poolfs[idx].pidx;

     /*
      *  Check the authorization to create a shift file in the pool
      */
	if ( (reply = authorized_to_create(pidx,clientuid,clientgid)) != 1 )
	    return(reply);

     /*
      *  Check the pool file size limit
      */
	if ( !sizeflag ) {
	    maxsize = pool_table[pidx].size;
	    sizeflag = maxsize <= 0 || size <= maxsize ;
	    if ( maxsize > himaxsize ) himaxsize = maxsize;
	}

    } /* for */

    if ( !sizeflag ) {
	vperror(0,"Size specified (%sBytes) is over the size limit (%sBytes)",
		  ftoanu((float)size),ftoanu((float)himaxsize));
	return(EXCONF);
    }

 /*
  *  Select the file system where to place the shift file 
  */
    if ( (reply = select_file_system(size,nofs,poolfs,scan1,&fsys,&scan)) != 1 )
	return(reply);

#if defined( CREATMP )

  if ( atomic ) {

 /*
  *  Create the temporary file in the same file system as for the shift file
  */
    file = scan1[scan].p_tmp;
    index = strlen(poolfs[scan].mount);
#if defined( TRACE )
    if ( TRACE_dosfget ) printf("creating tmp %s [%d]\n",file,scan);
#endif
    switch ( (reply = create_file(file,index,clientuid,clientgid,pathuid,pathgid)) ) {
	case 1  : scan1[scan].tmp = 1;
		  break;
	case 0  : scan1[scan].tmp = 1;
		  goto readtmps;
	case -1 : return(EXSYS);
	default : vperror(0,"Internal: unexpected create_file(tmp) reply (%d)",reply);
		  return(EXINT);
    }

 /*
  *  Scan the shift disk pool
  */
    if ( scandiskpool(2,clientuid,pathuid,pathgid,filename,nofs,poolfs,&scan2,atomic,&length2) < 0 )
	return(EXSYS);

 /*
  *  Look for the shift file
  */
    switch ( (reply = lookforfile(nofs,scan2,scan1,&remove,FALSE,client,msg)) ) {
	case 1      : setgid(clientgid);
		      setuid(clientuid);
		      return(EXFND);
	case 2      : break;
	case 3      :
	case EXPERM :
	default     : vperror(0,"Internal: unexpected lookforfile(2) reply (%d)",reply);
		      return(EXINT);
    }

 /*
  *  Write the disk tmps configuration in the created tmp file
  */
    switch ( (reply = write_tmp(scan1[scan].p_tmp,nofs,scan1,scan2)) ) {
	case 2     : remove_file(scan1[scan].p_tmp);
		     continue;
	case 1     : break;
	case EXSYS : return(EXSYS);
	default    : vperror(0,"Internal: unexpected write_tmp() reply (%d)",reply);
		     return(EXINT);
    }

 /*
  *  Read list in the temporary files found
  */
    switch ( (reply = read_tmps(nofs,poolfs,scan1,scan2,&fsys,&scan)) ) {
	case 3     : if ( (reply = getnfspaths(pool,pathuid,-1,&nofs,&poolfs)) != 1 )
			 return(reply);
	case 2     : remove_tmps(nofs,scan1,scan2,TRUE);
		     continue;
	case 1     : break;
	case EXSYS : return(EXSYS);
	case EXINT : return(EXINT);
	default    : vperror(0,"Internal: unexpected read_tmps(2) reply (%d)",reply);
		     return(EXINT);
    }

    } /* atomic */

#endif

creation:

 /*
  *  Create the shift file in the selected file system
  */
    if ( !table ) {
	file = scan1[scan].p_file;
	index = strlen(poolfs[scan].mount);
#if defined( TRACE )
	if ( TRACE_dosfget ) printf("creating file %s [%d]\n",file,scan);
#endif
	reply = create_file(file,index,clientuid,clientgid,pathuid,pathgid);
    }
    switch ( !table ? reply : 1 ) {
	case 1  :
	case 0  :
#if defined( CREATMP )
		  remove_tmps(nofs,scan1,scan2,TRUE);
#endif
		  strcpy(msg,scan1[scan].p_file);
		  setgid(clientgid);
		  setuid(clientuid);
		  return( reply == 1 ? EXCREA : EXFND );
	case -1 : return(EXSYS);
	default : vperror(0,"Internal: unexpected create_file(file) reply (%d)",reply);
		  return(EXINT);
    }

    } /* sfget loop */
}

/*
 *  INPUT
 *	nfspath		shift file path to be echoed by sfget
 *	linkpath	link path to be created
 *
 *  RETURN
 *	0	the link points to an existing and different shift file
 *	1	the link already points to the shift file
 *	2	the link points to a different shift file that doesn't exist
 *     < 0	an error occured
 */
boolean issamelink(nfspath,linkpath)
char *nfspath,*linkpath;
{
  path_t targetpath;
  struct stat buf;
  int rc ;

    if ( (rc=readlink(linkpath,targetpath,(int)sizeof(path_t))) < 0 ) {
	vperror(1,"readlink(%s,,)",linkpath);
	return(EXSYS);
    }

    targetpath[rc]='\0' ;

    if ( !strcmp(nfspath,targetpath) ) return(1);

    if ( stat(targetpath,&buf) < 0 )
	return(errno == ENOENT || errno == ENOTDIR ? 2 : EXSYS);
    else
	return(0);
}

/*
 *  Create/Remove the symbolic link to the shift file
 */
int symlinkop(op,file,nfspath,linkdir)
int op;
char *file,*nfspath,*linkdir;
{
  path_t linkpath;
  int reply;

    if ( strlen(linkdir)+(linkdir[strlen(linkdir)-1]!='/'?1:0)+strlen(file)+2 > sizeof(path_t) ) {
	vperror(0,"file name (%s%s%s) too long",
		   linkdir,linkdir[strlen(linkdir)-1]!='/'?"/":"",file);
	return(EXINT);
    }

    strcpy(linkpath,linkdir);
    if ( linkdir[strlen(linkdir)-1] != '/' ) strcat(linkpath,"/");
    strcat(linkpath,file);

    switch ( op ) {
	case 0 :
	    if ( symlink(nfspath,linkpath) < 0 )
		if ( errno != EEXIST ) {
		    vperror(1,"symlink(...,%s)",linkpath);
		    return(EXSYS);
		}
		else
		switch ( (reply = issamelink(nfspath,linkpath)) ) {
		  case 0  : errno = EEXIST;
			    vperror(1,"symlink(...,%s)",linkpath);
			    return(EXSYS);
		  case 1  : break;
		  case 2  : if ( unlink(linkpath) < 0 && errno != ENOENT ) {
				vperror(1,"unlink(%s)",linkpath);
				return(EXSYS);
			    }
			    if ( symlink(nfspath,linkpath) < 0 ) {
				vperror(1,"symlink(...,%s)",linkpath);
				return(EXSYS);
			    }
			    break;
		  default : return(reply);
		} /* switch */
	    break;
	case 1 :
	    if ( unlink(linkpath) < 0 && errno != ENOENT ) {
		vperror(1,"unlink(%s)",linkpath);
		return(EXSYS);
	    }
	    break;
	default :
	    vperror(0,"unknown operation (%d) in symlinkop()",op);
	    return(EXINT);
    }

    return(1);
}

/*
 *  Look for the shift file in the scan disk pool structure
 *
 *  INPUT
 *	nofs		length of scan
 *	scan		scan disk pool structure
 *	prevscan	first scan structure if 'scan' is the second
 *	remove		sfget remove option
 *	look		sfget look option
 *
 *  OUTPUT
 *	remove		the remove option is disabled
 *
 *  RETURN
 *	3		shift file found and removed ( remove option )
 *	2		no shift file found
 *	1		shift file found
 *     < 0		an error occured ( EXPERM )
 */
int lookforfile(nofs,scan,prevscan,remove,look,client,msg)
int nofs;
scanp_t *scan,*prevscan;
boolean *remove,look;
char *client;
char *msg;
{
  int idx,lkidx = UNDEF_IDX ;

 /*
  *  Look for the most recent shift file copy in the scan disk pool structure
  *  unless remove is specified otherwise look for the first copy
  */
    for (idx=0; idx<nofs; idx++)
	if ( scan[idx].file > 0 && (lkidx == UNDEF_IDX ||
		scan[lkidx].stat.st_mtime < scan[idx].stat.st_mtime) ) {
	    lkidx = idx;
	    if ( *remove ) break;
	}

 /*
  *  At least one shift file copy is found
  */
    if ( lkidx != UNDEF_IDX ) {
     /*
      *  If the remove option is specified and this is the first scan then
      *  remove the shift files unless the client is not authorized
      */
	if ( *remove && prevscan == NULL )
	 /*
	  *  'lkidx' is not the most recent copy of the file because of the short cut
	  *  in the previous loop ( it should be as in sfrm! )
	  */
	    if ( scan[lkidx].perm ) {
		remove_files(nofs,scan);
#if defined( CREATMP )
		remove_tmps(nofs,scan,prevscan,TRUE);
#endif
#if defined( TRACE )
		if ( TRACE_dosfget ) printf("shift file ( and tmp ) removed\n");
#endif
		*remove = FALSE;
		return(3);
	    }
	    else {
		vperror(0,"You (%s) are not authorized to remove <%s>",
			  client,
			  scan[lkidx].p_file);
		return(EXPERM);
	    }
     /*
      *  If remove is not specified then remove the tmp files and return the shift file
      */
#if defined( CREATMP )
	remove_tmps(nofs,scan,prevscan,!look);
#endif
	strcpy(msg,scan[lkidx].p_file);
	return(1);
    }

 /*
  *  No shift file found
  */
#if defined( TRACE )
    if ( TRACE_dosfget ) printf("no file\n");
#endif
    return(2);
}

/*
 *  Remove from disk all shift file copies found in the scan disk pool structure
 *
 *  INPUT
 *	nofs		length of scan
 *	scan		scan disk pool structure
 */
int remove_files(nofs,scan)
int nofs;
scanp_t *scan;
{
  int idx;
    if ( scan != NULL )
	for (idx=0; idx<nofs; idx++)
	    if ( scan[idx].file > 0 ) remove_file(scan[idx].p_file);
}
