/*
 * Copyright (C) 1992 by CERN/CN/SW/DC
 * All rights reserved
 */

#ifndef lint
static char sccsid[] = " @(#)dpmutil.c	3.15 07/12/93  CERN-SW/DC Fabrizio Cane";
#endif /* not lint */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <shift_types.h>
#include <strutil.h>
#include <strdup.h>
#include <strtok.h>
#include <user.h>
#include <parser.h>
#include <table.h>
#include <dpmutil.h>

#if defined( TRACE )
int tracemode = 0;
#endif

#define ENVPREFIX		"DPMPOOL"
#define ENVPREFIX_LENGTH	7

/*
 *  Build the nfs-pathname of a shift file system
 *
 *  INPUT
 *	fsysidx		file system index
 *
 *  OUTPUT
 *	mountpath	the nfs-path pointing to the file system
 */
void create_mountpath(fsysidx,mountpath)
index_t_s fsysidx;
char *mountpath;
{
  extern disk_t *disk_table;		/*  pointer to the disk server table	*/
  extern fsys_t *fsys_table;		/*  pointer to the file system table	*/

    strcpy(mountpath,disk_table[fsys_table[fsysidx].diskidx].mount);
    strcat(mountpath,fsys_table[fsysidx].name);
}

/*
 *  Build the complete nfs-pathname of a shift file
 *
 *  INPUT
 *	fsysidx		file system index
 *	uid		uid of the user whose home directory is used
 *	userpath	file path supplied by the user
 *
 *  OUTPUT
 *	nfspath		the complete nfs-path
 */
int create_nfspath(fsysidx,uid,gid,userpath,nfspath)
index_t_s fsysidx;
int uid,gid;
char *userpath;
char *nfspath;
{
  static user_t user;
  static group_t group;

 /*
  *  Build the nfs-pathname pointing to the file system
  */
    create_mountpath(fsysidx,nfspath);

 /*
  *  Concatenate the group and user directory names
  */
    if ( !( uid < 0 ) )
	if ( get_userbyid(user,uid,&gid) < 0 ) {
	    vperror(0,"unknown user id <%d>",uid);
	    return(-1);
	}

    if ( !( uid < 0 ) || !( gid < 0 ) )
	if ( get_groupbyid(group,gid) < 0 ) {
	    vperror(1,"getgrgid()");
	    return(-1);
	}

    strcat(nfspath,"/");
    strcat(nfspath,group);
    strcat(nfspath,"/");

    if ( !( uid < 0 ) ) {
	strcat(nfspath,user);
	strcat(nfspath,"/");
    }

 /*
  *  Concatenate the file path unless it's undefined
  */
    if ( userpath != NULL ) strcat(nfspath,userpath);

    return(1);
}

#ifdef DEBUG

/*
 *  Print the pool graph ( string containing the leaves name )
 *
 *  INPUT
 *	ptr	root node of the pool sub graph to print
 *	first	TRUE if it is the first call and FALSE otherwise
 */
void printgraph(ptr,first)
pnode_t *ptr;
boolean first;
{
    do {

	if ( !ptr->skip )
	    if ( ptr->pexp )
		(void)printgraph(ptr->pexp,FALSE);
	    else
		printf("%s[%c]",
			ptr->pname,
			ptr->lev==PSPCD ? PSPCH : (ptr->lev==PSCCD ? PSCCH : '\n'));

	ptr = ptr->plist;

    } while ( ptr != NULL );

    if ( first ) printf("\n");
}

#endif

/*
 *   Either load the graph leaves information or count them
 *
 *  INPUT
 *	ptr		root node of the pool sub graph to load
 *
 *  OUTPUT
 *	count		counter ( initialized before the call )
 * 	poolidx		pool index list ( only if not NULL in input )
 *	priority	priority list ( only if poolidx is not NULL in input )
 */
void loadgraphinfo(ptr,count,poolidx,priority)
pnode_t *ptr;
int *count;
index_t_s *poolidx;
int *priority;
{
    do {
	if ( !ptr->skip )
	    if ( ptr->pexp )
		(void)loadgraphinfo(ptr->pexp,count,poolidx,priority);
	    else {
		if ( poolidx ) {
		    poolidx[*count] = ptr->pidx;
		    priority[*count] = ptr->lev;
		}
		(*count)++;
	    }
	ptr = ptr->plist;

    } while ( ptr != NULL );
}

/*
 *  Deallocate the pool graph
 *
 *  INPUT
 *	ptr		root node of the pool sub graph to deallocate
 */
void freegraph(ptr)
pnode_t *ptr;
{
    if ( ptr->pexp ) (void)freegraph(ptr->pexp);

    if ( ptr->plist ) (void)freegraph(ptr->plist);

    if ( ptr->pname ) free(ptr->pname);
    if ( ptr->penv ) free(ptr->penv);
    free(ptr);
}

/*
 *  Detect cycles in a given pool graph path
 *
 *  INPUT
 *	leaf		node at the end of the graph path
 *	ptr		current node in the graph path ( = leaf at first call )
 *	cycle		0 if no cycle was detected and 1 otherwise ( < 0 at first call )
 *
 *  OUTPUT
 *	cycle		0 if no cycle was detected and 1 otherwise
 *	history		path containing the cycle
 */
void detect_cycle(leaf,ptr,cycle,history)
pnode_t *leaf,*ptr;
int *cycle;
string *history;
{
    *cycle = *cycle < 0 ? 0 : *cycle || !strcmp(leaf->pname,ptr->pname);

    if ( ptr->pback != NULL ) {
	(void)detect_cycle(leaf,ptr->pback,cycle,history);
	if ( *cycle ) {
	    CAT_string(history,"->");
	    CAT_string(history,ptr->pname);
	}
    }
    else
	if ( *cycle ) CPY_string(history,ptr->pname);
}

/*
 *  Check if there is a copy of the given node 'leaf' in the graph pointed to by 'ptr'
 *
 *  INPUT
 *	leaf		node to search
 *	ptr		root node of the pool sub graph to scan
 */
int detect_copy(leaf,ptr)
pnode_t *leaf,*ptr;
{
    if ( leaf == ptr ) return(0);

    if ( !strcmp(leaf->pname,ptr->pname) ) return(1);

    if ( ptr->pexp && detect_copy(leaf,ptr->pexp) ) return(1);

    if ( ptr->plist && detect_copy(leaf,ptr->plist) ) return(1);

    return(0);
}

/*
 *  Allocate a graph node
 */
pnode_t *allocate_pnode()
{
  static pnode_t *p;
    if ( (p = (pnode_t*)malloc(sizeof(pnode_t))) == NULL )
	vperror(1,"malloc(%d)",sizeof(pnode_t));
    else
	INIT_pnode(p);
    return(p);
}

/*
 *
 */
int expand_environment(ptr)
pnode_t *ptr;
{
  static char *e,poolenv[ENVPREFIX_LENGTH+POOL_LENGTH];

    sprintf(poolenv,"%s%s",ENVPREFIX,ptr->pname);
    if ( (e = getenv(poolenv)) == NULL ) {
	vperror(0,"unknown shift pool <%s>",ptr->pname);
	return(EXOPT);
    }
    if ( (ptr->penv = strdup(e)) == NULL ) {
	vperror(1,"malloc(%d)",strlen(e)+1);
	return(EXSYS);
    }
    return(1);
}

/*
 *
 */
int expand_superpool(ptr,supidx)
pnode_t *ptr;
index_t_s supidx;
{
  extern supool_t *supool_table;
  static int len;
  static index_t_s idx;

    len = 0;
    for (idx=0; idx<supool_table[supidx].idx; idx++)
	len += strlen(supool_table[supidx+1+idx].name)+1;
    if ( (ptr->penv = malloc(len)) == NULL ) {
	vperror(1,"malloc(%d)",len);
	return(EXSYS);
    }
    *ptr->penv = '\0';
    for (idx=0; idx<supool_table[supidx].idx; idx++) {
	strcat(ptr->penv,supool_table[supidx+1+idx].name);
	if ( idx < supool_table[supidx].idx - 1 )
	    strcat(ptr->penv,supool_table[supidx+1+idx].idx?":":"+");
    }
    return(1);
}

/*
 *  Build the pool graph
 */
int buildgraph(root,ptrhead,pool,history)
pnode_t *root,*ptrhead;
char *pool;
string *history;
{
  pnode_t *ptr;
  index_t_s supidx;
  char *p;
  char *str,*strp,strch;
  int stri,cycle,reply;
  boolean first;

    first = pool != NULL ;

 /*
  *  Get the current pool name list to scan
  */
    if ( !first )
	str = ptrhead->penv;
    else
	str = pool;

 /*
  *  Look for the first token ( pool name ) in the pool name list
  */
    stri = 0;
    p = NULL;
    if ( str[stri] != '\0' ) {
    	p = str+stri;
    	while ( str[stri]!=PSCCH && str[stri]!=PSPCH && str[stri]!='\0' ) stri++;
    	strp = str+stri;
    	strch = *strp;
    	*strp = '\0';
    }

 /*
  *  Set the current node ( ptr ) and the backward link
  */
    if ( !first ) {
	if ( (ptrhead->pexp = allocate_pnode()) == NULL ) return(EXSYS);
	ptr = ptrhead->pexp;
	ptr->pback = ptrhead;
    }
    else {
	ptr = ptrhead;
	ptr->pback = NULL;
    }

 /*
  *  Scan the pool name list 
  */
    do { 

    	if ( (ptr->pname = strdup(p)) == NULL ) {
	    vperror(1,"malloc(%d)",strlen(p)+1);
	    return(EXSYS);
     	}
	*strp = strch;

        if ( strch == '\0' )
	    if ( ptr->pback )
		ptr->lev = ptr->pback->lev;
	    else
		ptr->lev = -1;
	else
	    ptr->lev = strch == PSPCH ? PSPCD : PSCCD ;

     /*
      *  Check if there is any cycle
      */
	cycle = -1;
	(void)detect_cycle(ptr,ptr,&cycle,history);
	if ( cycle ) {
	    vperror(0,"cycle detected in pool definition <%s>",GET_string(history));
	    return(EXOPT);
	}

     /*
      *  Check if there is any copy of the current node
      */
	if ( !(ptr->skip = detect_copy(ptr,root)) ) {

	 /*
	  *  Expand the current pool name
	  */
	    if ( (ptr->pidx = look_pool(ptr->pname)) < 0 ) {

		if ( (supidx = look_supool(ptr->pname)) < 0 ) {
		    if ( (reply = expand_environment(ptr)) < 0 ) return(reply);
		}
		else
		    if ( (reply = expand_superpool(ptr,supidx)) < 0 ) return(reply);

		if ( (reply = buildgraph(root,ptr,NULL,history)) < 0 ) return(reply);

	    }

	} /* skip */

     /*
      *  Look for the next token ( pool name ) in the pool name list
      */
 	p = NULL;
	while ( str[stri]==PSCCH || str[stri]==PSPCH ) stri++;
	if ( str[stri] != '\0' ) {
    	    p = str+stri;
    	    while ( str[stri]!=PSCCH && str[stri]!=PSPCH && str[stri]!='\0' ) stri++;
    	    strp = str+stri;
    	    strch = *strp;
    	    *strp = '\0';
	}

     /*
      *  Set the current node ( ptr ) and the backward link 
      */
	if ( p != NULL ) {
	    if ( (ptr->plist = allocate_pnode()) == NULL ) return(EXSYS);
	    ptr = ptr->plist;
	    if ( !first )
		ptr->pback = ptrhead;
	    else
		ptr->pback = NULL;
	}

    } while ( p != NULL ) ;

    *strp = strch;
    return(1);
}

/*
 *  Get the pool list and associated priorities
 */
int getpoollist(pool,poolidx,priority,length)
char *pool;
index_t_s **poolidx;
int **priority;
int *length;
{
  pnode_t *root;
  int reply;
  string history;

    INIT_string(&history,0);
    if ( (root = allocate_pnode()) == NULL ) return(EXSYS);

    if ( (reply = buildgraph(root,root,pool,&history)) > 0 ) {
#ifdef DEBUG
	(void)printgraph(root,1); */
#endif
	*length = 0;
	(void)loadgraphinfo(root,length,NULL,NULL);

	if ( (*poolidx = (index_t_s*)malloc(sizeof(index_t_s)*(*length))) == NULL ) {
	    vperror(1,"malloc(%d*%d)",sizeof(index_t_s),length);
	    reply = EXSYS;
	}
	else
	if ( (*priority = (int*)malloc(sizeof(int)*(*length))) == NULL ) {
	    vperror(1,"malloc(%d*%d)",sizeof(int),length);
	    free(*poolidx);
	    reply = EXSYS;
	}
	else {
	    *length = 0;
	    (void)loadgraphinfo(root,length,*poolidx,*priority);
	}
    }

    FREE_string(&history);
    freegraph(root);
    return(reply);
}

/*
 *  Build the file system nfs-path list of a pool
 *
 *  INPUT
 *	pool		pool name
 *	uid		uid of the user whose home directory is used
 *
 *  OUTPUT
 *	nofs		length of the file system list
 *	poolfs		the file system list
 *
 *  RETURN
 *	1		successful completion
 *     < 0		an error occured ( EXOPT - EXSYS )
 */
int getnfspaths(pool,uid,gid,nofs,poolfs)
char *pool;
int uid,gid;
int *nofs;
poolfs_t **poolfs;
{
  extern fsys_t *fsys_table;		/*  pointer to the file system table	*/
  extern index_t_s fsysidx;		/*  length of the file system table	*/
  static int length		= 0;
  index_t_s idx,k,*poolidx;
  int *priority;
  int nopools,first,reply;

 /*
  *  Get the pool list
  */
    if ( (reply = getpoollist(pool,&poolidx,&priority,&nopools)) < 0 )
	return(reply);

 /*
  *  Compute the number of file system included in the pool
  */
    *nofs = 0;
    for (idx=0;idx<fsysidx;idx++)
	for (k=0; k<nopools; k++)
	    if ( fsys_table[idx].poolidx == poolidx[k] ) (*nofs)++;

 /*
  *  Allocate the memory required by the poolfs array
  */
    if ( *nofs > length ) {
	if ( !length ) {
	    if ( (*poolfs=(poolfs_t*)malloc(sizeof(poolfs_t)*(*nofs))) == NULL ) {
		vperror(1,"malloc(%d*%d)",sizeof(poolfs_t),(*nofs));
		return(EXSYS);
	    }
	}
	else
	if ( (*poolfs=(poolfs_t*)realloc(*poolfs,sizeof(poolfs_t)*(*nofs))) == NULL ) {
	    vperror(1,"realloc(%x,%d*%d)",*poolfs,sizeof(poolfs_t),(*nofs));
	    return(EXSYS);
	}
	length = *nofs;
   }

 /*
  *  Build the poolfs array
  */
    *nofs = 0;
    for (k=0; k<nopools; k++) {
	first = -1;
	for (idx=0; idx<fsysidx; idx++)
	    if ( fsys_table[idx].poolidx == poolidx[k] ) {
		(*poolfs)[*nofs].pidx = poolidx[k];
		(*poolfs)[*nofs].fsidx = idx;
		(*poolfs)[*nofs].level = priority[k];
		if ( first < 0 ) {
		    first = *nofs;
		    (*poolfs)[first].num = 0;
		}
		(*poolfs)[first].num++;
		create_mountpath(idx,(*poolfs)[*nofs].mount);
		if ( uid < 0 && gid < 0 )
		    strcpy((*poolfs)[*nofs].dir,(*poolfs)[*nofs].mount);
		else
		    if ( create_nfspath(idx,uid,gid,NULL,(*poolfs)[*nofs].dir) < 0 )
			return(EXSYS);
		(*nofs)++;
	    }
    }

    return(1);
}

/*
 *  Check the client be authorized to remove the shift file on disk
 *
 *  INPUT
 *	clientuid	uid of the user calling sfrm
 *	pathuid		uid of the user whose home directory is used
 *	pathgid		gid of the user whose home directory is used
 *	buffer		information about the file
 *
 *  RETURN
 *	1		the client is authorized
 *	0		the client is NOT authorized
 */
int authorized_to_remove(clientuid,pathuid,pathgid,buffer)
int clientuid,pathuid,pathgid;
struct stat *buffer;
{
    return(
 /*
  *  The client must be the owner of either the file or the directory containing it.
  */ 
	    buffer->st_uid == clientuid || clientuid == pathuid || 
 /*
  *  If root is the owner of the file and the gid is not equal to the pathgid then
  *  it's assumed that the file was being created but the creation operation wasn't
  *  completed ( see create_file() )
  */
	    ( buffer->st_uid == 0 && buffer->st_gid != pathgid )
	  );
}

/*
 *  Check the client be authorized to create a shift file on disk
 *
 *  INPUT
 *	poolidx		pool index
 *	clientuid	uid of the user calling sfget
 *	clientgid	gid of the user calling sfget
 *
 *  RETURN
 *	1		the client is authorized
 *     < 0		an error occured ( EXPERM - EXOPT )
 */
int authorized_to_create(poolidx,clientuid,clientgid)
int poolidx;
int clientuid,clientgid;
{
  extern pool_t *pool_table;
  extern accs_t *accs_table;		/*  pointer to the shift account table	*/
  extern index_t_s accsidx;		/*  length of the shift account table	*/
  index_t_s idx;
  user_t client;

 /*
  *  Look for the client in the account table
  */
    for (idx=0; idx<accsidx; idx++)
	if ( accs_table[idx].poolidx == poolidx &&
	      (accs_table[idx].uid == clientuid || accs_table[idx].uid == ANY_GID) &&
		(accs_table[idx].gid == clientgid || accs_table[idx].gid == ANY_GID) )
	    return(1);

 /*
  *  Print an error message
  */
    if ( get_userbyid(client,clientuid,NULL) < 0 )
	sprintf(client,"uid %d",clientuid,clientgid);

    vperror(0,"You (%s) are not authorized to create file in pool <%s>",
		client,
		pool_table[poolidx].name);
    return(EXPERM);
}

/*
 *  Scan the shift disk pool on behalf of root
 *
 *  INPUT
 *	mode		scan mode ( 0/sfrm - 1/first of sfget - 2/second of sfget )
 *	client		uid of the user calling sfrm
 *	uid		uid of the user whose home directory is used
 *	gid		gid of the user whose home directory is used
 *	filename	file name supplied by the user
 *	nofs		length of nfspaths
 *	poolfs		list of file system path making the pool
 *
 *  OUTPUT
 *	scan		information about the pool
 *
 *  RETURN
 *	1		successful completion
 *	-1		an error occured ( malloc() )
 */
int scandiskpool(mode,client,uid,gid,filename,nofs,poolfs,scan,atomic,length)
int mode;
int client,uid,gid;
char *filename;
int nofs;
poolfs_t *poolfs;
scanp_t **scan;
boolean atomic;
int *length;
{
  int idx;

#if defined( TRACE )
    if ( TRACE_dosfget || TRACE_dosfrm )
	switch ( mode ) {
	    case 0 : printf("sfrm scan\n"); break;
	    case 1 : printf("first sfget scan\n"); break;
	    case 2 : printf("second sfget scan\n"); break;
	}
#endif

 /*
  *  Allocate the memory required by the scanpool array
  */
    if ( *scan == NULL || nofs > *length ) {
	if ( *scan == NULL || !(*length) ) {
	    if ( (*scan = (scanp_t*)malloc(sizeof(scanp_t)*nofs)) == NULL ) {
		vperror(1,"malloc(%d*%d)",sizeof(scanp_t),nofs);
		return(-1);
	    }
	}
	else
	    if ( (*scan = (scanp_t*)realloc(*scan,sizeof(scanp_t)*nofs)) == NULL ) {
		vperror(1,"realloc(%x,%d*%d)",*scan,sizeof(scanp_t),nofs);
		return(-1);
	    }
	*length = nofs;
    }

    for (idx=0; idx<nofs; idx++) {

#if defined( CREATMP )
	if ( atomic ) {
	 /*
	  *  Look for the temporary file
	  */
	    strcpy((*scan)[idx].p_tmp,poolfs[idx].dir);
	    strcat((*scan)[idx].p_tmp,filename);
	    strcat((*scan)[idx].p_tmp,".tmp");
	    (*scan)[idx].tmp = is_file_there((*scan)[idx].p_tmp,&(*scan)[idx].stat);
#if defined( TRACE )
	    if ( TRACE_scan_pool )
		if ( (*scan)[idx].tmp > 0 ) printf("found tmp %s\n",(*scan)[idx].p_tmp);
#endif
	}
#endif

     /*
      *  Look for the shift file
      */
	strcpy((*scan)[idx].p_file,poolfs[idx].dir);
	strcat((*scan)[idx].p_file,filename);
	(*scan)[idx].file = is_file_there((*scan)[idx].p_file,&(*scan)[idx].stat);
#if defined( TRACE )
	if ( TRACE_scan_pool )
	    if ( (*scan)[idx].file > 0 ) printf("found file %s\n",(*scan)[idx].p_file);
#endif

     /*
      *  If the shift file exists then check the authorization
      */
	switch ( (*scan)[idx].file ) {
	    case -1 :
	    case 0 :
		(*scan)[idx].scerrno = errno;
		break;
	    case 1 :
		(*scan)[idx].perm = 
			authorized_to_remove(client,uid,gid,&(*scan)[idx].stat);
		break;
	} /* switch - is_file_there() reply */
    }

    return(1);
}

#define rootuid		(int)0

static int clientuid = -1;
static int clientgid = -1;

/*
 *  Set the root ids
 */
int beroot()
{
  static int rootgid = -1;

    if ( clientuid < 0 ) {
	clientuid = getuid();
	clientgid = getgid();
    }
    if ( rootgid < 0 && get_userbyid((char*)NULL,rootuid,&rootgid) < 0 )
	rootgid = 0;
    return( setuid(rootuid) < 0 || setgid(rootgid) < 0 ? -1 : 1 );
}

/*
 *  Set the client ids
 */
int beclient()
{
    if ( clientuid < 0 ) {
	clientuid = getuid();
	clientgid = getgid();
    }
    return( setgid(clientgid) < 0 || setuid(clientuid) < 0 ? -1 : 1 );
}

/*
 *  Get the group ids of the client and of the user whose name is in the nfs-path
 *
 *  RETURN
 *	1		successful completion
 *	-1		an error occured
 */
int getgroupids(clientuid,clientgid,client,pathuid,pathgid,user)
int clientuid,*clientgid;
char *client;
int pathuid,*pathgid;
char *user;
{
 /*
  *  Get the group id of the client
  */
    if ( get_allbyuid(client,(char*)NULL,clientgid,clientuid) < 0 ) {
	if ( *clientgid < 0 )
	    vperror(0,"unknown uid <%d>",clientuid);
	else
	    vperror(1,"getgrgid(%d)",*clientgid);
	return(-1);
    }

 /*
  *  Get the id of the group name that is a component of the nfs-path
  */
    if ( get_allbyuid(user,(char*)NULL,pathgid,pathuid) < 0 ) {
	if ( *pathgid < 0 )
	    vperror(0,"unknown uid <%d>",pathuid);
	else
	    vperror(1,"getgrgid(%d)",*pathgid);
	return(-1);
    }

    return(1);
}

/*
 *
 */
void sfexit(exitcode)
int exitcode;
{
    if ( exitcode > 1 ) {
	vperror(0,"Internal: exit code (%d) out of range [1,0,-1,-2,...]",exitcode);
	exitcode = EXINT;
    }
    exit( exitcode < 0 ? 1 - exitcode : exitcode );
}
