/*
 * Copyright (C) 1990,1991,1992 by CERN/CN/SW/DC
 * All rights reserved
 */

#ifndef lint
static char sccsid[] = "@(#)sfrn.c	1.3 06/25/92 CERN-SW/DC Fabrizio Cane";
#endif /* not lint */

/*

  Moving a shift file across the file systems ( 'shift move' ) is an operation that:

    1 - copies the data contained in the original file to a file copy
    2 - sets the mode and owner of the copy to the mode and owner of the original file
    3 - removes the original file

  The above operations should be executed under the following constraint:

    4 - during the 'shift move' only one valid copy must be accessible through sfget 

  Thus the shift move:

    6 - copies 'original-file' 'temporary-file'
    7 - set the mode and owner of the 'temporary-file'
    8 - remove 'original-file' and rename 'temporary-file' 'original-file'

   The constraint (4) is valid if (8) is atomic.

   sfrn does steps (7) and (8)

   sfmvfs does step (6)

*/

#ifndef apollo 
#include <unistd.h>
#endif
#include <string.h>
#include <errno.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <shift_types.h>
#include <optutil.h>
#include <pwd.h>

#define LEAVE_DESCR	"leave the source file"
#define PREFIX_DESCR	"rename the source file (.prefix.source)"
#define REMOVE_DESCR	"remove the source file"

boolean       unix_style;

/*
 *  sfrn usage
 *
 *  exit code
 *	0	ok
 *	1	bad usage
 *	2	system call failures
 *	3	can't get state of the original file - stat() failure
 *	4	can't change state of the temporary file - chmod()/chown() failure
 *	5	can't rename the temporary file - rename() failure
 *	6	can't remove the original file - unlink() failure
 *	7	can't rename the original file - rename() failure
 */
void usage(argv,help)
char *argv[];
boolean help;
{
    printf("usage: %s -help\n",argv[0]);
    if ( unix_style )
	printf("       %s [-l] [-p<prefix>] [-r] source target\n",argv[0]);
    else
	printf("       %s [-leave] [-prefix <string>] [-remove] source target\n",argv[0]);
    if ( help ) {
	printf("where:\n");
	printf("  source %*s shift source file path\n",31,"");
	printf("  target %*s shift target file path\n",31,"");
	HELP_options();
    }
}

/*
 *  sfrn
 */
main(argc,argv)
int   argc;
char  *argv[];
{
  char		*userpathsrc,*userpathtrg,*prefix;
  string	source,target,old;
  struct stat   buffer;
  boolean	remove,leave;
  boolean       show_help;
  boolean       show_options;
  char	        *options_style;

 /*
  *  Get the options style desired
  */
    options_style = getenv("DPMOPTIONSTYLE");
    unix_style = options_style == NULL || strcasecmp(options_style,"XWINDOW");

 /*
  *  Define the options available
  */
    INIT_options();

    if ( unix_style )
	DEFAULT_mask(UNIX_MASK);
    else
	DEFAULT_mask(VMS_MASK);

    DEFAULT_mask(FILIST | ALLOC | DESCR);

    ADD_OPT_options(&show_help,&show_options);

    ADD_options("leave",EXTSEL,NONE,&leave,LEAVE_DESCR);
    ADD_options("prefix",EXTVAL,STRING,&prefix,PREFIX_DESCR);
    ADD_options("remove",EXTSEL,NONE,&remove,REMOVE_DESCR);

 /*
  *  Initialize the options value before calling READ_options()
  */
    prefix = NULL;

 /*
  *  Get the option values
  */
    if ( READ_options(argc,argv) < 0 ) {
	usage(argv,FALSE);
	exit(1);
    }

    if ( show_help && NO_opts()==1 ) {
	usage(argv,TRUE);
	exit(0);
    }

    if ( show_options )
	LIST_options();

 /*
  *  Analyse the options
  */
    switch ( ((leave>0) + (remove>0) + (prefix!=NULL)) ) {
	case 0  : vperror(0,"option required");
		  usage(argv,FALSE);
		  exit(1);
	case 1  : break;
	default : vperror(0,"too many options");
		  usage(argv,FALSE);
		  exit(1);
    }

    if ( NO_args() != 2 ) {
	vperror(0,"%s-path required",NO_args() ? "target" : "source");
	usage(argv,FALSE);
	exit(1);
    }

    if ( (userpathsrc = strrchr(argv[argc-2],'/')) == NULL ) {
	vperror(0,"invalid source file path %s\n",argv[argc-2]);
	usage(argv,FALSE);
	exit(1);
    }
    *userpathsrc = '\0';
    userpathsrc++;

    if ( (userpathtrg = strrchr(argv[argc-1],'/')) == NULL ) {
	vperror(0,"invalid target file path %s\n",argv[argc-1]);
	usage(argv,FALSE);
	exit(1);
    }
    *userpathtrg = '\0';
    userpathtrg++;

    INIT_string(&source,100);
    CAT_string(&source,argv[argc-2]);
    CAT_string(&source,"/");
    CAT_string(&source,userpathsrc);

    if ( LEN_string(&source) < 0 ) {
	vperror(1,"malloc/realloc()");
	exit(2);
    }

 /* 
  *  Get mode and onwer of the original file 
  */
    if ( stat(GET_string(&source),&buffer) < 0 ) {
	vperror(1,"stat(%s)",GET_string(&source));
	exit(3);
    }

    INIT_string(&old,0);
    if ( DUP_string(&old,&source) < 0 )
	exit(2);

    CPY_string(&source,argv[argc-1]);
    CAT_string(&source,"/");
    CAT_string(&source,userpathtrg);

    INIT_string(&target,100);
    CAT_string(&target,argv[argc-1]);
    CAT_string(&target,"/");
    CAT_string(&target,userpathsrc);

    if ( LEN_string(&source) < 0 || LEN_string(&target) < 0 ) {
	vperror(1,"malloc/realloc()");
	exit(2);
    }

 /* 
  *  Change the mode and owner of the temporary file ( to the original file )
  */
    if ( chown(GET_string(&source),buffer.st_uid,buffer.st_gid) < 0 ||
				chmod(GET_string(&source),buffer.st_mode) < 0 ) {
	vperror(1,"chown/chmod(%s)",GET_string(&source));
	exit(4);
    }

 /*
  *  Rename the temporary file ( to the target file )
  */
    if ( rename(GET_string(&source),GET_string(&target)) < 0 ) {
	vperror(1,"rename(%s,%s)",GET_string(&source),GET_string(&target));
	exit(5);
    }

    if ( remove ) {

     /* 
      *  Remove the original file
      */
	if ( unlink(GET_string(&old)) < 0 ) {
	    vperror(1,"unlink(%s)",GET_string(&old));
	    exit(6);
	}
    }
    else 
    if ( prefix != NULL ) {

	CPY_string(&target,argv[argc-2]);
	CAT_string(&target,"/.");
	CAT_string(&target,prefix);
	CAT_string(&target,".");
	CAT_string(&target,userpathsrc);

     /* 
      *  Rename the original file ( to a special file name containing the prefix )
      */
        if ( rename(GET_string(&old),GET_string(&target)) < 0 ) {
	    vperror(1,"rename(%s,%s)",GET_string(&old),GET_string(&target));
	    exit(7);
        }
    }

    exit(0);
}
