/*
 * Copyright (C) 1992 by CERN/CN/SW/DC
 * All rights reserved
 */

#ifndef lint
static char sccsid[] = "@(#)sfsh.c	3.8 07/26/94 CERN-SW/DC Fabrizio Cane";
#endif /* not lint */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <shift_types.h>
#include <strutil.h>
#include <strerror.h>
#include <strdup.h>
#include <optutil.h>
#include <parser.h>
#include <table.h>
#include <shiftconf.h>
#include <getconfent.h>
#if defined(SOLARIS)
#include <sys/socket.h>
#include <netdb.h>
#endif
#include <dpmutil.h>

#define INIT_SHELL_LEN		100

#define POOL_DESCR		"name of the shift disk pool to be used"
#define USER_DESCR		"user whose home directories are used"
#define GROUP_DESCR		"group whose directories are used"
#define VERBOSE_DESCR		"the standard error is not suppressed"
#define DIRECTORY_DESCR		"show the home directories path"
#define CSHELL_DESCR		"use the csh to execute the command"
#ifndef RFIO
#define RSHELL_DESCR		"use the rsh to execute the command"
#else
#define RFIO_DESCR		"use rfio to execute the command"
#endif /* RFIO */
#define SPACE_DESCR		"show the file system space statistic"
#define HOST_DESCR		"remote host to be used with rshell option"

int serrno;		/* required because sfsh is not linked to the correct library */
int maxtmpattempts;	/* not used but required to link */
int tmpsleep;		/* not used but required to link */

boolean  option_style;	/* option style flag */

/*
 *  sfsh usage
 */
void usage(argv,help)
char *argv[];
boolean help;
{
    if ( option_style ) {
	printf("usage: %s -help\n",argv[0]);
	printf("       %s -p<name> [-u<name>] [-c] [-r] [-v] [-d] [-s] ",argv[0]);
	printf("[command [arguments]]\n");
    }
    else {
	printf("usage: %s -help\n",argv[0]);
	printf("       %s -pool <name> [-user <name>] [-cshell] [-rshell] ");
	printf("[-verbose]\n\t    [-directory] [-space] [command [arguments]]\n");
    }
    if ( help ) {
	printf("where:\n");
	printf("  command [arguments] %18s shell command to be executed\n","");
	HELP_options();
    }
}

/*
 *  sfsh
 */
main(argc,argv)
int argc;
char *argv[];
{
  extern boolean reloadconfiguration;
  boolean  show_help,show_options;
  char     *pool,*user,*group;
  boolean  verbose,directory,space;
  boolean  cshell,rshell;
  string   command;
  boolean  local;
  char	   *host;
  boolean  rsfsh	= FALSE;
  int	   uid,gid;
  char     *ent;

    clear_vperror();

  /*
   *  Load the DPM and PAIR configurations
   */
    reloadconfiguration = FALSE;
    if ( load_configuration(TRUE,TRUE,TRUE) < 0 ) {
	vperror(0,"can't load the configuration");
	sfexit(EXCONF);
    }

 /*
  *  Get the option style setup
  */
    getoptstyle(&option_style);

 /*
  *  Define the options available
  */
    INIT_options();
    if ( option_style )
        DEFAULT_mask(UNIX_MASK);
    else
	DEFAULT_mask(VMS_MASK);

    DEFAULT_mask(FILIST | NMLIST);
    ADD_OPT_options(&show_help,&show_options);

    DEFAULT_mask(EXTVAL | ALLOC | DESCR);
    ADD_options("pool",ENVIR | NAMENV,STRING,"DPMPOOL",&pool,POOL_DESCR);
    ADD_options("user",ENVIR | NAMENV,STRING,"DPMUSER",&user,USER_DESCR);
    ADD_options("group",ENVIR | NAMENV,STRING,"DPMGROUP",&group,GROUP_DESCR);

    DEFAULT_mask(EXTSEL | NOEXTVAL);
    ADD_options("verbose",0,NONE,&verbose,VERBOSE_DESCR);
    ADD_options("directory",0,NONE,&directory,DIRECTORY_DESCR);
#ifndef RFIO
    ADD_options("cshell",0,NONE,&cshell,CSHELL_DESCR);
    ADD_options("rshell",0,NONE,&rshell,RSHELL_DESCR);
#else
    ADD_options("rfio",0,NONE,&rshell,RFIO_DESCR);
#endif /* RFIO */
    ADD_options("space",0,NONE,&space,SPACE_DESCR);

    DEFAULT_mask(FIXED_MASK);
    DEFAULT_mask(EXTVAL | ALLOC | ENVIR | NAMENV | HIDDEN | DESCR | FILIST | NMLIST);
    ADD_options("shifthost",0,STRING,"SHIFTRHOSTSFSH",&host,HOST_DESCR);

 /*
  *  Initialize the option values before calling READ_options()
  */
    pool = NULL;
    user = NULL;
    group = NULL;
    host = NULL;
#ifdef RFIO
  /*
   * rfio is set
   */
     rshell = 1 ;
#endif


 /*
  *  Get the option values
  */
    if ( READ_options(argc,argv) < 0 ) {
        usage(argv,FALSE);
        sfexit(EXOPT);
    }

    if ( show_help && NO_opts()==1 ) {
        usage(argv,TRUE);
        sfexit(EXHELP);
    }

    if ( show_options )
        LIST_options();

 /*
  *  The pool name must be supplied
  */
    getoptpool(&pool,argv,usage);

 /*
  *  Get the either the uid or gid used to build the nfs-path
  */
    getoptuser(user,group,&uid,&gid);

 /*
  *  Get the local/remote mode and rshell/rsfsh mode
  */
    getsfmode(0,&local,NULL);
    rsfsh = ( rshell || !local ) && 
		(ent = getconfent("DPM","REMOTESFSH",0)) && !strcmp(ent,"yes");

 /*
  *  rsfsh mode requires remote host name ( it's optional for the rshell mode )
  */
    if ( host == NULL )
	if ( (ent = getconfent("DPM","RHOSTSFSH",0)) ) {
	    if ( gethostbyname(ent) == NULL ) {
		vperror(1,"gethostbyname(%s)",ent);
		sfexit(EXSYS);
	    }
	    if ( (host = strdup(ent)) == NULL ) {
		vperror(1,"strdup(%s)",ent);
		sfexit(EXSYS);
	    }
	}
	else 
	if ( rsfsh ) {
	    vperror(0,"getconfent(DPM,RHOSTSFSH): not found\nincomplete installation");
	    sfexit(EXCONF);
	}

 /*
  *  Space option and remote shell mode are incompatible
  */
#ifdef RFIO
    if ( rshell && space )
	rshell = 0 ;
#endif
    if ( rshell && !rsfsh && space ) {
	vperror(0,"space option and remote shell mode are incompatible");
	usage(argv,FALSE);
        sfexit(EXOPT);
    }

 /*
  *  Get the command and its arguments
  */
    if ( !NO_args() && !space ) {
	vperror(0,"command required");
	usage(argv,FALSE);
        sfexit(EXOPT);
    }

    {
      int idx;

	INIT_string(&command,100);

	if ( rsfsh ) {
	 /*
	  *  Use the local pool name
	  */
	    CAT_string(&command,"sfsh -p ");
	    CAT_string(&command,pool);
	    CAT_string(&command," ");

	    if ( user ) {
		CAT_string(&command,"-u ");
		CAT_string(&command,user);
		CAT_string(&command," ");
	    }

	    if ( group ) {
		CAT_string(&command,"-g ");
		CAT_string(&command,group);
		CAT_string(&command," ");
	    }

	    if ( cshell )
		CAT_string(&command,"-c ");

	    if ( verbose )
		CAT_string(&command,"-v ");

	    if ( directory )
		CAT_string(&command,"-d ");

	    if ( space ) 
		CAT_string(&command,"-s ");

	    CAT_string(&command,"\"");
	}

	idx = argc - NO_args();
	while ( idx < argc ) {
            CAT_string(&command,argv[idx]);
            CAT_string(&command," ");
            idx++;
	}
	if ( rsfsh ) 
	    CAT_string(&command,"\"");

	if ( LEN_string(&command) < 0 ) {
	    vperror(1,"malloc/realloc()");
	    sfexit(EXSYS);
	}
    }

 /*
  *  Do it
  */
    sfexit( dosfsh( pool,
		    uid,
		    gid,
		    GET_string(&command),
		    verbose,
		    directory,
		    cshell,
		    rshell,
		    space,
		    host,
		    rsfsh )
	  );
}

/*
 *  Do sfsh
 *
 *  INPUT
 *	pool		pool name
 *	pathuid		uid of the user whose home directory is in the nfs-pathname
 *	command		the command and its arguments
 *	verbose		verbose mode
 *	directory	directory mode
 *	cshell		cshell mode
 *	rshell		rshell mode
 *	space		space mode
 *	host		remote host to be used for the rsh
 *	rsfsh		remote sfsh / remote command
 *
 *  RETURN
 *	0		successful completion
 *	1		at least one file system wasn't accessible
 *	EXOPT		option error
 *	EXSYS		system error
 */
int dosfsh(pool,pathuid,pathgid,command,verbose,directory,cshell,rshell,space,host,rsfsh)
char *pool;
int pathuid,pathgid;
char *command;
boolean verbose,directory;
boolean cshell,rshell,space;
char *host;
boolean rsfsh;
{
  extern disk_t *disk_table;
  extern fsys_t *fsys_table;
  extern pool_t *pool_table;
  string shcommand;
  int nofs;
  poolfs_t *poolfs		= NULL;
  char *dshost;
  long tot,free,bsize;
  int idx,width;
  int reply,frpy;
  int pidx;
#ifdef RFIO
  int rc ;
  extern int rfio_errno ;
#endif /* RFIO */

 /*
  *  Build the nfs-path list
  */
    if ( (reply = getnfspaths(pool,pathuid,pathgid,&nofs,&poolfs)) != 1 ) return(reply);

 /*
  *  Get the width of the file system name
  */
    width = UNDEF_IDX;
    for (idx=0; idx<nofs; idx++)
	if ( width == UNDEF_IDX || (int)strlen(poolfs[idx].mount) > width ) width = idx;
    width = -(strlen(poolfs[width].mount)+3);

 /*
  *  Scan the nfspath list
  */
    INIT_string(&shcommand,INIT_SHELL_LEN);
    reply = 0;

 /*
  *  The fflush() calls are useful when the sfsh is called by the remote shell
  */

    pidx = -1;

    while ( nofs-- > 0 ) {

	if ( !rshell && !rsfsh) {
#ifdef RFIO
	    if (!space) {
		fprintf(stderr,"Software compiled with RFIO option. Impossible to execute locally\n");
		return(EXSYS);
	    }
#endif
		

	    if ( pidx != poolfs[nofs].pidx ) {
		if ( pidx > -1 ) printf("\n");
		pidx = poolfs[nofs].pidx;
		printf("POOL %s\n",pool_table[pidx].name);
		fflush(stdout);
	    }

	 /*
	  *  Check the file system be mounted and accessible
	  */
	    if ( (frpy = is_filesystem(poolfs[nofs].mount,NULL,0)) != FSMNTD ) {
		filesystem_error(poolfs[nofs].mount,frpy,errno);
		fflush(stderr);
		reply = 1;
		continue;
	    }

	 /*
	  *  Get the file system space statistic
	  */
	    if ( space || (fsys_table[poolfs[nofs].fsidx].bits & FSYSWLOCK) ) {
		if ( space && getfsystat(poolfs[nofs].mount,&tot,&free,&bsize,NULL) < 0 ) {
		    vperror(1,"statfs(poolfs[nofs].mount)");
		    fflush(stderr);
		    reply = EXSYS;
		    break;
		}
		printf("%*s",width,poolfs[nofs].mount);
		if ( space )
		    printf(" %5s total   %5s free   %4s%% free",
					ftoanu((double)tot*(double)bsize),
					ftoanu((double)free*(double)bsize),
					ftoanu((double)free*100.0/(double)tot));
		if ( fsys_table[poolfs[nofs].fsidx].bits & FSYSWLOCK ) {
		    if ( space ) printf("    ");
		    printf(" write LOCKED");
		}
		printf("\n");
		fflush(stdout);
		if ( strlen(command) == 0 ) continue;
	    }

	 /*
	  *  Change directory
	  */
	    if ( chdir(poolfs[nofs].dir) < 0 ) continue;

	 /*
	  *  Bourne Shell  &  C shell  ->  sh/csh -c "pwd;<command>" 2>/dev/null
	  */
	    if ( !cshell )
		CPY_string(&shcommand,"sh -c \"");
	    else
		CPY_string(&shcommand,"csh -c \"");
	    if ( directory && !space )
		CAT_string(&shcommand,"pwd;");
	    CAT_string(&shcommand,command);
	    CAT_string(&shcommand,"\"");
	    if ( !verbose )
		CAT_string(&shcommand," 2>/dev/null");
	    if ( LEN_string(&shcommand) < 0 ) {
		vperror(1,"malloc/realloc()");
		fflush(stderr);
		reply = EXSYS;
		break;
	    }
	    else {
		system(GET_string(&shcommand));
		fflush(stdout);
		fflush(stderr);
	    }

	} /* cshell or bshell */

	else 

        if ( !rsfsh ) {

	 /*
	  *  Remote Shell  ->  rsh <host> "cd <nfs-path>;pwd;<command>" 2>/dev/null
	  */
#ifdef RFIO
	    CPY_string(&shcommand,"");
#else
	    CPY_string(&shcommand,"rsh ");
#endif /* RFIO */
	    if ( host == NULL )
		dshost = disk_table[fsys_table[poolfs[nofs].fsidx].diskidx].host;
	    else
		dshost = host;
	    CAT_string(&shcommand," cd ");
	    CAT_string(&shcommand,poolfs[nofs].dir);
	    CAT_string(&shcommand,";");
	    if ( directory )
		CAT_string(&shcommand,"pwd;");
	    CAT_string(&shcommand,command);
	    if ( !verbose )
		CAT_string(&shcommand," 2>/dev/null");
	    if ( LEN_string(&shcommand) < 0 ) {
		vperror(1,"malloc/realloc()");
		fflush(stderr);
		reply = EXSYS;
		break;
	    }
	    else {
#ifdef RFIO
		if ( (rc=rfio_system(GET_string(&shcommand),dshost)) < 0  ) {
			rfio_perror("Can't execute command");
			fflush(stderr);
			FREE_string(&shcommand);
			return ( EXSYS );
		}
#else
		system(GET_string(&shcommand));
#endif /* RFIO */
                fflush(stdout);
                fflush(stderr);
            }

	} /* remote command */

	else {

	 /*
	  *  Remote Shell  ->  rsh <host> '<command line>'
	  */
#ifdef RFIO
	    CPY_string(&shcommand,"");
#else
            CPY_string(&shcommand,"rsh ");
            CAT_string(&shcommand,host);
#endif /* RFIO */
	    CAT_string(&shcommand," '");
	    CAT_string(&shcommand,command);
	    CAT_string(&shcommand,"'");

	    if ( LEN_string(&shcommand) < 0 ) {
		vperror(1,"malloc/realloc()");
		fflush(stderr);
		reply = EXSYS;
	    }
	    else {
#ifdef RFIO
		if ( (rc = rfio_system(GET_string(&shcommand),host)) < 0 ){
			rfio_perror("Can't execute command") ;
			fflush(stderr);
			FREE_string(&shcommand);
			return ( EXSYS );
		}
#else
		system(GET_string(&shcommand));
#endif /* RFIO */
                fflush(stdout);
                fflush(stderr);
            }

	 /*
	  *  Do not loop 
	  */
	    break;

	} /* remote sfsh */

    } /* scan list of nfs-pathname */

    FREE_string(&shcommand);

    return( reply );
}
