/*
 * Copyright (C) 1992 by CERN/CN/SW/DC
 * All rights reserved
 */

#ifndef lint
static char sccsid[] = " @(#)tmps.c	3.3 05/07/93  CERN-SW/DC Fabrizio Cane";
#endif /* not lint */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <netinet/in.h>
#include <fcntl.h>
#include <shift_types.h>
#include <strutil.h>
#include <osdep.h>
#include <bits.h>
#include <strdup.h>
#include <parser.h>
#include <table.h>
#include <dpmutil.h>

#if defined( CREATMP )

#define NOESTALE	3		   /* number of tries to recover ESTALE	*/

/*
 *  Read a tmp file being written and removed by a process through the network
 *
 *  INPUT
 *	scan		scan entry structure
 *	data		buffer to store the data
 *	size		expected number of bytes to read
 *
 *  RETURN
 *	2		file disappeared
 *	1		file successfully read
 *	0		read in progress
 */
int tmpfread(scan,data,size)
scanp_t *scan;
char *data;
int size;
{
  static struct stat buffer;
  static int reply;
  static int noestale;

#if defined( TRACE )
  if ( TRACE_read_tmps ) printf("tmpfread() on %s\n",scan->p_tmp);
#endif

  noestale = 0;

/*
 *  Loop on read()
 */
  for (;;) {

 /*
  *  Call read() and switch on its reply
  */
    reply = read(scan->fd,data+scan->offset,size-scan->offset);
#if defined( TRACE )
    if ( TRACE_read_tmps ) printf("read() %d\n",reply);
#endif
    switch ( reply ) {

     /*
      *  At least one byte was read  -  update the offset and return
      */
	default : 
	    if ( (scan->offset += reply) < size )
		return(0); 				    /* read in progress	*/
	    close(scan->fd);
	    return(1); 				 /* read successfully completed	*/

     /*
      *  No byte was read  -  check the file still exist
      */
	case 0 :
	    for (;;) { 					      /* loop on stat()	*/
		if ( stat(scan->p_tmp,&buffer) < 0 ) {
#if defined( TRACE )
		    if ( TRACE_read_tmps ) printf("stat(): %s\n",strerror(errno));
#endif
		    if ( errno == EINTR ) 
			continue;
		    if ( errno == ESTALE && noestale++ < NOESTALE ) {
#if defined( sun )  && ! defined( SOLARIS)
			usleep(100);
#endif
			continue;
		    }
		    close(scan->fd);
		    return(2);				    /* file disappeared	*/
		}
		break;
	    }
#if defined( TRACE )
	    if ( TRACE_read_tmps ) printf("size: %d\n",buffer.st_size);
#endif
	    return(0);					    /* read in progress	*/


     /*
      *  The read() call failed
      */
	case -1 :
#if defined( TRACE )
	    if ( TRACE_read_tmps ) printf("read(): %s\n",strerror(errno));
#endif
	    if ( errno == EINTR ) 
		continue;
	    if ( errno == ESTALE && noestale++ < NOESTALE ) {
#if defined( sun )  && ! defined( SOLARIS)
		usleep(100);
#endif
		continue;
	    }
	    close(scan->fd);
	    return(2);					    /* file disappeared	*/
    }
  }
}

/*
 *  Remove from disk all tmp files found during the scan disk pool
 *
 *  INPUT
 *	nofs		length of scan
 *	scan1		first scan disk pool structure
 *	scan2		second scan disk pool structure
 *	remove		remove/check mode
 */
void remove_tmps(nofs,scan1,scan2,remove)
int nofs;
scanp_t *scan1,*scan2;
boolean remove;
{
  static int idx;
  static scanp_t *scan;

 /*
  *  Check mode consists in removing the tmp files only if the 
  *  creation time is older than a threeshold ( not implemented yet! )
  */
    if ( !remove ) return;

    for (idx=0; idx<nofs; idx++)
	if ( ((scan = scan1) && scan1[idx].tmp > 0 ) ||
			((scan = scan2) && scan2[idx].tmp > 0 ) ) {
#if defined( TRACE )
	    if ( TRACE_remove_tmps ) printf("removing tmp %s\n",scan[idx].p_tmp);
#endif
	    remove_file(scan[idx].p_tmp);
	}
}

/*
 *  Write the disk tmps configuration in the tmp file
 *
 *  INPUT
 *	file		nfs-pathname of the tmp file
 *	nofs		length of the scan structures
 *	scan1		first disk pool scan structure
 *	scan2		second disk pool scan structure
 *
 *  RETURN
 *	2		a tmp file disappeared
 *	1		successful completion
 *     < 0		an error occured ( EXSYS )
 */
int write_tmp(file,nofs,scan1,scan2)
char *file;
int nofs;
scanp_t *scan1,*scan2;
{
  extern time_t config_version;			 /* local configuration version	*/
  static char *data,*bits;
  static int length 		= 0;
  int size,idx,fd;

#if defined( TRACE )
    if ( TRACE_dosfget ) printf("write_tmp()\n");
#endif

 /*
  *  Compute the size of the disk tmps configuration
  */
    size = sizeof(LONG) + nofs/BITSOFBYTE + ( nofs%BITSOFBYTE ? 1 : 0 ) ;

 /*
  *  Allocate memory to build the disk tmps configuration
  */
    if ( size > length ) {
	if ( !length ) {
	    if ( (data = malloc(size)) == NULL ) {
		vperror(1,"malloc(%d)",size);
		return(EXSYS);
	    }
	}
	else {
	    if ( (data = realloc(data,size)) == NULL ) {
		vperror(1,"realloc(%x,%d)",data,size);
		return(EXSYS);
	    }
	}
	length = size;
	bits = data + sizeof(LONG);
    }

 /*
  *  Build in memory the disk tmps configuration
  */
    memset(bits,(int)0x00,size-sizeof(LONG));

    *(long*)data = htonl((long)config_version);

    for (idx=0; idx<nofs; idx++) {

     /*
      *  If a tmp file disappeared ( after the first scan ) then return
      */
	if ( scan1[idx].tmp > 0 && scan2[idx].tmp < 1 )
		return(2);			      /* a tmp file disappeared	*/

     /*
      *  Set the bit associated to the file system containing a tmp file
      */
	if ( scan1[idx].tmp > 0 || scan2[idx].tmp > 0)
		BIT_SET(bits,idx)
    }

 /*
  *  Open the tmp file
  */
    if ( (fd = open(file,O_WRONLY,0)) < 0 )
	switch ( errno ) {
	    case ENOENT :
	    case ESTALE :
		return(2);			    /* the tmp file disappeared	*/
	    default :
		vperror(1,"open(%s,w)",file);
		return(EXSYS);						/* error */
	}

 /*
  *  Write the disk tmps configuration in the tmp file
  */
    if ( write(fd,data,size) < 1 )
	switch ( errno ) {
	    case EBADF :
	    case ESTALE :
		return(2);			    /* the tmp file disappeared	*/
	    default :
		vperror(1,"write(%s[%d])",file,fd);
		return(EXSYS);						/* error */
	}

 /*
  *  Close the tmp file and return
  */
    close(fd);

#if defined(TRACE)
    if ( TRACE_read_tmps ) dumptmp(nofs,data);
#endif

    return(1);					      /* successfully completed	*/
}

/*
 *  Open a tmp file for reading
 *
 *  INPUT
 *	scan		scan entry structure
 *
 *  RETURN
 *	2		the tmp file doesn't exist
 *	1		successful completion
 *     < 0 		an error occured ( EXSYS )
 */
int open_tmp(scan)
scanp_t *scan;
{
    if ( (scan->fd = open(scan->p_tmp,O_RDONLY,0)) < 0 ) {
#if defined( TRACE )
	if ( TRACE_read_tmps ) printf("Can't open %s: %s\n",scan->p_tmp,strerror(errno));
#endif
	if ( errno != ENOENT ) {
	    vperror(1,"open(%s)",scan->p_tmp);
	    return(EXSYS);
	}
	return(2);
    }
#if defined( TRACE )
    if ( TRACE_read_tmps ) printf("open %s\n",scan->p_tmp);
#endif
    scan->offset = 0;
    return(1);
}

/*
 *  Read the tmp files on disk
 *
 *  INPUT
 *	nofs		length of poolfs and scan structures
 *	poolfs		list of file system making the pool
 *	scan1		first disk pool scan structure
 *	scan2		either the second disk pool scan structure or NULL
 *	fsidx		previous selected file system index unless scan2 is NULL
 *	scidx		scan index associated to fsidx unless fsidx is undefined
 *
 *  OUTPUT
 *	fsidx		selected file system index
 *	scidx		scan index associated to fsidx
 *
 *  RETURN
 *	3	wrong version
 *	2	a tmp file disappeared - read timed out
 *	1	successful completion ( file system selected )
 *     < 0	an error occured ( EXSYS - EXINT - EXCONF )
 */
int read_tmps(nofs,poolfs,scan1,scan2,fsidx,scidx)
int nofs;
poolfs_t *poolfs;
scanp_t *scan1,*scan2;
int *fsidx,*scidx;
{
  extern time_t config_version;			 /* local configuration version	*/
  extern int maxtmpattempts;			   /* sfget configuration value	*/
  extern int tmpsleep;				   /* sfget configuration value	*/
  static char *data,*know,*tmps;
  static int length 			= 0;
  int reply,datasize,tmpsize,bitsize,attempts,loop;
  int notmp,tmpidx,idx,idx2;

#if defined( TRACE )
    if ( TRACE_read_tmps ) printf("read_tmps(%d)\n",scan2 ? 2 : 1);
#endif

 /*
  *  Compute the size of the disk tmps configurations
  */
    bitsize = nofs/BITSOFBYTE + ( nofs%BITSOFBYTE ? 1 : 0 );
    tmpsize = sizeof(LONG) + bitsize ;
    datasize = sizeof(LONG) + 3*bitsize;

 /*
  *  Allocate memory to store the disk tmps configurations
  */
    if ( datasize > length ) {
	if ( !length ) {
	    if ( (data = malloc(datasize)) == NULL ) {
		vperror(1,"malloc(%d)",datasize);
		return(EXSYS);					       /* error	*/
	    }
	}
	else {
	    if ( (data = realloc(data,datasize)) == NULL ) {
		vperror(1,"realloc(%x,%d)",data,datasize);
		return(EXSYS);					       /* error	*/
	    }
	}
	length = datasize;

     /*
      *  The 'know' mask is the tmp locations instersection
      */
	know = data + tmpsize;

     /*
      *  The 'tmps' mask specifies the known tmp locations
      */
	tmps = know + bitsize;

    }

 /*
  *  Initialize the bit masks
  */
    memset(know,(int)0xFF,bitsize);
    memset(tmps,(int)0x00,bitsize);

 /*
  *  Check there be either zero or one or more then one tmp file in the scan structures
  */
    notmp = 0;
    for (idx=0; idx<nofs; idx++)
	if ( scan1[idx].tmp > 0 || ( scan2 && scan2[idx].tmp > 0 ) ) {
	    if ( ++notmp > 1 ) break;
	    tmpidx = idx;
	}

 /*
  *  If there are no tmp file then return error
  */
    if ( notmp == 0 ) {
	vperror(0,"Internal: no tmp file in read_tmps()");
	return(EXINT);					      /* internal error	*/
    }

 /*
  *  If there is only the tmp file created in write_tmp() then return
  */
    if ( scan2 && notmp == 1 && poolfs[tmpidx].fsidx == *fsidx ) {
#if defined( TRACE )
	if ( TRACE_read_tmps ) printf("only my tmp\n");
#endif
	return(1);					/* file system selected	*/
    }
#if defined( TRACE )
    if ( TRACE_read_tmps ) printf("other tmp\n");
#endif

 /*
  *  Open all tmp files
  */
    for (idx=0; idx<nofs; idx++)
	if ( scan1[idx].tmp > 0 || (scan2 && scan2[idx].tmp > 0 ) ) {
	    BIT_SET(tmps,idx);
	    if ( (reply = open_tmp(&scan1[idx])) != 1 )
		return(reply);			/* error - tmp file disappeared	*/
	}

 /*
  *  Read all open tmp files
  */
    attempts = 0;
    do {

	attempts++;

	loop = FALSE;
	for (idx=0; idx<nofs; idx++)
	    if ( BIT_ISSET(tmps,idx) && scan1[idx].fd > 0 ) {

		switch ( tmpfread(&scan1[idx],data,tmpsize) ) {
		    case 2  : return(2);		/* tmp file disappeared	*/
		    case 1  : scan1[idx].fd = -1;
			      break;
		    case 0  : loop = TRUE;
			      continue;
		    default : vperror(0,"Internal: unexpected tmpfread() reply");
			      return(EXINT);			       /* error	*/
		}
#if defined( TRACE )
		if ( TRACE_read_tmps ) dumptmp(nofs,data,1);
#endif

	     /*
	      *  Check the configuration version be correct
	      */
		if ( ntohl(*(long*)data) != config_version ) {
#if defined( TRACE )
		    if ( TRACE_read_tmps ) printf("wrong version\n");
#endif
		    if ( load_configuration(TRUE,FALSE,TRUE) < 0 ) {
			vperror(0,"can't load the configuration");
			return(EXCONF);			 /* wrong configuration	*/
		    }
		    return(3);			      /* configuration reloaded	*/
		}

	     /*
	      *  Make intersection of the tmp configurations
	      */
		for (idx2=0; idx2<bitsize; idx2++)
		    know[idx2] = know[idx2] & data[sizeof(LONG)+idx2];

	     /*
	      *  Open the unknown tmp files
	      */
		if ( scan2 == NULL )
		    for (idx2=0; idx2<nofs; idx2++)
			if ( BIT_ISSET(data+sizeof(LONG),idx2) && 
							!BIT_ISSET(tmps,idx2) ) {
			    BIT_SET(tmps,idx2);
			    if ( (reply = open_tmp(&scan1[idx2])) != 1 )
				return(reply);  /* error - tmp file disappeared	*/
			}

	} /* if open tmp file */

     /*
      *  Sleep for a while if not all tmp files have been read
      */
	if ( attempts < maxtmpattempts && loop ) {
#if defined( TRACE )
	    if ( TRACE_read_tmps ) printf("sleep\n");
#endif
	    sleep(tmpsleep);
	}

    } while ( attempts < maxtmpattempts && loop ) ;

 /*
  *  If not all tmp files has been read then return
  */
    if ( loop ) {
#if defined( TRACE )
	if ( TRACE_read_tmps ) printf("timeout\n");
#endif
	return(2);						     /* timeout	*/
    }

#if defined( TRACE )
    if ( TRACE_read_tmps ) dumptmp(nofs,know,0);
#endif

 /*
  *  Select the first common file system having a tmp file
  */
    for(idx=0;idx<nofs;idx++)
	if ( BIT_ISSET(know,idx) ) {
	    *scidx = idx;
	    *fsidx = poolfs[*scidx].fsidx;
#if defined( TRACE )
	    if ( TRACE_read_tmps ) 
		printf("selected tmp is %s [%d:%d]\n",scan1[*scidx].p_tmp,*scidx,*fsidx);
#endif
	    return(1);					/* file system selected	*/
	}

 /*
  *  Internal error
  */
    vperror(0,"Internal: empty knowledge intersection");
    return(EXINT);						       /* error	*/
}

#if defined( TRACE )

/*
 *  Print the bit mask of the tmp file configuration
 */
int dumptmp(nofs,data,vers)
int nofs;
char *data;
boolean vers;
{
  int idx;
    if ( vers ) {
	printf("version: %s - ",ctimeln((time_t*)data));
	data += sizeof(LONG);
    }
    printf("bit mask: ");
    for (idx=0; idx<nofs; idx++) printf("%s", BIT_ISSET(data,idx) ? "1" : "0");
    printf("\n");
}

#endif

#endif
