/*
 * Copyright (C) 1992 by CERN/CN/SW/DC
 * All rights reserved
 */

#ifndef lint
static char sccsid[] = " @(#)command.c	3.7 05/07/93  CERN-SW/DC Fabrizio Cane";
#endif /* not lint */

#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <sys/time.h>
#include <sys/types.h>
#include <shift_types.h>
#include <strdup.h>
#include <osdep.h>
#include <parser.h>
#include <table.h>
#include <config.h>
#include <data.h>
#include <command.h>

extern int code_version;
extern time_t config_version;
extern char *shifthosts;

/*
 *  Print the configuration contained in the binary file
 *
 *  INPUT
 *	mode		bit mask that specifies the configuration to print
 *
 *  RETURN
 *	0		successful completion
 *	1		an error occured
 */
int read_binary(mode)
int mode;
{
    clear_vperror();

 /*
  *  Load the configuration contained in the binary file
  */
    if ( read_binary_file((boolean)(mode&3),(boolean)(mode&4)) == ERROR ) return(1);

 /*
  *  Print the configuration in memory
  */
    print_configuration(mode);
    printf("Configuration binary file correct\n");

    return(0);
}

/*
 *  Import the binary file unless the local configuration is the most recent
 *
 *  INPUT
 *	hosts		list of hosts to contact
 *
 *  RETURN
 *	0		successful completion
 *	1		an error occured
 */
int import_binary(imphosts)
char *imphosts;
{
    clear_vperror();

 /*
  *  Open and lock the binary file
  */
    errno = 0;
    if ( open_binary(ACCESS_READ,LOCK_EXREAD) < 0 && errno != ENOENT ) return(1);

 /*
  *  Load the version and the shift environment host list
  */
    read_version();
    read_hosts();

 /*
  *  Unlock and close the configuration binary file - leave the exclusive access
  *
  *  REMARKS: The shift lock file is unlocked when the process will terminate
  */
    close_binary(FALSE,LOCK_LEAVE);

 /*
  *  If the shift environment host table cannot be loaded then use the given
  *  import host list and set the local version to zero ( the oldest )
  */
    if ( clear_vperror() < 0 ) {
	printf("No local configuration\n");
	config_version = 0;
    }
    else {
	getshifthosts(TRUE);
	imphosts = shifthosts;
	print_versions();
	print_environment();
    }

 /*
  *  Import the configuration binary file
  */
    return(import_binary_file(imphosts));
}

/*
 *  Update the shift environment host list and export the binary file
 *
 *  INPUT
 *	hosts		list of hosts to add/remove to the shift environment
 *
 *  RETURN
 *	0		successful completion
 *	1		an error occured
 */
int export_binary(exphosts,touch)
char *exphosts;
boolean touch;
{
    clear_vperror();

 /*
  *  Open and lock the binary file
  */
    open_binary(ACCESS_UPDATE,LOCK_WRITE);

 /*
  *  Load the shift environment host list
  */
    read_hosts();

#if !defined( NOAUTHORIZE )

 /*
  *  Check the authorization of the local host to export the binary file
  */
    if ( is_host_authorized(EXPORTAUTHORIZE,(char*)NULL) != 1) return(1);

#endif

 /*
  *  Merge the export host list and the shift environment host list
  *  and update the binary file if the shift environment is modified
  */
    switch ( merge_hosts(exphosts) ) {

	case TRUE :
		printf("Overwriting configuration binary file\n");
		read_tables();
		read_pairs();

	     /*
	      *  Update the configuration and code version
	      */
		code_version = SHIFT_VERSION;
		config_version = time((time_t*)NULL);
		write_version();

	    /*
	     *  Update the host list
	     */
		write_hosts();

	    /*
	     *  Rewrite the tables
	     */
		write_tables();
		write_pairs();

		print_versions();
		print_environment();
		break;

	case FALSE :
		if ( !touch )
		    read_version();
		else {
		    code_version = SHIFT_VERSION;
		    config_version = time((time_t*)NULL);
		    write_version();
		}
		read_tables();
		read_pairs();

		print_versions();
		print_environment();

	case ERROR :
		break;

	default : vperror(0,"Internal: unexpected merge_hosts() reply");
		  break;

    } /* switch - merge_hosts() reply */

 /*
  *  Unlock and close the configuration binary file - leave the exclusive access
  *
  *  REMARKS: The shift lock file is unlocked when the process will terminate
  */
    close_binary(TRUE,LOCK_LEAVE);

 /*
  *  Propagate the configuration to the whole shift environment
  */
    return(export_binary_file(shifthosts));
}

/*
 *  Produce the configuration binary file from the configuration ascii files
 *
 *  INPUT
 *	output		output stream used by the parser
 *			(if NULL no output is produced by the parser)
 *
 *  RETURN
 *	0		successful completion
 *	1		an error occured
 */
int produce_binary(output,create)
FILE *output;
boolean create;
{
  extern index_t_s hostidx;
  extern host_t *host_table;
  boolean new_binary;

    clear_vperror();

#if !defined( NOAUTHORIZE )

 /*
  *  Check the authorization of the local host to produce the binary file
  */
    if ( is_host_authorized(BINARYAUTHORIZE,(char*)NULL) != 1 ) return(1);

#endif

 /*
  *  First check if the configuration ascii files are correct
  */
    if ( shift_parser(output) ) return(1);

 /*
  *  Open/Create and lock the binary file
  */
    if ( (new_binary=open_binary(create?ACCESS_NEW:ACCESS_WRITE,LOCK_WRITE)) == ERROR )
	return(1);

    if ( new_binary )
	printf("Creating new configuration binary file\n");
    else
	printf("Overwriting configuration binary file\n");

 /*
  *  Update the configuration and code version
  */
    code_version = SHIFT_VERSION;
    config_version = time((time_t*)NULL);
    write_version();
    print_versions();

 /*
  *  If the file was created then the new shift environment must contain the
  *  local host only otherwise load the list from the existing binary file
  */
    if ( !new_binary ) 
	read_hosts();
    else {
	hostidx = 1;
	if ( alloc_host(&host_table,hostidx) < 0 ) exception(e1);
	gethostname(host_table[0],sizeof(host_t));
     /*
      *  Set the authorization of the local host
      */
	host_table[0][sizeof(host_t)-1] = '\0';
	getshifthosts(FALSE);
	write_hosts();
    }
    print_environment();

 /*
  *  Write the Disk Pool Manager configuration
  */
    write_tables();
    write_pairs();

 /*
  *  Unlock and close the configuration binary - leave the exclusive access
  *
  *  REMARKS: The shift lock file is unlocked when the process will terminate
  */
    close_binary(TRUE,LOCK_LEAVE);

 /*
  *  Propagate the configuration to the whole shift environment
  *  unless the binary file has been created just now
  */
    if ( !new_binary )
	return(export_binary_file(shifthosts));

    return( get_vperror() == ERROR ? 1 : 0 );

handle_exception(e1) :   close_binary(FALSE,UNLOCK);
			 return(1);
}

/*
 *  Produce the configuration ascii files from the configuration binary file
 *
 *  INPUT
 *	output		output stream used by the parser
 *			(if NULL no output is produced by the parser) 
 *
 *  RETURN
 *	0		successful completion
 *	1		an error occured
 */
int produce_ascii(output)
FILE *output;
{
  extern path_t shift_defaults,shift_pools,shift_users,shift_pairs;
  FILE *fd_defaults,*fd_pools,*fd_users,*fd_pairs;

    clear_vperror();

#if !defined( NOAUTHORIZE )

 /*
  *  Check the authorization of the local host to produce the ascii files
  */
    if ( is_host_authorized(ASCIIAUTHORIZE,(char*)NULL) != 1 ) return(1);

#endif

 /*
  *  Load the configuration contained in the binary file
  */
    if ( read_binary_file(TRUE,TRUE) == ERROR ) return(1);

 /*
  *  Open the configuration ascii files for writing
  */
    if ( (fd_defaults=fopen(shift_defaults,"w")) == NULL ) {
	vperror(1,"fopen(%s,w)",shift_defaults);
	return(1);
    }
    if ( (fd_pools=fopen(shift_pools,"w")) == NULL ) {
	vperror(1,"fopen(%s,w)",shift_pools);
	return(1);
    }
    if ( (fd_users=fopen(shift_users,"w")) == NULL ) {
	vperror(1,"fopen(%s,w)",shift_users);
	return(1);
    }
    if ( (fd_pairs=fopen(shift_pairs,"w")) == NULL ) {
	vperror(1,"fopen(%s,w)",shift_pairs);
	return(1);
    }

 /*
  *  Write the loaded configuration to the ascii files
  */
    print_tables(fd_defaults,fd_pools,fd_users,fd_pairs);

 /*
  *  Close the configuration ascii files
  */
    if ( fclose(fd_defaults) fclose_ERROR ) {
	vperror(1,"fclose(%s)",shift_defaults);
	return(1);
    }
    if ( fclose(fd_pools) fclose_ERROR ) {
	vperror(1,"fclose(%s)",shift_pools);
	return(1);
    }
    if ( fclose(fd_users) fclose_ERROR ) {
	vperror(1,"fclose(%s)",shift_users);
	return(1);
    }
    if ( fclose(fd_pairs) fclose_ERROR ) {
	vperror(1,"fclose(%s)",shift_pairs);
	return(1);
    }

 /*
  *  Print the version number and the environment host list
  */
    print_versions();
    print_environment();

 /*
  *  Check the produced ascii files after having deleted the configuration in memory
  */
    free_configuration();
    if ( shift_parser(output) ) {
	fprintf(stderr,"Produced configuration ascii files NOT correct\n");
	return(1);
    }

    return(0);
}

/*
 *  Write lock/unlock a set of file systems
 *
 *  INPUT
 *	lock		lock mode ( 0 - unlock / 1 - lock )
 *	str		file systems to lock/unlock
 *
 *  RETURN
 *	0		successful completion
 *	1		an error occured
 */
int lock_filesystem(lock,str)
int lock;
char *str;
{
  extern index_t_s fsysidx;
  extern fsys_t *fsys_table;
  extern disk_t *disk_table;
  extern pool_t *pool_table;
  boolean write,empty;
  index_t_s disk,fsys;
  char *t,*p;
  path_t path;

    clear_vperror();

 /*
  *  Open (mode r+) and lock (exclusive shared lock) the binary file
  */
    if ( str == NULL ) {
	if ( open_binary(ACCESS_READ,LOCK_READ) < 0 ) return(1);
    }
    else {
	if ( open_binary(ACCESS_UPDATE,LOCK_WRITE) < 0 ) return(1);
	lock_binary(LOCK_EXREAD);
    }

#if !defined( NOAUTHORIZE )

 /*
  *  Check the authorization of the local host to export the binary file
  */
    if ( str != NULL ) {
	read_hosts();
	if ( is_host_authorized(EXPORTAUTHORIZE,(char*)NULL) != 1) return(1);
    }

#endif

 /*
  *  Load the shift tables
  */
    read_tables();

    if ( get_vperror() < 0 ) return(1);
    write = FALSE;

 /*
  *  Print the list of file systems write locked 
  */
    if ( str == NULL ) {
	empty = TRUE;
	for (fsys=0; fsys<fsysidx; fsys++)
	    if ( fsys_table[fsys].bits & FSYSWLOCK ) {
		if ( empty ) empty = FALSE;
		printf("  file system <%s:%s> %*s pool <%s>\n",
			disk_table[fsys_table[fsys].diskidx].host,
			fsys_table[fsys].name,
			20-strlen(disk_table[fsys_table[fsys].diskidx].host)-
			strlen(fsys_table[fsys].name),"",
			pool_table[fsys_table[fsys].poolidx].name);
	    }
	if ( empty ) printf("no file system write locked\n");
    }

 /*
  *  Scan the specified file system list whose format is:
  *
  *	filesystemlist ::= filesystem { , filesystem }
  *	filesystem     ::= server_name : file_system_name | / complete-nfs-pathname
  */
    else if ( strcmp(str,"all") || lock ) {
	t = strtok(str,",");
	while ( t != NULL ) {
	    if ( (p = (char*)strchr(t,':')) != NULL ) {
		*p = EOS;
		if ( (disk = look_disk(t)) < 0 ) {
		    vperror(0,"unknown server <%s>",t);
		    *p = ':';
	    	    t = strtok(NULL,",");
		    continue;
		}
		*p = ':';
		if ( (fsys = look_fsys(disk,p+1)) < 0 ) {
		    vperror(0,"unknown file system <%s>",t);
	    	    t = strtok(NULL,",");
		    continue;
		}
	    }
	    else {
		for (fsys=0; fsys<fsysidx; fsys++) {
		    create_mountpath(fsys,path);
		    if ( !strcmp(t,path) ) break;
		}
		if ( fsys == fsysidx ) {
		    vperror(0,"unknown file system <%s>",t);
	    	    t = strtok(NULL,",");
		    continue;
		}
	    }

	 /*
	  *  Check the file system be already locked/unlocked
	  */
	    if ( ( (fsys_table[fsys].bits & FSYSWLOCK) > 0 )  == ( lock > 0 ) ) {
		vperror(0,"  file system <%s> %*s ALREADY %slocked",
			   t,21-strlen(t),"",lock?"":"un");
	    	t = strtok(NULL,",");
		continue;
	    }

	 /*
	  *  Lock/Unlock the file system
	  */
	    if ( lock )
		fsys_table[fsys].bits = fsys_table[fsys].bits | FSYSWLOCK;
	    else
		fsys_table[fsys].bits = fsys_table[fsys].bits & ~FSYSWLOCK;
	    printf("  file system <%s> %*s %sLOCKED\n",
		    t,21-strlen(t),"",lock?"":"UN");

	 /*
	  *  Continue to scan the list of file system
	  */
	    if ( !write ) write = TRUE;
	    t = strtok(NULL,",");
	}
    }

 /*
  *  Unlock all file systems write locked
  */
    else {
	for (fsys=0; fsys<fsysidx; fsys++)
	    if ( fsys_table[fsys].bits && FSYSWLOCK ) {
		write = TRUE;
		fsys_table[fsys].bits = fsys_table[fsys].bits & ~FSYSWLOCK;
		printf("  file system <%s:%s> %*s UNLOCKED\n",
			disk_table[fsys_table[fsys].diskidx].host,
			fsys_table[fsys].name,
			20-strlen(disk_table[fsys_table[fsys].diskidx].host)-
			strlen(fsys_table[fsys].name),"");
	    }
	if ( !write ) printf("no file system write locked\n");
    }

    clear_vperror();

 /*
  *  Update and propagate the binary configuration on disk if required
  */
    if ( write ) {
        read_version();
	read_pairs();
	lock_binary(LOCK_WRITE);
	config_version = time((time_t*)NULL);
	write_version();
	write_tables();
	close_binary(FALSE,LOCK_LEAVE);
	return(export_binary_file(shifthosts));
    }

    close_binary(FALSE,UNLOCK);
    return(0);
}
