/*
 * Copyright (C) 1991,1992 by CERN/CN/SW/DC
 * All rights reserved
 */

#ifndef lint
static char sccsid[] = " @(#)config.c	3.10 03/29/94  CERN-SW/DC Fabrizio Cane";
#endif /* not lint */

#include <stdio.h>
#include <sys/types.h>
#include <sys/file.h>
#include <sys/time.h>
#include <netinet/in.h>
#ifndef apollo
#include <malloc.h>
#endif
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <shift_types.h>
#include <osdep.h>
#include <strtok.h>
#include <strerror.h>
#include <strutil.h>
#include <flock.h>
#include <parser.h>
#include <table.h>
#include <data.h>
#include <config.h>

extern index_t_s diskidx,poolidx,fsysidx,accsidx,hostidx;
extern disk_t *disk_table;
extern pool_t *pool_table;
extern fsys_t *fsys_table;
extern accs_t *accs_table;
extern host_t *host_table;

path_t shift_defaults,shift_pools,shift_users,shift_pairs;
path_t shift_binary,shift_binary_net,shift_binary_copy,shift_binary_lock;
path_t shift_log;
time_t config_version;
int code_version;
char *shifthosts	= NULL ;
FILE *fd_binary;
FILE *fd_binary_net;
boolean locktrace = 0;

static FILE *fd_binary_lock;
static boolean binary_corrupted;
static boolean binary_copy;

/*
 *  Make a copy of the configuration binary file
 *
 *  RETURN
 *	1		successful completion
 *	ERROR		an error occured
 */
int make_binary_copy()
{
  FILE *fd_copy,*fd;
  char buffer[BUFSIZ];
  long size;
  int length;

    binary_copy = FALSE;

    if ( (fd_copy=fopen(shift_binary_copy,"w")) == NULL ) {
	vperror(1,"fopen(%s,w)",shift_binary_copy);
	exception(e0);
    }
    if ( (fd=fopen(shift_binary,"r")) == NULL ) {
	vperror(1,"fopen(%s,r)",shift_binary);
	exception(e2);
    }
    if ( (size=filesize(shift_binary)) < 0 ) {
	vperror(1,"stat(%s)",shift_binary);
	exception(e3);
    }

    do {
	length = size < BUFSIZ ? size : BUFSIZ ;
	if ( fread(buffer,length,1,fd) fread_ERROR ) {
	    vperror(1,"fread(,%d,1,%s[%d])",length,shift_binary,fdtono(fd));
	    exception(e3);
	}
	if ( fwrite(buffer,length,1,fd_copy) fwrite_ERROR ) {
	    vperror(1,"fwrite(,%d,1,%s[%d])",length,shift_binary_copy,fdtono(fd_copy));
	    exception(e3);
	}
    } while ( (size -= BUFSIZ) > 0 );

    if ( fclose(fd) fclose_ERROR ) {
	vperror(1,"fclose(%s[%d])",shift_binary,fdtono(fd));
	exception(e2);
    }

    if ( fclose(fd_copy) fclose_ERROR ) {
	vperror(1,"fclose(%s[%d])",shift_binary_copy,fdtono(fd_copy));
	exception(e1);
    }

    binary_copy = TRUE;

    return(1);

handle_exception(e3):	fclose(fd);
handle_exception(e2):	fclose(fd_copy);
handle_exception(e1):	unlink(shift_binary_copy);
handle_exception(e0):	return(ERROR);
}

/*
 *  Recover the configuration binary file using its copy if available
 *
 *  RETURN
 *	1		successful completion
 *	ERROR		an error occured
 */
int recover_binary_copy()
{
    if ( !binary_copy || rename(shift_binary_copy,shift_binary) < 0 ) {
	vperror(1,"rename(%s,%s)",shift_binary_copy,shift_binary);
	return(ERROR);
    }

    binary_copy = FALSE;

    return(1);
}

/*
 *  Check if the binary file is corrupted
 *
 *  RETURN
 *	TRUE		binary file corrupted
 *	FALSE		binary file NOT corrupted
 *	ERROR		an error occured
 */
boolean is_binary_corrupted()
{
  int smallest;
  int size;
  index_t_s index;
  int length;

    binary_corrupted = TRUE;

    if ( (size = filesize(shift_binary)) < 0 ) {
	vperror(1,"stat(%s)",shift_binary);
	return(ERROR);
    }

 /*
  *  The binary file contains at least:
  *
  *	code version 		sizeof(int)
  *	config version		sizeof(time_t)
  *	table length		one sizeof(indext_t) for each table
  *	table data		one entry for each table (no account / no supool entry )
  *	pair table length	sizeof(int)
  *	pair table index	sizeof(index_t_s)
  */

    smallest = sizeof(int)+sizeof(time_t)+6*sizeof(index_t_s);
    smallest += sizeof(host_t)+sizeof(disk_t)+sizeof(pool_t)+sizeof(fsys_t);
    smallest += sizeof(int)+sizeof(index_t_s);

    if ( size < smallest )
	return(binary_corrupted);

    if ( fseek(fd_binary,HOSTS_OFFSET,0) fseek_ERROR )
	exception(eseek);
    if ( fread(&index,sizeof(index_t_s),1,fd_binary) fread_ERROR )
	exception(eread);
    if ( size < (smallest += (index-1)*sizeof(host_t)) )
	return(binary_corrupted);

    if ( fseek(fd_binary,index*sizeof(host_t),1) fseek_ERROR )
	exception(eseek);
    if ( fread(&index,sizeof(index_t_s),1,fd_binary) fread_ERROR )
	exception(eread);
    if ( size < (smallest += (index-1)*sizeof(disk_t)) )
	return(binary_corrupted);

    if ( fseek(fd_binary,index*sizeof(disk_t),1) fseek_ERROR )
	exception(eseek);
    if ( fread(&index,sizeof(index_t_s),1,fd_binary) fread_ERROR )
	exception(eread);
    if ( size < (smallest += (index-1)*sizeof(pool_t)) )
	return(binary_corrupted);

    if ( fseek(fd_binary,index*sizeof(pool_t),1) fseek_ERROR )
	exception(eseek);
    if ( fread(&index,sizeof(index_t_s),1,fd_binary) fread_ERROR )
	exception(eread);
    if ( size < (smallest += index*sizeof(supool_t)) )
	return(binary_corrupted);

    if ( fseek(fd_binary,index*sizeof(supool_t),1) fseek_ERROR )
	exception(eseek);
    if ( fread(&index,sizeof(index_t_s),1,fd_binary) fread_ERROR )
	exception(eread);
    if ( size < (smallest += (index-1)*sizeof(fsys_t)) )
	return(binary_corrupted);

    if ( fseek(fd_binary,index*sizeof(fsys_t),1) fseek_ERROR )
	exception(eseek);
    if ( fread(&index,sizeof(index_t_s),1,fd_binary) fread_ERROR )
	exception(eread);
    if ( size < (smallest += index*sizeof(accs_t)) )
	return(binary_corrupted);

    if ( fseek(fd_binary,index*sizeof(accs_t),1) fseek_ERROR )
	exception(eseek);
    if ( fread(&length,sizeof(int),1,fd_binary) fread_ERROR )
	exception(eread);
    if ( size != (smallest += length - sizeof(int) - sizeof(index_t_s)) )
	return(binary_corrupted);

    binary_corrupted = FALSE;

    return(binary_corrupted);

handle_exception(eseek):
	vperror(1,"fseek(%s[%d],,1)",shift_binary,fdtono(fd_binary));
	return(ERROR);

handle_exception(eread):
	vperror(1,"fread(,,,%s[%d])",shift_binary,fdtono(fd_binary));
	return(ERROR);
}

/*
 *  Call flock()
 *
 *  INPUT
 *	name		file name
 *	fd		file descriptor
 *	lock		lock mode - see flock()
 *
 *  RETURN
 *	1		successful completion
 *	ERROR		an error occured
 */
int lock_file(name,fd,lock)
char *name;
int fd;
int lock;
{
    if ( locktrace ) {
	printf("lock");
	fflush(stdout);
    }
    if ( flock(fd,lock) < 0 ) {
	vperror(1,"flock(%s[%d],%d)",name,fd,lock);
	return(ERROR);
    }
    if ( locktrace ) {
	printf("ed\10\10\10\10\10\10");
	fflush(stdout);
    }
    return(1);
}

/*
 *  Lock/Unlock the binary file
 *
 *  INPUT
 *	lock		lock mode
 *			  LOCK_READ / LOCK_WRITE / LOCK_EXREAD / LOCK_LEAVE / UNLOCK
 *
 *  RETURN
 *	1		successful completion
 *	ERROR		an error occured
 *
 *  DESCRIPTION
 *
 *	lock mode	   lock on the "lock file"	  lock on the "binary file"
 *
 *	LOCK_READ		    ---				SHARED lock
 *	LOCK_EXREAD		EXCLUSIVE lock			SHARED lock
 *	LOCK_WRITE		EXCLUSIVE lock			EXCLUSIVE lock
 *	LOCK_LEAVE		    ---				    ---	
 *	UNLOCK			   unlock			  unlock
 */
int lock_binary(lock)
int lock;
{
    if ( lock == LOCK_LEAVE )
	return(1);

    if ( lock == UNLOCK ) {
	if ( fdtono(fd_binary_lock) > -1 )
	    fclose(fd_binary_lock);
	if ( fdtono(fd_binary) > -1 )
	    fclose(fd_binary);
	return(1);
    }

    if ( lock != LOCK_READ && fdtono(fd_binary_lock) < 0 ) {
	if ( (fd_binary_lock=fopen(shift_binary_lock,"w")) == NULL ) {
	    vperror(1,"fopen(%s,w)",shift_binary_lock);
	    return(ERROR);
	}
	if ( lock_file(shift_binary_lock,fdtono(fd_binary_lock),LOCK_EX) < 0 ) {
	    fclose(fd_binary_lock);
	    return(ERROR);
	}
    }

    if ( fdtono(fd_binary) > -1 ) {
	lock = (lock == LOCK_WRITE) ? LOCK_EX : LOCK_SH ;
	if ( lock_file(shift_binary,fdtono(fd_binary),lock) < 0 ) {
	    fclose(fd_binary_lock);
	    fclose(fd_binary);
	    return(ERROR);
 	}
    }

    return(1);
}

/*
 *  Open/Create and lock the configuration binary file
 *
 *  INPUT
 *	access	    access mode - ACCESS_READ / ACCESS_UPDATE / ACCESS_WRITE / ACCESS_NEW
 *	lock	    lock mode - see lock_binary()
 *
 *  RETURN
 *	TRUE		successful completion ( binary file created )
 *	FALSE		successful completion ( binary file NOT created )
 *	ERROR		error
 */
int open_binary(access,lock)
int access;
int lock;
{
  boolean created;
  int cversion;
  time_t tversion;
  char mode_fopen[3];
  int mode_lock;

 /*
  *  The binary copy is not available
  */
    binary_copy = FALSE;

 /*
  *  Set the correct fopen and flock mode
  */
    strcpy(mode_fopen,(access == ACCESS_READ) ? "r" : "r+");
    mode_lock = (access == ACCESS_READ) ? lock : LOCK_WRITE ;

 /*
  *  Open/Create the configuration binary file
  */
    if ( (created = ((fd_binary = fopen(shift_binary,mode_fopen)) == NULL))  ) {
	if ( errno != ENOENT || (access != ACCESS_WRITE && access != ACCESS_NEW) ) {
	    vperror(1,"fopen(%s,%s)",shift_binary,mode_fopen);
	    return(ERROR);
	}
	if ( !(created = ((fd_binary = fopen(shift_binary,"w+")) != NULL)) ) {
	    vperror(1,"fopen(%s,w+)",shift_binary);
	    return(ERROR);
	}
    }
#if defined (TRACE )
	printf(" Opening file >%s< with mode >%s< \n",shift_binary,mode_fopen) ;
	printf( " File is created ? %d\n",created);
#endif

 /*
  *  Lock the configuration binary file 
  */
    if ( lock_binary(mode_lock) < 0 )
	return(ERROR);

 /*
  *  Truncate the file if access is ACCESS_NEW
  */
#if defined (TRACE )
        printf( " BEFORE : File is created ? %d\n",created);
        printf( " BEFORE : Access is %d\n",access) ;
        printf( " BEFORE : ACCESS_NEW is %d\n", ACCESS_NEW);
#endif

    if ( access == ACCESS_NEW ) {
	created=TRUE ;
	if (ftruncate(fdtono(fd_binary),0) < 0 ) {
		vperror(1,"ftruncate(%d[%s],%d)",fd_binary,shift_binary,0);
		return(ERROR);
    	}
    }
#if defined (TRACE )
        printf( " AFTER : File is created ? %d\n",created);
        printf( " AFTER : Access is %d\n",access) ;
        printf( " AFTER : ACCESS_NEW is %d\n", ACCESS_NEW);
#endif

 /*
  *  Check the configuration binary file be not corrupted unless it's been created
  */
    if ( !created && is_binary_corrupted() ) {
	vperror(0,"binary file corrupted (%s)",shift_binary);
	if ( access == ACCESS_WRITE ) {
	    clear_vperror();
	    if (!(created=((fd_binary=freopen(shift_binary,"w+",fd_binary))!=NULL))) {
		vperror(1,"fopen(%s,w+)",shift_binary);
		return(ERROR);
	    }
	    if ( lock_binary(mode_lock) < 0 )
		return(ERROR);
	}
	else
	    return(ERROR);
    }

 /*
  *  Check the code version of the binary file be correct unless it's been created
  */
    if ( !created ) {
	cversion = code_version;
	tversion = config_version;
	read_version();
	if ( get_vperror() > 0 &&  SHIFT_VERSION != code_version )
	    vperror(0,"incompatible binary file code version %x",code_version);
	code_version = cversion;
	config_version = tversion;
	if ( get_vperror() < 0 )
	    return(ERROR);
    }

 /*
  *  Make a copy of the binary file if access is not READ and it's not been created
  */
    if ( access != ACCESS_READ && !created )
	if ( make_binary_copy() < 0 )
		return(ERROR);

    return(created);
}

/*
 *  Truncate unlock and close the configuration binary file
 *
 *  INPUT
 *	truncate	to truncate the binary file at the current offset
 *	lock		lock mode - LOCK_LEAVE / UNLOCK - see lock_binary()
 *
 *  RETURN
 *	1		successful completion
 *	ERROR		error
 */
int close_binary(truncate,lock)
boolean truncate;
int lock;
{
 /*
  *  Return if the file descriptor is invalid
  */
    if ( fdtono(fd_binary) < 0 ) {
	if ( get_vperror() > 0 )
	    vperror(0,"close_binary(): invalid binary file descriptor");
	exception(elock);
    }

 /*
  *  If an error occured then try to recover the binary file
  */
    if ( get_vperror() < 0 && binary_copy )
	recover_binary_copy();

 /*
  *  Write the buffers otherwise the ftruncate fails on some systems
  */
    if ( truncate && get_vperror() > 0 ) {
	if ( fflush(fd_binary) fflush_ERROR ) {
		vperror(1,"fflush(%d[%s])",fdtono(fd_binary),shift_binary);
		exception(elock);
	}
    }

 /*
  *  Truncate the file to the current offset
  */
    if ( truncate && get_vperror()>0 && ftruncate(fdtono(fd_binary),ftell(fd_binary))<0 ) {
	vperror(1,"ftruncate(%d[%s],%d)",fdtono(fd_binary),shift_binary,ftell(fd_binary));
	exception(elock);
    }

 /*
  *  Remove the lock and close the configuration binary file
  */
    if ( fclose(fd_binary) fclose_ERROR ) {
	vperror(1,"fclose(%s[%d])",shift_binary,fd_binary);
	exception(elock);
    }

 /*
  *  Leave/Remove the current lock
  */
    if ( lock != LOCK_LEAVE && lock != UNLOCK ) {
	vperror(0,"close_binary(): invalid lock mode (%d)",lock);
	exception(elock);
    }
    if ( lock_binary(lock) < 0 )
	return(ERROR);

    return(get_vperror());

handle_exception(elock):  lock_binary(UNLOCK);
			  return(ERROR);
}

/*
 *  Remove the ascii files
 */
int remove_ascii_files()
{
    if ( unlink(shift_defaults) < 0 && errno != ENOENT ) {
	vperror(1,"unlink(%s)",shift_defaults);
	return(ERROR);
    }

    if ( unlink(shift_pools) < 0 && errno != ENOENT ) {
	vperror(1,"unlink(%s)",shift_pools);
	return(ERROR);
    }

    if ( unlink(shift_users) < 0 && errno != ENOENT ) {
	vperror(1,"unlink(%s)",shift_users);
	return(ERROR);
    }

    if ( unlink(shift_pairs) < 0 && errno != ENOENT ) {
	vperror(1,"unlink(%s)",shift_pairs);
	return(ERROR);
    }

    return(1);
}

/*
 *  Build the shift host list
 *
 *  RETURN
 *	-1		an error occured
 *	0		empty shift host list
 *	1		successful completion
 */
int getshifthosts(authorized)
boolean authorized;
{
  static int shifthostslength = 0;	/* memory currently allocated for shifthosts */
  boolean first;
  index_t_s idx;
  int length;

 /*
  *  Compute the length of the new shift host list
  */
#if defined(TRACE)
    printf("Hosts index contains %d hosts (hostidx)\n",hostidx);
#endif
    length = 0;
    for (idx=0; idx<hostidx; idx++)
#if !defined( NOAUTHORIZE )
	if ( !authorized || !(int)host_table[idx][sizeof(host_t)-1] )
#endif
	    length += strlen(host_table[idx]) + 1;
    if ( !length ) return(0);

 /*
  *  Allocate the required memory
  */

    if ( length > shifthostslength ) {
#if defined (TRACE)
    printf("Allocating %d bytes to store hosts list\n",length) ;
#endif
	if ( (shifthosts = remalloc(shifthosts,length)) == NULL ) {
	    vperror(1,"remalloc(%x,%d)",shifthosts,length);
	    return(-1);
	}
	shifthostslength = length;
    }

 /*
  *  Build the new shift host list
  */
    first = TRUE;
    for (idx=0; idx<hostidx; idx++)
#if !defined( NOAUTHORIZE )
	if ( !authorized || !(int)host_table[idx][sizeof(host_t)-1] )
#endif
	    if ( first ) {
		strcpy(shifthosts,host_table[idx]);
		first = FALSE;
	    }
	    else {
		strcat(shifthosts,HOSTSEP);
		strcat(shifthosts,host_table[idx]);
#if defined (TRACE)
		 printf(" shifthosts list is %s\n",shifthosts) ;
#endif
	    }

    return(1);
}

/*
 *  Extend/Reduce the shift environment host table as specified in the export command
 *
 *  INPUT
 *	exphosts	export host list as specified in the export command
 *
 *  RETURN
 *	TRUE		the shift environment host table has been modified
 *	FALSE		the shift environment host table is not changed
 *	ERROR		error
 */
boolean merge_hosts(exphosts)
char *exphosts;
{
  static host_t *expary = NULL;
  static int exparylen = 0;
  char *host;
  int explength,expidx,envidx,idx,offs;
  int expanded,reduced;

 /*
  *  Return if an error occured
  */
    if ( get_vperror() == ERROR )
	return(ERROR);

 /*
  *  Compute the export host table length
  */
    explength = 0;
    host = strtok(exphosts,HOSTSEPSET);
    while ( host != NULL ) {
	if ( strlen(host) >
#if defined( NOAUTHORIZE )
			    sizeof(host_t)-1 ) {
#else
			/*
			 *  The last character of host_t is used as authorization flag
			 */
			    sizeof(host_t)-2 ) {
#endif
	    vperror(0,"host name <%s> too long (max %d)",host,sizeof(host_t)-2);
	    return(ERROR);
	}
	explength++;
	host = strtok((char*)NULL,HOSTSEPSET);
    }
    if ( !explength ) return( FALSE );

 /*
  *  Allocate the required memory
  */
    if ( explength > exparylen ) {
	if ( (expary=(host_t*)remalloc((char*)expary,sizeof(host_t)*explength))==NULL ) {
	    vperror(1,"remalloc(%x,%d*%d)",expary,sizeof(host_t),explength);
	    return(ERROR);
	}
	exparylen = explength;
    }

 /*
  *  Build the export host table and remove duplications
  */
    expanded = 0;
    expidx = 0;
    host = strtok(exphosts,HOSTSEPSET);
    while ( host != NULL ) {
	for (idx=0; idx<expidx; idx++) 
	    if ( !strcmp(host,expary[idx]) ) {
		vperror(0,"too many <%s> in the export list",host);
		return(ERROR);
	    }
	if ( !( idx < expidx && expidx ) ) {
	    strcpy(expary[expidx++],host);
	    expanded += *host == '-' ? 0 : 1 ;
	}
	host = strtok((char*)NULL,HOSTSEPSET);
    }

 /*
  *  Scan the export and shift environment tables
  */
    reduced = 0;
    for (expidx=0; expidx<explength; expidx++) {
	offs = expary[expidx][0] == '-' || expary[expidx][0] == '+' ? 1 : 0 ;
	for (envidx=0; envidx<hostidx; envidx++) {
	    if ( expary[expidx][0] == EOS )
		break;
	    if ( host_table[envidx][0] == EOS )
		continue;
	    switch ( equal_hosts(host_table[envidx],expary[expidx]+offs,(char**)NULL) ) {
		case -1 : return(ERROR);
		case  1 : if ( expary[expidx][0] != '-' ) {
			      expary[expidx][0] = EOS;
			      expanded--;
			  }
			  else {
			      host_table[envidx][0] = EOS;
			      reduced++;
			  }
	    } /* switch - equal_hosts() reply */
	} /* for - shift environment table */
    } /* for - export table */

 /*
  *  Reduce the shift environment host table
  */
    if ( reduced )
	for (envidx=0; envidx<hostidx; envidx++)
	    if ( host_table[envidx][0] == EOS ) {
		while ( --hostidx > envidx )
		    if ( host_table[hostidx][0] != EOS )
			break;
		if ( hostidx > envidx )
		    memcpy(host_table[envidx],host_table[hostidx],sizeof(host_t));
	    }

 /*
  *  Extend the shift environment host table
  */
    if ( expanded ) {
	if ( alloc_host(&host_table,hostidx+expanded) < 0 )
	    return(ERROR);
	for (expidx=0; expidx<explength; expidx++) {
	    if ( expary[expidx][0] == EOS || expary[expidx][0] == '-' )
		continue;
	    offs = expary[expidx][0] == '+' ? 1 : 0 ;
	    strcpy(host_table[hostidx],&expary[expidx][offs]);
#if !defined( NOAUTHORIZE )
	    host_table[hostidx][sizeof(host_t)-1] = (char)offs;
#endif
	    hostidx++;
	}
    }

 /*
  *  Build the shift host list
  */
    if ( (expanded || reduced) && getshifthosts(FALSE) < 0 )
	return(ERROR);

    return( (expanded || reduced) ? TRUE : FALSE );
}

/*
 *  Load the configuration binary file
 *
 *  INPUT
 *	dpmconf		load the shift dpm configuration
 *	shiftconf	load the shift configuration ( old shift.conf file )
 *
 *  RETURN
 *	1		successful completion
 *	ERROR		error
 */
int read_binary_file(dpmconf,shiftconf)
boolean dpmconf,shiftconf;
{
 /*
  *  Open and lock the binary file
  */
    open_binary(ACCESS_READ,LOCK_READ);

 /*
  *  Load the version and the shift environment host list
  */
    read_version();
    read_hosts();

 /*
  *  Load the required configurations
  */
    if ( dpmconf ) read_tables();
    if ( shiftconf ) read_pairs();

 /*
  *  Unlock and close the configuration binary file
  */
    close_binary(FALSE,UNLOCK);

    return(get_vperror());
}

/*
 *  Write the configuration binary file to the disk
 *
 *  RETURN
 *	OK		successful completion
 *	ERROR		error
 */
int write_binary_file()
{
 /*
  *  Open/Create and lock the binary file
  */
    open_binary(ACCESS_WRITE,LOCK_WRITE);

 /*
  *  Write the version and the shift environment host list
  */
    write_version();
    write_hosts();

 /*
  *  Write the Disk Pool Manager tables
  */
    write_tables();

 /*
  *  Write the Pair table
  */
    write_pairs();

 /*
  *  Unlock and close the configuration binary file
  */
    close_binary(TRUE,UNLOCK);

    return(get_vperror());
}

/*
 *  Produce the shift configuration binary file in the network data representation
 */
int marshall_binary_file()
{
    if ( get_vperror() == ERROR )
	return(ERROR);

    if ( (fd_binary_net = fopen(shift_binary_net,"w")) == NULL ) {
	vperror(1,"fopen(%s,w)",shift_binary_net);
	return(ERROR);
    }

    marshall_version();

    marshall_hosts();

    marshall_tables();

    marshall_pairs();

    if ( fclose(fd_binary_net) fclose_ERROR ) {
	vperror(1,"fclose(%s[%d])",shift_binary_net,fd_binary_net);
	return(ERROR);
    }

    return(get_vperror());
}

/*
 *  Produce the shift configuration binary file from the network data representation
 */
int unmarshall_binary_file()
{
    if ( get_vperror() == ERROR )
	return(ERROR);

    if ( (fd_binary_net = fopen(shift_binary_net,"r")) == NULL ) {
	vperror(1,"fopen(%s,r)",shift_binary_net);
	return(ERROR);
    }

    unmarshall_version();

    unmarshall_hosts();

    unmarshall_tables();

    unmarshall_pairs();

    if ( fclose(fd_binary_net) fclose_ERROR ) {
	vperror(1,"fclose(%s[%d])",shift_binary_net,fd_binary_net);
	return(ERROR);
    }

    return(get_vperror());
}

/*
 *  Print the configuratio version (i.e. the date of the last produce_binary() call)
 */
void print_versions()
{
    printf("Local version: %s",ctime(&config_version));
}

#define MAXNOHPL	6
/*
 *  Print the shift environment host list
 */
void print_environment()
{
  index_t_s idx;
  int nohpl;
  boolean auth;
  int width;

    printf("Environment  : ");
    nohpl = 0;
    for (idx=0;idx<hostidx;idx++) {
	auth = (int)host_table[idx][sizeof(host_t)-1] > 0 ? 1 : 0 ;
	nohpl += (int)strlen(host_table[idx]) + auth <= 10 ? 1 : 2 ;
	width = nohpl == MAXNOHPL ? 0 - strlen(host_table[idx]) - auth : 
			( (int)strlen(host_table[idx]) + auth <= 10 ? -10 : -20 ) ;
	printf("%s%*s",auth ? "*" : " ",width,host_table[idx]);
	if ( nohpl == MAXNOHPL && idx < hostidx - 1 ) {
	    printf("\n%15s","");
	    nohpl = 0;
	}
    }
    if ( nohpl ) printf("\n");
}

/*
 *  Print the configuration contained in memory
 *
 *  INPUT
 *	mode	bit mask that specifies the configuration to print
 */
void print_configuration(mode)
int mode;
{
    print_versions();
    print_environment();
    if ( mode ) print_tables( mode&1?stdout:NULL,mode&1?stdout:NULL,
			      mode&2?stdout:NULL,
			      mode&4?stdout:NULL );
}

/*
 *   Remove the configuration stored in memory
 */
int free_configuration()
{
    free_disk();
    free_pool();
    free_supool();
    free_fsys();
    free_accs();
    free_host();
    free_pair();
}

/*
 *  Build the shift ascii paths, shift binary paths and shift log paths
 *
 *  INPUT
 *	ascii	     shift ascii directory path (if NULL no shift ascii path is build)
 *	binary	     shift binary directory path (if NULL no shift binary path is build)
 *	log	     shift log directory path (if NULL no shift log path is build)
 */
void build_shift_paths(ascii,binary,log)
char *ascii,*binary,*log;
{
    if ( ascii ) {

	strcpy(shift_defaults,ascii);
	if ( ascii[strlen(ascii)-1] != '/' ) strcat(shift_defaults,"/");
	strcpy(shift_pools,shift_defaults);
	strcpy(shift_users,shift_defaults);
	strcpy(shift_pairs,shift_defaults);

	strcat(shift_defaults,"shift.defaults");
	strcat(shift_pools,"shift.pools");
	strcat(shift_users,"shift.users");
	strcat(shift_pairs,"shift.pairs");
    }

    if ( binary ) {

	strcpy(shift_binary,binary);
	if ( binary[strlen(binary)-1] != '/' ) strcat(shift_binary,"/");

	strcat(shift_binary,"shiftdb");

	strcpy(shift_binary_net,shift_binary);
	strcpy(shift_binary_copy,shift_binary);
	strcpy(shift_binary_lock,shift_binary);

	strcat(shift_binary_net,".net");
	strcat(shift_binary_copy,".copy");
	strcat(shift_binary_lock,".lock");
    }

    if ( log ) {

	strcpy(shift_log,log);
	if ( log[strlen(log)-1] != '/' ) strcat(shift_log,"/");
	strcat(shift_log,"sfmaked.log");
    }

}

#if !defined( NOAUTHORIZE )

/*
 *  Check the authorization of the local host to performe an operation
 *
 *  INPUT
 *	code		operation code
 *	rhost		remote host ( only if the request is not local )
 *
 *  RETURN
 *	TRUE		authorized
 *	FALSE		not authorized
 *	ERROR		an error occured
 */
boolean is_host_authorized(code,rhost)
int code;
char *rhost;
{
  host_t localhost;
  boolean authorized;
  int idx;

 /*
  *  Get the local host name
  */
    gethostname(localhost,sizeof(host_t));

 /*
  *  Load the environment host table if required
  */
    switch ( code ) {
	case BINARYAUTHORIZE :
        case ASCIIAUTHORIZE :
	    if ( open_binary(ACCESS_READ,LOCK_READ) < 0 ) {
		close_binary(FALSE,UNLOCK);
		clear_vperror();
		return(TRUE);
	    }
	    if ( read_hosts() < 0 ) {
		close_binary(FALSE,UNLOCK);
		clear_vperror();
		return(TRUE);
	    }
	    if ( close_binary(FALSE,UNLOCK) < 0 ) clear_vperror();
	    break;
	case EXPORTAUTHORIZE :
	case REXPORTAUTHORIZE :
	case RIMPORTAUTHORIZE :
	    if ( get_vperror() < 0 ) return(TRUE);
	    break;
	default :
	    vperror(0,"internal: unknown operation code in is_host_authorized()");
	    return(ERROR);
    }

 /*
  *  Look for the local host in the environment table
  */
    for (idx=0; idx<hostidx; idx++)
	if ( equal_hosts(localhost,host_table[idx],(char**)NULL) == 1 )
	    break;
    if ( idx == hostidx ) {
	vperror(0,"local host <%s> not found in the current environment",localhost);
	return(ERROR);
    }
    clear_vperror();

 /*
  *  Get the authorization of the local host
  */
    authorized = !(int)host_table[idx][sizeof(host_t)-1];

 /*
  *  Check the authorization do execute the specified operation
  */
    switch ( code ) {
	case BINARYAUTHORIZE :
	case ASCIIAUTHORIZE :
	case EXPORTAUTHORIZE :
	 /*
	  *  It can't do it unless it's authorized
	  */
	    if ( !authorized ) vperror(0,"local host not authorized");
	    break;
	case RIMPORTAUTHORIZE :
	 /*
	  *  It can't export unless it's authorized
	  */
	    if ( !authorized ) {
		vperror(0,"host <%s> not authorized to export",localhost);
		break;
	    }
	    if ( rhost == NULL ) break;

	 /*
	  *  Look for the remote host in the environment table
	  */
	    for (idx=0; idx<hostidx; idx++)
		if ( equal_hosts(rhost,host_table[idx],(char**)NULL) == 1 ) break;
	 /*
	  *  It can't export the configuration to a host that is not in the 
	  *  current environment unless the request is local
	  */
	    clear_vperror();
	    authorized = idx < hostidx;
	    if ( !authorized )
		vperror(0,"host <%s> not in the current environment",rhost);
	    break;
	case REXPORTAUTHORIZE :
	 /*
	  *  Look for the remote host in the environment table
	  */
	    for (idx=0; idx<hostidx; idx++)
		if ( equal_hosts(rhost,host_table[idx],(char**)NULL) == 1 ) break;
	 /*
	  *  It can't import the configuration from a host that is neither
	  *  in the environment table nor authorized to export
	  */
	    clear_vperror();
	    authorized = idx < hostidx && !(int)host_table[idx][sizeof(host_t)-1];
	    if ( !authorized && idx == hostidx )
		vperror(0,"host <%s> not in the current environment",rhost);
	    if ( !authorized && idx < hostidx && (int)host_table[idx][sizeof(host_t)-1] )
		vperror(0,"host <%s> not authorized to export",rhost);
	    break;
    }

    return(authorized);
}

#endif
