/*
 * Copyright (C) 1992 by CERN/CN/SW/DC
 * All rights reserved
 */

#ifndef lint
static char sccsid[] = "@(#)mknet.c	3.4 04/06/94 CERN-SW/DC Fabrizio Cane";
#endif /* not lint */

#include <stdio.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#if defined( _AIX ) && defined( _IBMR2 )
#include <sys/select.h>
#endif
#include <errno.h>
#include <netinet/in.h>
#include <netdb.h>
#include <string.h>
#include <shift_types.h>
#include <osdep.h>
#include <strerror.h>
#include <sockutil.h>
#include <strtok.h>
#include <strdup.h>
#include <pktutil.h>
#include <config.h>
#include <mknet.h>

extern path_t shift_binary_net;
extern time_t config_version;

port_t shift_port;

/*
 *  Return the port number assigned for the DPMCONF network service
 * 
 *  RETURN
 *	-1	    port number not found
 *   otherwise	    port number
 */
port_t getdpmconfport()
{
  struct servent *server;
  static boolean init = FALSE;

    if ( !(shift_port > 0) && !init ) {
	init = TRUE;
	errno = 0;
	if ( (server = getservbyname("shiftmake","tcp")) == NULL ) {
	    if ( errno )
		vperror(1,"getservbyname(shiftmake,tcp)");
	    else
		vperror(0,"network service not found\nincomplete installation");
	    return(-1);
	}
	shift_port = ntohs(server->s_port);
    }
    return(shift_port);
}

/*
 *  Handle the read_packet() reply
 *
 *  INPUT
 *	reply		read_packet() reply
 *	copy		connection structure
 *	pkt		packet structure
 *	len		length of req
 *	req		list of expected requests
 *
 *  RETURN
 *	the input reply is returned
 */
int handle_read_packet(reply,copy,pkt,len,req)
int reply;
remote_t *copy;
pkt_t *pkt;
int len;
int *req;
{
  boolean connecting = copy->status == ST_CONNECTING;

    switch ( reply ) {

    /*
     *  The read_packet() succeed
     */
	case EPKTOK :
	    switch ( pkt->request ) {
	     /*
	      *  A message error received from the peer process
	      */
		case PKT_BADSTATUS :
		    copy->status = ST_PEERFAILURE;
		    copy->strerr = strdup(pkt->netbody);
		    break;
	     /*
	      *  Check if the packet was expected in the sequence 
	      */
		default :
		    while ( len-- > 0 )
			if ( pkt->request == req[len] )
			    return(reply);
		 /*
		  *  It wasn't expected
		  */
		    vperror(0,"unexpected packet (%d)",pkt->request);
		    copy->status = ST_SEQUENCE;
		    copy->strerr = strdup_vperror();
		    break;
	    } /* switch - request */
	    break;

    /*
     *  The read_packet() failed
     */
	case EPKTWAIT :
	    copy->status = ST_TIMEDOUT;
	    break;

	case EPKTZERO :
	    copy->status = ST_ZEROMSG;
	    break;

	case EPKTFAIL :
	case EPKTSIZE :
	case EPKTVERS :
	default :
	    copy->status = ST_FAILURE;
	    copy->strerr = strdup_vperror();
	    break;

    } /* switch - read_packet() reply */

 /*
  *  Some connect() calls that were in progress can fail here
  */
    if ( connecting ) {
	if ( copy->status == ST_ZEROMSG || copy->status == ST_TIMEDOUT )
		copy->strerr = strdup_vperror();
	else
		copy->strerr = strdup(strerror(errno));
	copy->status = ST_IMPOSSIBLE;
    }

 /*
  *  Send back to the peer process the error message
  */
    if ( !connecting && copy->status != ST_PEERFAILURE && pkt->request != EPKTFAIL )
	send_badstatus(copy,str_vperror());

 /*
  *  Close the connection and return the read_packet() reply
  */
    close(copy->sock);
    return(reply);
}

/*
 *  Handle the write_packet() reply
 *
 *  INPUT
 *	reply		write_packet() reply
 *	copy		connection structure
 */
int handle_writepacket(reply,copy)
int reply;
remote_t *copy;
{
    switch ( reply ) {

    /*
     *  The write_packet() succeed
     */
	case EPKTOK :
		break;

    /*
     *  The write_packet() failed
     */
	case EPKTPIPE :
	case EPKTWAIT :
	case EPKTFAIL :
	   /*
	    *  If the connection was already failed then ignore the reply
	    */
		if ( !(copy->status & ERRORMASK) ) {
		    copy->status = ST_FAILURE;
		    copy->strerr = strdup_vperror();
		    close(copy->sock);
		}
		break;
    }
    return(reply);
}

/*
 *  Send a request
 *
 *  INPUT
 *	copy		the connection
 *	request		the request to send
 *	status		the next state unless it fails
 */
int send_request(copy,request,status)
remote_t *copy;
int request;
int status;
{
  pkt_t pkt;

    pkt.request = request;
    pkt.length = 0;
    copy->status = status;
    return(handle_writepacket(write_packet(copy->sock,&pkt),copy));
}

/*
 *  Send a message error
 *
 *  INPUT
 *	copy		the connection
 *	errmsg		the error message to send
 */
int send_badstatus(copy,errmsg)
remote_t *copy;
char *errmsg;
{
  pkt_t pkt;

    pktinit(&pkt);
    pkt.request = PKT_BADSTATUS;
    pkt.netbody = errmsg;
    pkt.length = strlen(errmsg)+1;
    return(handle_writepacket(write_packet(copy->sock,&pkt),copy));
}

/*
 *  Receive unexpected packets and handle failures
 */
void recv_error(copy)
remote_t *copy;
{
  pkt_t pkt;

    pktinit(&pkt);
    handle_read_packet(read_packet(copy->sock,&pkt),copy,&pkt,0,(int*)NULL);
    pktfree(&pkt);
}

#define NETBUFFERSIZE	10240

/*
 *  Send the configuration network binary file
 *
 *  INPUT
 *	copy		the connection
 */
void send_binary(copy)
remote_t *copy;
{
  static pkt_t pkt;
  static boolean init = TRUE;

 /*
  *  Initialize the local packet structure
  */
    if ( init ) {
	pktinit(&pkt);
	init = FALSE;
    }

 /*
  *  Open the network binary file if this is the first call
  */
    if ( copy->fd == NULL ) {
	if ( (copy->fd = fopen(shift_binary_net,"r")) == NULL ) {
	    vperror(1,"fopen(%s,r)",shift_binary_net);
	    exception(e2);
	}
	if ( (copy->size = fsize(copy->fd)) < 0 ) {
	    vperror(1,"fsize(%s[%d])",shift_binary_net,fdtono(copy->fd));
	    exception(e1);
	}
    }

 /*
  *  Set the next packet body length and update the remaining data length
  */
    pkt.length = copy->size < NETBUFFERSIZE ? copy->size : NETBUFFERSIZE;
    copy->size -= pkt.length;

 /*
  *  Allocate memory for the packet body
  */
    if ( pktalloc(&pkt,pkt.length) == NULL )
	exception(e1);

 /*
  *  Load the next piece of network binary file
  */
    if ( fread(pkt.netbody,pkt.length,1,copy->fd) fread_ERROR ) {
	vperror(1,"fread(,,,%s[%d])",shift_binary_net,fdtono(copy->fd));
	exception(e1);
    }

 /*
  *  Send the piece of network binary file just loaded
  */
    pkt.request = copy->size == 0 ? PKT_ENDBINARY : PKT_SENDBINARY;
    if ( handle_writepacket(write_packet(copy->sock,&pkt),copy) != 1 ) {
	fclose(copy->fd);
	return;
    }

 /*
  *  Close the network binary file if this is the last call
  */
    if ( copy->size == 0 ) {
	fclose(copy->fd);
	copy->status = ST_ACKRECV;
    }
    return;

handle_exception(e1):	fclose(copy->fd);
handle_exception(e2):	copy->status = ST_FAILURE;
			copy->strerr = strdup_vperror();
			send_badstatus(copy,str_vperror());
			close(copy->sock);
}

/*
 *  Receive the configuration network binary file
 *
 *  INPUT
 *	copy		the connection
 */
void recv_binary(copy,pkt)
remote_t *copy;
pkt_t *pkt;
{
    switch ( pkt->request ) {
	case PKT_SENDBINARY :
	case PKT_ENDBINARY :
	 /*
	  *  Open the network binary file if this is the first call
	  */
	    if ( copy->fd == NULL )
		if ( (copy->fd = fopen(shift_binary_net,"w")) == NULL ) {
		    vperror(1,"fopen(%s,w)",shift_binary_net);
		    exception(e2);
		}

	 /*
	  *  Write to disk the piece of network binary file just received
	  */
	    if ( pkt->length > 0 && 
			fwrite(pkt->netbody,pkt->length,1,copy->fd) fwrite_ERROR ) {
		vperror(1,"fwrite(,,,%s[%d])",shift_binary_net,fdtono(copy->fd));
		exception(e1);
	    }

	 /*
	  *  Close and load the network binary file if this is the last call
	  */
	    if ( pkt->request == PKT_ENDBINARY ) {
		fclose(copy->fd);
		if ( unmarshall_binary_file() < 0 ) exception(e2);
		if ( check_tables() < 0 ) exception(e2);
		copy->status = ST_END;
	     /*
	      *  Store the version of the binary file received
	      */
		copy->version = config_version;
	    }
	    break;

    } /* switch - packet request */
    return;

handle_exception(e1):   fclose(copy->fd);
handle_exception(e2):	copy->status = ST_FAILURE;
			copy->strerr = strdup_vperror();
			send_badstatus(copy,str_vperror());
			close(copy->sock);
}

/*
 *  Connect to a set of remote hosts
 *
 *  INTPUT
 *	hostlist	list of host to connect
 *			(duplicated hosts and the local host are ignored)
 *
 *  OUTPUT
 *	copy		connection list
 *	counter		length of copy
 *
 *  RETURN
 *	ERROR		an error occured
 *	0		no host to connect
 *	1		successful completion
 */
int connect_copies(hostlist,copy,counter)
char *hostlist;
remote_t **copy;
int *counter;
{
  host_t localhost;
  port_t port;
  char *rhost;
  int idx;

 /*
  *  Get the local host
  */
    gethostname(localhost,sizeof(host_t));

 /*
  *  Get the number of hosts contained in the input list (excluded the local host)
  */
    *counter = 0;
    rhost = strtok(hostlist,HOSTSEPSET);
    while ( rhost!=NULL ) {
	if ( equal_hosts(localhost,rhost,(char**)NULL) == 0 )
	    (*counter)++;
	rhost = strtok((char*)NULL,HOSTSEPSET);
    }

 /*
  *  Return if the list is either empty or contains just the local host
  */
    if ( *counter == 0 ) return(0);

 /*
  *  Allocate memory to store the connections
  */
    if ( (*copy = (remote_t*)malloc(sizeof(remote_t)*(*counter))) == NULL ) {
	vperror(1,"malloc(%d)",sizeof(remote_t)*(*counter));
	return(ERROR);
    }

 /*
  *  Get the assigned service port
  */
    if ( (port=getdpmconfport()) < 0 )
	return(ERROR);

 /*
  *  Connect to the hosts contained in the input list
  */
    *counter = 0;
    rhost = strtok(hostlist,HOSTSEPSET);
    while ( rhost != NULL ) {

     /*
      *  Ignore host name duplications and the local host
      */
	for (idx=0;idx<*counter;idx++)
	    if ( equal_hosts((*copy)[idx].host,rhost,(char**)NULL) != 0 ) break;
	if ( idx == *counter && equal_hosts(localhost,rhost,(char**)NULL) == 0 ) {

	 /*
	  *  Initialize the connection structure
	  */
	    strcpy((*copy)[*counter].host,rhost);
	    (*copy)[*counter].fd = NULL;
	    (*copy)[*counter].version = 0;
	    (*copy)[*counter].imported = 0;
	    (*copy)[*counter].strerr = NULL;
	    (*copy)[*counter].printed = 0;
	    (*copy)[*counter].oper = ST_CONNECTING;

	 /*
	  *  Connect
	  */
	    if ( ((*copy)[*counter].sock = connect_to(UNDEF_SOCK,rhost,port,1)) < 0 ) {
		(*copy)[*counter].status = ST_IMPOSSIBLE;
		(*copy)[*counter].strerr = strdup(strerror(errno));
	    }
	    else
		(*copy)[*counter].status = ST_CONNECTING;

	    (*counter)++;	
	}
	rhost = strtok((char*)NULL,HOSTSEPSET);
    }

    return(1);
}

/*
 *  Dialog through the network
 *
 *  INPUT
 *	copy		connection list
 *	counter		length of copy
 *	netread		function handling the message to receive
 *	netwrite	function handling the message to send
 */
int netdialog(copy,counter,netread,netwrite,show)
remote_t *copy;
int counter;
void (*netread)();
void (*netwrite)();
boolean show;
{
  fd_set rmask,wmask,emask;
  int idx;
  int noconnections;

    if ( show ) print_connection_header();

 /*
  *  Loop until connections are closed
  */
    for(;;) {

     /*
      *  Build the read and write masks
      */
	FD_ZERO(&rmask);
	FD_ZERO(&wmask);
	noconnections = 0;
	for (idx=0;idx<counter;idx++) {
	    if ( (copy[idx].status & READMASK) )
		FD_SET(copy[idx].sock,&rmask);
	    if ( (copy[idx].status & WRITEMASK) )
		FD_SET(copy[idx].sock,&wmask);
	    if ( (copy[idx].status & (READMASK | WRITEMASK)) )
		noconnections++;
	}

     /*
      *  If no connection is active then return
      */
	if ( !noconnections )   break;

	switch ( select(FD_SETSIZE,&rmask,&wmask,(fd_set*)NULL,get_timeout()) ) {
	    case -1 :
		vperror(1,"select()");
		return(ERROR);
	    case 0 :
		for (idx=0;idx<counter;idx++)
		    if ( !(copy[idx].status & ERRORMASK) &&
				(copy[idx].status & (READMASK | WRITEMASK)) )
			copy[idx].status = ST_TIMEDOUT;
		break;
	    default :
		for (idx=0;idx<counter;idx++) {
		    clear_vperror();
		    if ( (copy[idx].status & READMASK) 
					&& FD_ISSET(copy[idx].sock,&rmask) )
			(*netread)(copy,counter,&copy[idx]);
		    if ( (copy[idx].status & WRITEMASK) 
					&& FD_ISSET(copy[idx].sock,&wmask) )
			(*netwrite)(copy,counter,&copy[idx]);
		    if ( show && !copy[idx].printed &&
					!(copy[idx].status & (READMASK | WRITEMASK)) )
			print_connection_status(&copy[idx]);
		}
	}

   } /* loop */

   return(1);
}

/*
 *
 */
char *strnull(str)
char *str;
{
  static char *null = "(null)";

    return( str != NULL ? str : null );
}

/*
 *
 */
void print_connection_header()
{
    printf("%-12s%-28s%s\n","  HOST","    REMOTE VERSION","  STATUS");
}

/*
 *
 */
int print_connection_status(copy)
remote_t *copy;
{
  char operation[10];
  FILE *stream;
  boolean error;

    error = copy->status & ERRORMASK ;

    if ( copy->printed )
	return(error);
    else
	copy->printed = TRUE;

    stream = error ? stderr : stdout ;

 /*
  *
  */
    switch ( copy->oper ) {
	case ST_IMPORT :
		strcpy(operation,"import ");
		break;
	case ST_EXPORT : 
		strcpy(operation,"export ");
		break;
	case ST_ACCEPT :
		strcpy(operation,"handle ");
		break;
	case ST_CONNECTING : 
		strcpy(operation,"connect");
		break;
	default : 
		strcpy(operation,"(unknown)");
    }

 /*
  *  Print the host name
  */
    fprintf(stream,"%-12s",copy->host);

 /*
  *  Print the configuration version unless the connection failed
  */
    if ( !error )
	if ( copy->version > 0 )
	    fprintf(stream,"%-28s",ctimeln(&(copy->version)));
	else
	if ( copy->oper == ST_EXPORT )
	    fprintf(stream,"%-28s"," configuration missing ");
	else
	    fprintf(stream,"%-28s","");

 /*
  *  Print either the error messages or warnings
  */
    switch ( copy->status ) {

      case ST_END :
	if ( copy->oper == ST_IMPORT )
	    if ( copy->version == 0 )
		fprintf(stream,"WARNING: configuration missing\n");
	    else
	    if ( config_version > copy->version )
		fprintf(stream,"WARNING: too old\n");
	    else
	    if ( copy->imported )
		fprintf(stream,"imported\n");
	    else
		fprintf(stream,"\n");
	else
	    if ( config_version > copy->version )
		fprintf(stream,"exported\n");
	    else
	    if ( config_version == copy->version )
		fprintf(stream,"\n");
	    else
		fprintf(stream,"WARNING: too recent\n");
	break;

      case ST_IMPOSSIBLE :
	fprintf(stream,"Can't connect: %s\n",strnull(copy->strerr));
	break;

      case ST_TIMEDOUT :
	fprintf(stream,"Can't %s: protocol timed out\n",operation);
	break;

      case ST_FAILURE :
	fprintf(stream,"Can't %s: %s\n",operation,strnull(copy->strerr));
	break;

      case ST_PEERFAILURE :
	fprintf(stream,"Can't %s: peer failure (%s)\n",operation,strnull(copy->strerr));
	break;

      case ST_ZEROMSG :
	fprintf(stream,"Can't %s: zero-message received\n",operation);
	break;

      case ST_SEQUENCE :
	fprintf(stream,"Can't %s: %s\n",operation,strnull(copy->strerr));
	break;

      default :
	vperror(0,"print_connection_status(): unknown state (%d)",copy->status);

    } /* switch */

    return(error);
}

/*
 *  Print the status of all connections
 *
 *  RETURN
 *	0	all connections succeed
 *	1	at least one connection failed
 */
int print_connection_results(copy,counter)
remote_t *copy;
int counter;
{
  boolean error;
  int idx;

    error = FALSE;
    for (idx=0; idx<counter; idx++)
	error = print_connection_status(&copy[idx]) || error;

    return(error);
}

