/*
 * Copyright (C) 1991,1992 by CERN/CN/SW/DC
 * All rights reserved
 */

#ifndef lint
static char sccsid[] = " @(#)table.c	3.8 05/07/93  CERN-SW/DC Fabrizio Cane";
#endif /* not lint */

#include <sys/types.h>
#include <sys/time.h>
#include <stdio.h>
#include <string.h>
#ifndef apollo
#include <malloc.h>
#endif
#include <shift_types.h>
#include <strutil.h>
#include <strdup.h>
#include <parser.h>
#include <table.h>
#include <user.h>

index_t_s  lastpoolidx,lastsuperpool;
group_t  lastgroupname;
int      lastgroupid;
boolean  lastgroupgarbage;
int	 lastgrouptype;
int	 lastsuperpoolaccess;
index_t_s  categoryidx;
index_t_s  itemidx;

/*
 *  Pool default attributes
 */
static struct  {
    boolean  permanent,private;
    int	     min,max;
    int	     size;
} defaults = { FALSE, FALSE, 0, 0, 0 };

/*
 *  Macro to generate the table allocation/deallocation functions
 *
 *  alloc_<func>() : allocation functions
 *
 *  INPUT
 *	table		table to allocate
 *	size		size table required
 *
 *  OUTPUT
 *	table		allocated table whose size is at least that required
 *
 *  RETURN
 *	1		successful completion
 *	-1		an error occured
 *
 *  free_<func>() : deallocation functions
 *
 */
#define alloc_table(alloc_func,free_func,type,table,counter,index)\
\
type     *table = (type*)NULL;\
index_t_s  index = 0;\
index_t_s  counter = 0;\
\
int alloc_func(table,size)\
type **table;\
int  size;\
{\
  type *old_table;\
\
    if ( *table == NULL ) {\
	counter = size > 0 ? size : INIT_TABLESIZE ;\
	if ( (*table = (type*)malloc(sizeof(type)*counter)) == NULL ) {\
	    vperror(1,"malloc(%d*%d)",sizeof(type),counter);\
	    return(-1);\
	}\
    }\
    else {\
	if ( 0 < size && size < counter )\
	    return(1);\
	counter = size > counter + STEP_TABLESIZE ? size : counter + STEP_TABLESIZE ;\
	old_table = *table;\
	if ( (*table = (type*)realloc(old_table,sizeof(type)*counter)) == NULL ) {\
		vperror(1,"realloc(%x,%d*%d)",old_table,sizeof(type),counter);\
		free(old_table);\
		return(-1);\
	}\
    }\
    return(1);\
}\
\
void free_func()\
{\
    if ( table ) {\
	index = 0;\
	counter = 0;\
	free((char*)table);\
	table = (type*)NULL;\
    }\
}

/*
 *  Table allocation/deallocation functions ( see macro alloc_table )
 */
alloc_table(alloc_disk,free_disk,disk_t,disk_table,no_disks,diskidx)
alloc_table(alloc_pool,free_pool,pool_t,pool_table,no_pools,poolidx)
alloc_table(alloc_supool,free_supool,supool_t,supool_table,no_supools,supoolidx)
alloc_table(alloc_fsys,free_fsys,fsys_t,fsys_table,no_fsys,fsysidx)
alloc_table(alloc_accs,free_accs,accs_t,accs_table,no_accs,accsidx)
alloc_table(alloc_host,free_host,host_t,host_table,no_host,hostidx)
alloc_table(alloc_pair_tbl,free_pair_tbl,pair_t,pair_table,no_pair,pairidx)

/*
 *  Allocation of the pair table requires to initialize the field value
 */
int alloc_pair(table,size)
pair_t **table;
int size;
{
  int idx,old_no_pair = no_pair;
    if ( alloc_pair_tbl(table,size) < 0 )
	return(-1);
    for (idx=old_no_pair; idx<no_pair; idx++)
	pair_table[idx].value = NULL;
    return(1);
}

/*
 *  Deallocation of the pair table requires to deallocate first the field value 
 */
void free_pair()
{
  int idx;
    for (idx=0; idx<pairidx; idx++)
	if ( pair_table[idx].value )
	    free(pair_table[idx].value);
    free_pair_tbl();
}

#ifdef MKTBL_TRACE

/*
 *  Print the disk server table
 *
 *  FORMAT
 *
 *	SERVER	  <host-name>	 <mount-path>
 */
void print_disk_table()
{
  static index_t_s idx;
    for (idx=0;idx<diskidx;idx++)
	printf("SERVER\t\t%-12s %s\n",disk_table[idx].host,disk_table[idx].mount);
}

/*
 *  Print the file system table
 *
 *  INPUT
 *	cross		if false then the cross references to the disk server
 *			and pool tables are printed otherwise it navigates in
 *			order to print the complete disk server and pool name
 *
 *  FORMAT
 *
 *	FILE SYSTEM	<host-name>	<filesystem-name>	<pool-name>
 */
void print_fsys_table(cross)
boolean cross;
{
  static index_t_s idx;
    for (idx=0;idx<fsysidx;idx++) {
	printf("FILE SYSTEM\t");
	if ( cross )
	    printf("%-12s",disk_table[fsys_table[idx].diskidx].host);
	else
	    printf("%-12d",fsys_table[idx].diskidx);
	printf("%-20s",fsys_table[idx].name);
	if ( cross )
	    printf("%s\n",pool_table[fsys_table[idx].poolidx].name);
	else
	    printf("%d\n",fsys_table[idx].poolidx);
    }
}

/*
 *  Print the pool table
 *
 *  FORMAT
 *
 *	POOL	<pool-name>   PRIVATE/PUBLIC   PERMANENT/TEMPORARY   COLLECT()   SIZE()
 */
void print_pool_table()
{
  static index_t_s idx;
    for (idx=0;idx<poolidx;idx++) {
	printf("POOL\t\t%-15s%-10s%-12s%",
			pool_table[idx].name,
			pool_table[idx].private ? "PRIVATE" : "PUBLIC",
			pool_table[idx].permanent ? "PERMANENT" : "TEMPORARY" );
	if ( pool_table[idx].size )
	    printf("SIZE(%s)   ",itoanmu(pool_table[idx].size));
	if ( pool_table[idx].min || pool_table[idx].max )
	    printf("COLLECT(%d:%d)",pool_table[idx].min,pool_table[idx].max);
	newline;
    }
}

/*
 *  Print the super pool table
 *
 *  FORMAT
 *
 *	SUPERPOOL  <name>  <idx>
 */
void print_supool_table()
{
  static index_t_s idx;
    for (idx=0;idx<supoolidx;idx++)
	printf("SUPERPOOL %-12s %d\n",supool_table[idx].name,supool_table[idx].idx);
}

/*
 *  Print the account table
 *
 *  INPUT
 *	cross		if false then the cross reference to the pool table
 *			and the uid/gid are printed otherwise it navigates in
 *			order to print the complete pool and user/group name
 *
 *  FORMAT
 *
 *	ACCOUNT	<pool-name>	<group-name> : <user-name>	<garbage-authorization>
 */
void print_accs_table(cross)
boolean  cross;
{
  static index_t_s idx;
  static user_t  user;
  static group_t group;
    for (idx=0;idx<accsidx;idx++) {
	printf("ACCOUNT\t\t");
	if ( !cross )
	    printf("%-15d%d : %d %s\n",
			accs_table[idx].poolidx,
			accs_table[idx].gid,accs_table[idx].uid,
			accs_table[idx].garbage ? "(G)" : "");
	else {
	    get_shiftuserbyid(user,accs_table[idx].uid);
	    get_shiftgroupbyid(group,accs_table[idx].gid);
	    printf("%-15s%s : %s %s\n",
			pool_table[accs_table[idx].poolidx].name,
			group,user,
			accs_table[idx].garbage ? "(G)" : "");
	}
    }
}

/*
 *  Print the pair table
 *
 *  FORMAT
 *
 *	<name>		<next>		<value>
 */
void print_pair_table()
{
  static index_t_s idx;
    for (idx=0;idx<pairidx;idx++) {
	printf("%-20s   +%2d   ",pair_table[idx].name,pair_table[idx].next);
	if ( pair_table[idx].value != NULL )
	    printf("<%s>\n",pair_table[idx].value);
	else
	    printf("(nill)\n");
    }
}

#endif

/*
 *  Add a new server in the disk server table
 *
 *  INPUT
 *	name		disk server name
 *	mount		mount path prefix
 *
 *  RETURN
 *	1		successful completion
 *	-1		an error occured
 */
int add_server(name,mount)
char  *name,*mount;
{
#ifdef MKTBL_TRACE
    printf("::SERVER \t%s\t%s\n",name,mount);
#endif
    if ( gethostbyname(name) == NULL ) {
	yyprintf("\n");
	yyerror("unknown host <%s>",name);
	return(-1);
    }
    if ( strlen(name) > sizeof(host_t)-1 ) {
	yyprintf("\n");
	yyerror("host name <%s> too long (max %d)",name,sizeof(host_t)-1);
	return(-1);
    }
    if ( strlen(mount) > sizeof(path_t)-1 ) {
	yyprintf("\n");
	yyerror("mount path <%s> too long (max %d)",mount,sizeof(path_t)-1);
	return(-1);
    }
    if ( !( look_disk(name) < 0 ) ) {
	yyprintf("\n");
	yyerror("server <%s> already declared",name);
	return(-1);
    }
    if ( alloc_disk(&disk_table,diskidx) < 0 )
	return(-1);
    strcpy(disk_table[diskidx].host,name);
    strcpy(disk_table[diskidx].mount,mount);
    diskidx++;
    return(1);
}

/*
 *  Look for a disk server in the table
 *
 *  INPUT
 *	name		disk server name to look for
 *
 *  RETURN
 *	-1		disk server not found
 *   otherwise		disk server table index
 */
index_t_s look_disk(name)
char *name;
{
  static index_t_s idx;
    for (idx=0;idx<diskidx;idx++)
	if ( !strcmp(disk_table[idx].host,name) )
	    return( idx );
    return( -1 );
}

/*
 *  Add a new file system to the last disk server
 *
 *  INPUT
 *	name		file system name
 *
 *  RETURN
 *	1		successful completion
 *	-1		an error occured
 */
int add_filesystosrv(name,bits)
char  *name;
int   bits;
{
#ifdef MKTBL_TRACE
    printf("::\t%s\n",name);
#endif
    if ( strlen(name) > sizeof(filesys_name_t)-1 ) {
	yyprintf("\n");
	yyerror("file system name <%s> too long (max %d)",name,sizeof(filesys_name_t)-1);
	return(-1);
    }
    if ( !( look_fsys(diskidx-1,name) < 0 ) ) {
	yyprintf("\n");
	yyerror("file system <%s> already declared",name);
	return(-1);
    }
    if ( alloc_fsys(&fsys_table,fsysidx) < 0 )
	return(-1);
    strcpy(fsys_table[fsysidx].name,name);
    fsys_table[fsysidx].diskidx = diskidx-1;
    fsys_table[fsysidx].poolidx = -1;
    fsys_table[fsysidx].bits = bits;
    fsysidx++;
    return(1);
}

/*
 *  Look for a file system in the table
 *
 *  INPUT
 *	srvidx		index of the disk server
 *	name		file system name
 *
 *  RETURN
 *	-1		file system not found
 *   otherwise		file system table index
 */
index_t_s look_fsys(srvidx,name)
index_t_s srvidx;
char    *name;
{
  static index_t_s idx;
    for (idx=0;idx<fsysidx;idx++)
	if ( !strcmp(fsys_table[idx].name,name) && fsys_table[idx].diskidx==srvidx )
	    return( idx );
    return( -1 );
}

/*
 *  Add a new pool to the pool table
 *
 *  INPUT
 *	name		pool name
 *
 *  RETURN
 *	1		successful completion
 *	-1		an error occured
 */
int add_pool(name)
char  *name;
{
#ifdef MKTBL_TRACE
    printf("::POOL\t%s\n",name);
#endif
    if ( strlen(name) > sizeof(pool_name_t)-1 ) {
	yyprintf("\n");
	yyerror("pool name <%s> too long (max %d)",name,sizeof(pool_name_t)-1);
	return(-1);
    }
    if ( !( look_pool(name) < 0 ) ) {
	yyprintf("\n");
	yyerror("pool <%s> already declared",name);
	return(-1);
    }
    if ( alloc_pool(&pool_table,poolidx) < 0 )
	return(-1);
    strcpy(pool_table[poolidx].name,name);
    pool_table[poolidx].permanent = defaults.permanent;
    pool_table[poolidx].private = defaults.private;
    pool_table[poolidx].min = defaults.min;
    pool_table[poolidx].max = defaults.max;
    pool_table[poolidx].size = defaults.size;
    poolidx++;
    return(1);
}

/*
 *  Add a new file system to the last pool
 *
 *  INPUT
 *	server		disk server name
 *	path		file system name
 *
 *  RETURN
 *	1		successful completion
 *	-1		an error occured
 */
int add_filesystopool(server,path,bits)
char *server,*path;
int bits;
{
  static index_t_s srvidx,fsidx;
#ifdef MKTBL_TRACE
    printf("::\t%s\t%s\n",server,path);
#endif
    if ( (srvidx = look_disk(server)) < 0 ) {
	yyprintf("\n");
	yyerror("server <%s> not declared",server);
	return( -1 );
    }
    if ( (fsidx = look_fsys(srvidx,path)) < 0 ) {
	yyprintf("\n");
	yyerror("file system <%s> not declared in server <%s>",path,server);
	return( -1 );
    }
    if ( fsys_table[fsidx].poolidx != -1 ) {
	yyprintf("\n");
	yyerror("file system <%s:%s> already assigned to the pool <%s>",
		server,path,pool_table[fsys_table[fsidx].poolidx].name);
	return(-1);
    }
    fsys_table[fsidx].poolidx = poolidx-1;
    fsys_table[fsidx].bits = fsys_table[fsidx].bits | bits;
    return(1);
}

/*
 *  Look for a pool in the table
 *
 *  INPUT
 *	name		pool name
 *
 *  RETURN
 *	-1		pool not found
 *   otherwise		pool table index
 */
index_t_s look_pool(name)
char *name;
{
  static index_t_s idx;
    for (idx=0;idx<poolidx;idx++)
	if ( !strcmp(pool_table[idx].name,name) )
	    return( idx );
    return( -1 );
}

/*
 *  Add the permanent attribute to either the last pool 
 *  or to the default attributes set
 *
 *  RETURN
 *	1		successful completion
 */
int add_permanent()
{
#ifdef MKTBL_TRACE
    printf("::PERMANENT\n");
#endif
    if ( no_pools )
	pool_table[poolidx-1].permanent = 1;
    else
	defaults.permanent = 1;
    return(1);
}

/*
 *  Add the temporary attribute to either the last pool
 *  or to the default attributes set
 *
 *  RETURN
 *	1		successful completion
 */
int add_temporary()
{
#ifdef MKTBL_TRACE
    printf("::TEMPORARY\n");
#endif
    if ( no_pools )
	pool_table[poolidx-1].permanent = 0;
    else
	defaults.permanent = 0;
    return(1);
}

/*
 *  Add the public attribute to either the last pool
 *  or to the default attributes set
 *
 *  RETURN
 *	1		successful completion
 */
int add_public()
{
#ifdef MKTBL_TRACE
    printf("::PUBLIC\n");
#endif
    if ( no_pools )
	pool_table[poolidx-1].private = 0;
    else
	defaults.private = 0;
    return(1);
}

/*
 *  Add the private attribute to either the last pool
 *  or to the default attributes set
 *
 *  RETURN
 *	1		successful completion
 */
int add_private()
{
#ifdef MKTBL_TRACE
    printf("::PRIVATE\n");
#endif
    if ( no_pools )
	pool_table[poolidx-1].private = 1;
    else
	defaults.private = 1;
    return(1);
}

/*
 *  Add the collect attribute to either the last pool
 *  or to the default attributes set
 *
 *  RETURN
 *	1		successful completion
 */
int add_collect(collect)
collect_t  *collect;
{
#ifdef MKTBL_TRACE
    printf("::COLLECT(%d:%d)\n",collect->min,collect->max);
#endif
    if ( collect->min < 0 || collect->min > 100 ||
		collect->max < 0 || collect->max > 100 ||
			(collect->min >= collect->max && collect->min > 0) ) {
	yyprintf("\n");
	yyerror("invalid collect thresholds");
	return(-1);
    }
    if ( no_pools ) {
	pool_table[poolidx-1].min = collect->min;
	pool_table[poolidx-1].max = collect->max;
    }
    else {
	defaults.min = collect->min;
	defaults.max = collect->max;
    }
    return(1);
}

/*
 *  Add the size attribute to either the last pool
 *  or to the default attributes set
 *
 *  INPUT
 *	sizestr		string in the format :== number | number 'K' | number 'M'
 *
 *  RETURN
 *	1		successful completion
 *	-1		incorrect size
 */
int add_size(sizestr)
char *sizestr;
{
  static long size;
#ifdef MKTBL_TRACE
    printf("::SIZE(%s)\n",sizestr);
#endif
    if ( (size = (long)anutof(sizestr)) < 0 ) {
	yyprintf("\n");
	switch ( size ) {
	case -1 : yyerror("invalid unit character in size");
		  break;
	case -2 :
	default : yyerror("invalid number in size");
		  break;
	}
	return(-1);
    }
    if ( no_pools )
	pool_table[poolidx-1].size = size;
    else
	defaults.size = size;
    return(1);
}

/*
 *  Add a new super pool in the super pool table
 *
 *  INPUT
 *	name		pool name
 *	access		0 / READ access only  -  1 / WRITE and read access
 *
 *  RETURN
 *      1               successful completion
 *      -1              an error occured
 */
int add_superpool(name,access)
char *name;
int access;
{
#ifdef MKTBL_TRACE
    printf("::SUPERPOOL %s %d\n",name,access);
#endif
    if ( strlen(name) > sizeof(pool_name_t)-1 ) {
	yyprintf("\n");
	yyerror("super pool name <%s> too long (max %d)",name,sizeof(pool_name_t)-1);
	return(-1);
    }
    if ( !( look_pool(name) < 0 ) ) {
	yyprintf("\n");
	yyerror("<%s> already declared as a pool",name);
	return(-1);
    }
    if ( !( look_supool(name) < 0 ) ) {
	yyprintf("\n");
	yyerror("super pool <%s> already declared",name);
	return(-1);
    }
    if ( alloc_supool(&supool_table,supoolidx) < 0 )
	return(-1);
    strcpy(supool_table[supoolidx].name,name);
    supool_table[supoolidx].idx = 0;
    lastsuperpool = supoolidx;
    lastsuperpoolaccess = access;
    supoolidx++;
    return(1);
}

/*
 *  Add a new element in the last super pool definition
 *
 *  INPUT
 *	name		pool / super pool name that is element of the definition
 *	priority	0 / same priority (+)  -  1 / higher priority (:)
 *
 *  RETURN
 *      1               successful completion
 *      -1              an error occured
 */
int add_superpoolelement(name,priority)
char *name;
int priority;
{
#ifdef MKTBL_TRACE
    printf("::%s %d\n",name,priority);
#endif
    if ( look_pool(name) < 0 && look_supool(name) < 0 ) {
	yyprintf("\n");
	yyerror("<%s> is neither a pool nor a super pool",name);
	return(-1);
    }
    if ( alloc_supool(&supool_table,supoolidx) < 0 )
	return(-1);
    strcpy(supool_table[supoolidx].name,name);
    if ( priority < 0 )
	supool_table[supoolidx].idx = lastsuperpoolaccess;
    else
	supool_table[supoolidx].idx = priority;
    supoolidx++;
    supool_table[lastsuperpool].idx++;
    return(1);
}

/*
 *  Look for a super pool in the table
 *
 *  INPUT
 *	name		super pool name to look for
 *
 *  RETURN
 *      -1              super pool not found
 *   otherwise          super pool table index
 */
int look_supool(name)
char *name;
{
  static index_t_s idx;
    idx = 0;
    while ( idx < supoolidx ) {
	if ( !strcmp(supool_table[idx].name,name) )
	    return( idx );
	else
	    idx += supool_table[idx].idx + 1 ;
    }
    return( -1 );
}

/*
 *  Return the access mode specified for the given super pool
 *
 *  INPUT
 *	idx		super pool table index
 */
int getsupoolaccess(idx)
index_t_s idx;
{
    if ( idx < 0 )
	return(-1);
    else
	return( supool_table[ idx + supool_table[idx].idx ].idx );
}

/*
 *  Add the default account ( *.* ) to the public pools
 */
int init_poolaccount()
{
  int idx;
  boolean first = TRUE;
#ifdef MKTBL_TRACE
    printf("init_poolaccount()\n");
#endif
    for (idx=0; idx<poolidx; idx++)
	if ( !pool_table[idx].private ) {
	    if ( add_poolaccount(pool_table[idx].name) < 0 ) return(-1);
	    if ( first ) {
		if ( add_groupaccount("*",FALSE) < 0 ) return(-1);
		first = FALSE;
	    }
	    if ( add_useraccount("*",FALSE) < 0 ) return(-1);
	}
    return(1);
}

/*
 *  Set the pool the next accounts refer to
 *
 *  INPUT
 *	pool		pool name
 *
 *  RETURN
 *	1		successful completion
 *	-1		an error occured
 */
int add_poolaccount(pool)
char  *pool;
{
#ifdef MKTBL_TRACE
    printf("::POOL\t%s\n",pool);
#endif
    if ( (lastpoolidx = look_pool(pool)) < 0 ) {
	yyerror("pool <%s> not declared",pool);
	return( -1 );
    }
    return(1);
}

/*
 *  Set the group the next accounts refer to
 *
 *  INPUT
 *	group		shift group name
 *	garbage		garbage authorization flag 
 *
 *  RETURN
 *	1		successful completion
 *	-1		an error occured
 */
int add_groupaccount(group,garbage)
char     *group;
boolean  garbage;
{
#ifdef MKTBL_TRACE
    printf("::    %s%s : \n",group,garbage ? "(G)" : "");
#endif
    if ( strlen(group) > sizeof(group_t)-1 ) {
	yyprintf("\n");
	yyerror("group name <%s> too long (max %d)",group,sizeof(group_t)-1);
	return(-1);
    }
    if ( get_shiftgroupbyname(group,&lastgroupid,&lastgrouptype) < 0 ) {
	yyprintf("\n");
	yyerror("unknown group name <%s>",group);
	return( -1 );
    }
    lastgroupgarbage = garbage;
    strcpy(lastgroupname,group);
    return(1);
}

/*
 *  Add a new account (authorization) in the account table
 *
 *  INPUT
 *	user		shift user name
 *	garbage		garbage authorization flag
 *
 *  RETURN
 *	1		successful completion
 *	-1		an error occured
 */
int add_useraccount(user,garbage)
char     *user;
boolean  garbage;
{
  int gid,usertype;
#ifdef MKTBL_TRACE
    printf("::%s%s\n",user,garbage ? "(G)" : "");
#endif
    if ( strlen(user) > sizeof(user_t)-1 ) {
	yyprintf("\n");
	yyerror("user name <%s> too long (max %d)",user,sizeof(user_t)-1);
	return(-1);
    }
    if ( alloc_accs(&accs_table,accsidx) < 0 )
	return(-1);
    if ( get_shiftuserbyname(user,&accs_table[accsidx].uid,&gid,&usertype) < 0 ) {
	yyprintf("\n");
	yyerror("unknown user name <%s>",user);
	return(-1);
    }
#if !defined( NOUSRCHK )
    if ( !usertype && !lastgrouptype && lastgroupid != gid ) {
	yyprintf("\n");
	yyerror("user <%s> is not in group <%s>",user,lastgroupname);
	return(-1);
    }
#endif
    accs_table[accsidx].garbage = garbage || lastgroupgarbage;
    if ( !( look_account(lastpoolidx,
			 accs_table[accsidx].uid,lastgroupid,
			 accs_table[accsidx].garbage) < 0 ) ) {
     /*
      *  Init_poolaccount() adds the *.* account to the public pools ( skip those! )
      */
	if ( !pool_table[lastpoolidx].private && !accs_table[accsidx].garbage &&
		accs_table[accsidx].uid == ANY_UID && lastgroupid == ANY_GID )
	    return(1);
	yyprintf("\n");
	yyerror("account <%s@%s> already declared",user,pool_table[lastpoolidx].name);
	return(-1);
    }
    accs_table[accsidx].poolidx = lastpoolidx;
    accs_table[accsidx].gid = lastgroupid;	     /* gid is not always the same! */
    accsidx++;
    return(1);
}

/*
 *  Look for an account in the table
 *
 *  INPUT
 *	poolidx		index of the pool
 *	gid		group id
 *	uid		user id
 *	garbage		garbage authorization flag
 *
 *  RETURN
 *	-1		account not found
 *   otherwise		account table index
 */
index_t_s look_account(poolidx,uid,gid,garbage)
index_t_s poolidx;
int uid,gid;
boolean garbage;
{
  static index_t_s idx;
    for (idx=0;idx<accsidx;idx++)
	if ( accs_table[idx].poolidx == poolidx &&
		(accs_table[idx].uid == ANY_UID || accs_table[idx].uid == uid) &&
		    (accs_table[idx].gid == ANY_GID || accs_table[idx].gid == gid) &&
			(accs_table[idx].garbage>0) >= (garbage>0) )
	    return(idx);
    return(-1);
}

/*
 *  Add a new category in the pair table
 *
 *  INPUT
 *	name		category name
 *
 *  RETURN
 *	1		successful completion
 *	-1		an error occured
 */
int add_category(name)
char *name;
{
#ifdef MKTBL_TRACE
    printf("::category  %s\n",name);
#endif
    if ( strlen(name) > sizeof(pair_name_t)-1 ) {
	yyprintf("\n");
	yyerror("category name <%s> too long (max %d)",name,sizeof(pair_name_t)-1);
	return(-1);
    }
    if ( pairidx > 0 && pair_table[categoryidx].next == 1 ) {
	yyprintf("\n");
	yyerror("category <%s> has no item",pair_table[categoryidx].name);
	return(-1);
    }
    if ( !( look_pair(name,NULL,NULL) < 0 ) ) {
	yyprintf("\n");
	yyerror("category <%s> already declared");
	return(-1);
    }
    if ( alloc_pair(&pair_table,pairidx) < 0 )
	return(-1);
    strcpy(pair_table[pairidx].name,name);
    pair_table[pairidx].value = NULL;
    pair_table[pairidx].next = 1;
    categoryidx = pairidx;
    pairidx++;
    return(1);
}

/*
 *  Add a new item in the pair table
 *
 *  INPUT
 *	name		item name
 *	value		item value
 *
 *  RETURN
 *	1		successful completion
 *	-1		an error occured
 */
int add_item(name,value)
char *name,*value;
{
#ifdef MKTBL_TRACE
    printf("::item  %s",name);
    if ( value ) printf(" : <%s>",value);
    newline;
#endif
    if ( strlen(name) > sizeof(pair_name_t)-1 ) {
	yyerror("item name <%s> too long (max %d)",name,sizeof(pair_name_t)-1);
	return(-1);
    }
    if ( !( look_pair(pair_table[categoryidx].name,name,NULL) < 0 ) ) {
	yyerror("item <%s> already declared",name);
	return(-1);
    }
    if ( alloc_pair(&pair_table,pairidx) < 0 )
	return(-1);
    strcpy(pair_table[pairidx].name,name);
    pair_table[pairidx].value = NULL;
    if ( value && strlen(value) && (pair_table[pairidx].value=strdup(value)) == NULL ) {
	vperror(1,"strdup(%d)",strlen(value)+1);
	return(-1);
    }
    pair_table[pairidx].next = 1;
    itemidx = pairidx;
    pairidx++;
    pair_table[categoryidx].next++;
    return(1);
}

#if !defined( NCHKPHOST ) && !defined( CHKPHOST )
/*
 *  Check the host name be locally unknown
 *
 *  INPUT
 *	host		host name
 *	skip		skip the 'unknown' prefix
 */
boolean is_unknown_host(host,skip)
char **host;
boolean skip;
{
    if ( *host == NULL || strncmp(*host,UNKPHOST,strlen(UNKPHOST)) ) 
	return(FALSE);

    if ( skip ) (*host) += strlen(UNKPHOST);
    return(TRUE);
}
#endif

/*
 *  Add a new host specific item value in the pair table
 *
 *  INPUT
 *	name		host name
 *	value		item value to be used for the given host
 *
 *  RETURN
 *	1		successful completion
 *	-1		an error occured
 */
int add_hostvalue(name,value)
char *name,*value;
{
#ifdef MKTBL_TRACE
    printf("::host  %s",name);
    if ( value ) printf(" : <%s>",value);
    newline;
#endif
    if ( strlen(name) > sizeof(pair_name_t)-1 ) {
	yyerror("host name <%s> too long (max %d)",name,sizeof(pair_name_t)-1);
	return(-1);
    }
#if !defined( NCHKPHOST )
#if !defined( CHKPHOST )
    if ( !is_unknown_host(&name,FALSE) )
#endif
    if ( gethostbyname(name) == NULL ) {
	yyerror("unknown host <%s>",name);
	return(-1);
    }
#endif
    if ( !(look_pair(pair_table[categoryidx].name,pair_table[itemidx].name,name) < 0) ) {
	yyprintf("\n");
	yyerror("host <%s> already declared",name);
	return(-1);
    }
    is_unknown_host(&name,TRUE);
    if ( alloc_pair(&pair_table,pairidx) < 0 )
	return(-1);
    strcpy(pair_table[pairidx].name,name);
    if ( pair_table[pairidx].value ) free(pair_table[pairidx].value);
    pair_table[pairidx].value = NULL;
    if ( value && strlen(value) && (pair_table[pairidx].value=strdup(value)) == NULL ) {
	vperror(1,"strdup(%d)",strlen(value)+1);
	return(-1);
    }
    pair_table[pairidx].next = 1;
    pairidx++;
    pair_table[categoryidx].next++;
    pair_table[itemidx].next++;
    return(1);
}

/*
 *  Look for a pair in the table
 *
 *  CALL
 *	add_hostvalue() and getconfent()
 *
 *  INPUT
 *	category	category name
 *	item		item name
 *	host		host name
 *
 *  RETURN
 *	-1		pair not found
 *   otherwise		pair table index
 */
index_t_s look_pair(category,item,host)
char *category,*item,*host;
{
  static index_t_s idx,cidx,iidx;
#if !defined( NCHKPHOST ) && !defined( CHKPHOST )
  static boolean unknown;
    unknown = is_unknown_host(&host,TRUE);
#endif
    idx = 0;
    while( category && idx < pairidx )
	if ( strcmp(pair_table[idx].name,category) )
	    idx += pair_table[idx].next;
	else {
	    cidx = idx++;
	    while( item && idx < cidx + pair_table[cidx].next )
		if ( strcmp(pair_table[idx].name,item) )
		    idx += pair_table[idx].next;
		else {
		    iidx = idx++;
		    while ( host && idx < iidx + pair_table[iidx].next )
#if !defined( NCHKPHOST ) && !defined( CHKPHOST )
			if ( unknown )
#endif
#if !defined( CHKPHOST )
			if ( strcmp(pair_table[idx].name,host) )
#endif
#if !defined( NCHKPHOST ) && !defined( CHKPHOST )
			    idx++;
			else
			    return(idx);
			else
#endif
#if !defined( NCHPHOST )
			if ( equal_hosts(pair_table[idx].name,host,(char**)NULL) != 1 )
#endif
			    idx++;
			else
			    return(idx);
		    return( host ? -1 : iidx );
		}
	    return( item ? -1 : cidx );
	}
    return(-1);
}

/*
 *  Check the configuration stored in memory
 *
 *  CHECKS
 *	- The server names in the DPM configuration must be hosts locally known
 *	- The host names in the PAIRS configuration must be locally known
 *
 *  CALLS
 *	- when the configuration is received from a remote host!
 */
int check_tables()
{
  static index_t_s idx,cidx,iidx;

 /*
  *  Check the Disk Server names be locally known hosts
  */
    for (idx=0;idx<diskidx;idx++)
	if ( gethostbyname(disk_table[idx].host) == NULL ) {
		vperror(0,"unknown server host <%s>",disk_table[idx].host);
		return(-1);
	}

#if defined( CHKPHOST )

#if !defined( NOAUTHORIZE )
    if ( is_host_authorized(EXPORTAUTHORIZE,(char*)NULL) ) {
#endif

 /*
  *  Check the hosts in the pair table be locally known
  */
    idx = 0;
    while ( idx < pairidx ) {
	cidx = idx++;
	while ( idx < cidx + pair_table[cidx].next ) {
	    iidx = idx++;
	    while ( idx < iidx + pair_table[iidx].next ) {
		if ( gethostbyname(pair_table[idx].name) == NULL ) {
		    vperror(0,"unknown item host <%s>",pair_table[idx].name);
		    return(-1);
		}
		idx++;
	    }
	}
    }
#if !defined( NOAUTHORIZE )
    }

 /*
  *  Check the environment be locally known
  */
    idx = 0;
    for (idx=0; idx<hostidx; idx++)
	if ( gethostbyname(host_table[idx]) == NULL ) {
	    vperror(0,"unknown environment host <%s>",host_table[idx]);
	    return(-1);
	}
#endif

#endif

    return(1);
}

/*
 *  Generate the ascii configuration for the pool attributes
 */
void print_pool_attribute(fd,pool)
FILE   *fd;
pool_t *pool;
{
    if ( pool->private != defaults.private )
    		fprintf(fd,"%-10s",pool->private ? "PRIVATE" : "PUBLIC");
    if ( pool->permanent != defaults.permanent )
		fprintf(fd,"%-12s",pool->permanent ? "PERMANENT" : "TEMPORARY" );
    if ( pool->size != defaults.size )
		fprintf(fd,"SIZE(%s)   ",itoanmu(pool->size));
    if ( pool->min != defaults.min || pool->max != defaults.max )
		fprintf(fd,"COLLECT(%d:%d)",pool->min,pool->max);
}

/*
 *  Generate the shift ascii configuration
 */
void print_tables(fd_defaults,fd_pools,fd_users,fd_pairs)
FILE *fd_defaults,*fd_pools,*fd_users,*fd_pairs;
{
  FILE	     *fd;
  index_t_s  d_idx,f_idx,p_idx,a_idx,n_idx,i_idx,c_idx;
  int        gid;
  user_t     user;
  group_t    group;
  boolean    first_pool,first_group,first_user;
  boolean    group_garbage;
  int	     private;
  int	     permanent;
  int        collectidx;
  collect_t  *collect	= NULL ;
  int	     sizeidx;
  int	     *size	= NULL ;
  int	     *counter	= NULL ;
  int	     max_counter;
  int	     max_idx;
  int	     nsp;
  int	     access;

 /*
  *  Generate the DEFAULT pool attributes configuration
  */
    if ( (fd = fd_defaults) != NULL ) {

    fprintf(fd,"\n");
    private = 0;
    permanent = 0;
    for (p_idx=0;p_idx<poolidx;p_idx++) {
	private += pool_table[p_idx].private ? 1 : -1;
	permanent += pool_table[p_idx].permanent ? 1 : -1;
    }
    defaults.private = private > 0 ;
    defaults.permanent = permanent > 0 ;
    fprintf(fd,"%s\n",defaults.private ? "PRIVATE" : "PUBLIC");
    fprintf(fd,"%s\n",defaults.permanent ? "PERMANENT" : "TEMPORARY");
    if ( (collect = (collect_t*)malloc(sizeof(collect_t)*poolidx)) != NULL &&
		(counter = (int*)malloc(sizeof(int)*poolidx)) != NULL ) {
	collectidx = 0;
	for ( p_idx=0;p_idx<poolidx;p_idx++) {
	    for (c_idx=0;c_idx<collectidx;c_idx++)
		if ( collect[c_idx].min == pool_table[p_idx].min && 
				collect[c_idx].max == pool_table[p_idx].max )
		    break;
	    if ( c_idx < collectidx )
		counter[c_idx]++;
	    else {
		collect[collectidx].min = pool_table[p_idx].min;
		collect[collectidx].max = pool_table[p_idx].max;
		counter[collectidx] = 1;
		collectidx++;
	    }
	}
	for (c_idx=0;c_idx<collectidx;c_idx++)
	    if ( c_idx == 0  ||  counter[c_idx] > max_counter ) {
		max_counter = counter[c_idx];
		max_idx = c_idx;
	    }
	defaults.min = collect[max_idx].min;
	defaults.max = collect[max_idx].max;
	if ( collect[max_idx].min || collect[max_idx].max )
	    fprintf(fd,"COLLECT(%d:%d)\n",defaults.min,defaults.max);
    }
    if ( collect )
	free(collect);
    if ( counter ) {
	free(counter);
	counter = NULL;
    }
    if ( (size = (int*)malloc(sizeof(int)*poolidx)) != NULL &&
		(counter = (int*)malloc(sizeof(int)*poolidx)) != NULL ) {
	sizeidx = 0;
	for ( p_idx=0;p_idx<poolidx;p_idx++) {
	    for (c_idx=0;c_idx<sizeidx;c_idx++)
		if ( size[c_idx] == pool_table[p_idx].size )
		    break;
	    if ( c_idx < sizeidx )
		counter[c_idx]++;
	    else {
		size[sizeidx] = pool_table[p_idx].size;
		counter[sizeidx] = 1;
		sizeidx++;
	    }
	}
	for (c_idx=0;c_idx<sizeidx;c_idx++)
	    if ( c_idx == 0  ||  counter[c_idx] > max_counter ) {
		max_counter = counter[c_idx];
		max_idx = c_idx;
	    }
	defaults.size = size[max_idx];
	if ( size[max_idx] )
	    fprintf(fd,"SIZE(%s)\n",itoanmu(defaults.size));
    }
    if ( size )
	free(size);
    if ( counter )
	free(counter);
    fprintf(fd,"\nEND\n");

    } /* default pool attributes configuration */

 /*
  *  Generate the SERVER configuration
  */
    if ( (fd = fd_pools) != NULL ) {

    fprintf(fd,"\n");
    for (d_idx=0;d_idx<diskidx;d_idx++) {
	fprintf(fd,"SERVER  %-12s %s\n",disk_table[d_idx].host,disk_table[d_idx].mount);
	for (f_idx=0;f_idx<fsysidx;f_idx++)
	    if ( fsys_table[f_idx].diskidx == d_idx ) {
		fprintf(fd,"  %-12s",fsys_table[f_idx].name);
		if ( fsys_table[f_idx].bits & FSYSWLOCK )
		    fprintf(fd," WLOCK");
		fprintf(fd,"\n");
	    }
	fprintf(fd,"\n");
    }

 /*
  *  Generate the POOL configuration
  */
    fd = fd_pools;
    for (p_idx=0;p_idx<poolidx;p_idx++) {
	fprintf(fd,"POOL  %-15s ",pool_table[p_idx].name);
	print_pool_attribute(fd,&pool_table[p_idx]);
	fprintf(fd,"\n");
	for (f_idx=0;f_idx<fsysidx;f_idx++)
	    if ( fsys_table[f_idx].poolidx == p_idx ) {
		fprintf(fd,"  %-12s %-12s",
			disk_table[fsys_table[f_idx].diskidx].host,
			fsys_table[f_idx].name);
		if ( fsys_table[f_idx].bits & FSYSWLOCK )
		    fprintf(fd," WLOCK");
		fprintf(fd,"\n");
	    }
	fprintf(fd,"\n");
    }

 /*
  *  Generate the SUPER POOL configuration
  */
    p_idx = nsp = 0;
    while ( p_idx < supoolidx ) {
	if ( !nsp ) {
	    access = getsupoolaccess(p_idx);
	    fprintf(fd,"SUPERPOOL  %-12s %-5s  ",
		    supool_table[p_idx].name,
		    access==2?"":(access==1?"WRITE":"READ"));
	    nsp = supool_table[p_idx].idx ;
	}
	else {
	    fprintf(fd,"%s",supool_table[p_idx].name);
	    if ( --nsp )
		fprintf(fd,"%c",supool_table[p_idx].idx?':':'+');
	    else
		fprintf(fd,"\n\n");
	}
	p_idx++;
    }

    fprintf(fd,"END\n");

    } /* server and pool configuration */

 /*
  *  Generate the POOL users account
  */
    if ( (fd = fd_users) != NULL ) {

    fprintf(fd,"\n");
    first_pool = TRUE;
    for (a_idx=0;a_idx<accsidx;a_idx++) {
	if ( first_pool || accs_table[a_idx].poolidx != p_idx ) {
	    p_idx = accs_table[a_idx].poolidx;
	    fprintf(fd,"%sPOOL  %-15s",
			    first_pool ? "" : "\n\n",
			    pool_table[p_idx].name);
	    first_group = TRUE;
	    first_pool = FALSE;
	}
	if ( first_group || accs_table[a_idx].gid != gid ) {
	    first_group = FALSE;
	    gid = accs_table[a_idx].gid;
	    get_shiftgroupbyid(group,gid);
	    group_garbage = TRUE;
	    for (n_idx=a_idx;n_idx<accsidx;n_idx++)
		if ( accs_table[n_idx].gid != gid  ||
			!(group_garbage = (group_garbage && accs_table[n_idx].garbage)) )
		    break;
	    fprintf(fd,"\n  %s%s : ",
			    group,
			    group_garbage ? "(G)" : "");
	    first_user = TRUE;
	}
	get_shiftuserbyid(user,accs_table[a_idx].uid);
	fprintf(fd,"%s%s%s",
			first_user ? "" : ",",
			user,
			accs_table[a_idx].garbage && !group_garbage ? "(G)" : "");
	first_user = FALSE;
    }
    fprintf(fd,"\n\nEND\n");

    } /* pool users account configuration */

 /*
  *  Generate the PAIRS configuration
  */
    if ( (fd = fd_pairs) != NULL ) {

    fprintf(fd,"\n");
    p_idx = 0;
    while( p_idx < pairidx ) {
	fprintf(fd,"\ncategory  %s\n",pair_table[p_idx].name);
	c_idx = p_idx++;
	while ( p_idx < c_idx + pair_table[c_idx].next ) {
	    fprintf(fd,"  %-12s : ",pair_table[p_idx].name);
	    if ( pair_table[p_idx].value )
		fprintf(fd,"%s\n",pair_table[p_idx].value);
	    else
		fprintf(fd,"\n");
	    i_idx = p_idx++;
	    if ( pair_table[i_idx].next > 1 ) {
		fprintf(fd,"  (\n");
		while ( p_idx < i_idx + pair_table[i_idx].next ) {
#if !defined( NCHKPHOST ) && !defined( CHKPHOST )
		    if ( gethostbyname(pair_table[p_idx].name) == NULL )
			fprintf(fd,"    %s%-8s : ",UNKPHOST,pair_table[p_idx].name);
		    else
#endif
			fprintf(fd,"    %-10s : ",pair_table[p_idx].name);
		    if ( pair_table[p_idx].value )
			fprintf(fd,"%s\n",pair_table[p_idx].value);
		    else
			fprintf(fd,"\n");
		    p_idx++;
		}
		fprintf(fd,"  )\n");
	    }
	}
    }
    fprintf(fd,"\n");

    } /* pairs configuration */

}
