/*
 * Copyright (C) 1990,1991,1992 by CERN/CN/SW/DC
 * All rights reserved
 */

#ifndef lint
static char sccsid[] = "@(#)optutil.c	3.5 11/02/93 CERN-SW/DC Fabrizio Cane";
#endif /* not lint */

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/time.h>
#include <varargs.h>
#include <string.h>
#include <shift_types.h>
#include <strdup.h>
#include <strutil.h>
#include <strerror.h>
#include <optutil.h>
#if defined (SOLARIS)
#include <sys/filio.h>
#endif /* SOLARIS */

#define SPACE_MASK	0x4FFF

int     help_mode		= 0;
int	mask_mode		= 0;
int 	default_mask		= UNIX_MASK ;
char 	*mnemo_option_types[] 	= { "NONE", 
				    "CHAR",
				    "SHORT",
				    "INT",
				    "LONGI",
				    "STRING",
				    "DSTR" } ;
char	opt_error[256];

/*
 *
 */
int INIT_option_item(item)
option_item  *item;
{
    if ( INIT_string(&(item->name),0) < 0 )
	return( ERROR );

    if ( INIT_string(&(item->namenv),0) < 0 )
	return( ERROR );

    item->mask = 0;
    item->type = NONE;
    item->selected = 0;
    item->isvalue = 0;

    if ( INIT_string(&(item->value.s),0) < 0 )
	return( ERROR );

    if ( INIT_string(&(item->description),0) < 0 )
	return( ERROR );

    return( TRUE );
}

/*
 *
 */
int DUP_option_item(dst,src)
option_item  *dst,*src;
{
    memcpy((char*)dst,(char*)src,sizeof(option_item));

    dst->name.len = 0;
    dst->namenv.len = 0;
    dst->description.len = 0;
    if ( src->type == STRING || src->type == DSTR )
	dst->value.s.len = 0;

    if ( DUP_string(&dst->name,&src->name) < 0 )
	return( ERROR );

    if ( DUP_string(&dst->namenv,&src->namenv) < 0 )
	return( ERROR );

    if ( DUP_string(&dst->description,&src->description) < 0 )
	return( ERROR );

    if ( src->type == STRING || src->type == DSTR ) {
	if ( DUP_string(&dst->value.s,&src->value.s) < 0 )
	    return( ERROR );
    }

    return( TRUE );
}

/*
 *
 */
void FREE_option_item(item)
option_item  *item;
{
    FREE_string(&(item->name));
    FREE_string(&(item->namenv));
    FREE_string(&(item->description));
    if ( item->type == STRING  || item->type == DSTR )
    	FREE_string(&(item->value.s));
}

/*
 *
 */
char *VALUE_option_item(item)
option_item  *item;
{
    if ( (item->mask & OPTVAL_S) && item->type!=NONE && !item->isvalue )
	return( (char*)NULL );
    else
    switch ( item->type ) {

      case CHAR :
	return( (char*)&item->value.c );

      case SHORTI :
	return( (char*)&item->value.h );

      case INT :
	return( (char*)&item->value.i );

      case LONGI :
	return( (char*)&item->value.l );

      case DSTR :
      case STRING :
	return( GET_string(&item->value.s) );

      case NONE :
      default :
	return( (char*)NULL );

    } /* switch - type */
}

typedef struct {
	option_item	*item;
	int		size;
	int		max;
	int		no_opts;
	int		no_args;
	int		no_envs;
} option_list ;

#define STEP_OPTIONS_LEN	1
#define INIT_OPTIONS_LEN	STEP_OPTIONS_LEN

option_list  options__;
#define   OPTIONS_NAME		&options__

#define SELECT_option(opti)	(OPTIONS_NAME)->item[opti].selected
#define EXTSELECT_option(opti)	*(boolean*)(OPTIONS_NAME)->item[opti].extselected
#define MASK_option(opti)	(OPTIONS_NAME)->item[opti].mask

/*
 *
 */
void HELP_options()
{
  option_list  *list = OPTIONS_NAME;
  option_types  type;
  int		mask;
  char		*name,*namenv;
  int  	        k,len;

    if ( list->max > 0 )
        for ( k=0; k<list->size; k++ ) {

		mask = list->item[k].mask;

		if ( help_mode == 0 && mask & HIDDEN )	continue;

		name = GET_string(&list->item[k].name);
		namenv = GET_string(&list->item[k].namenv);
		type = list->item[k].type;

	      /*
	       *  Compute the length of the option name 
	       */
		len = strlen(name); 
		if ( (mask & SINGLE) || (mask & PREFIX) )
			len+=2;
		if ( mask & BLANK )
			len++;
		if ( (mask & OPTVAL_S) && type != NONE )
			len+=2;

	      /*
	       *  Print the option name in the following format :
	       *
	       *	-name		if the whole name must be supplied
	       *	-(n)ame		if only the first character must be supplied
	       *	-n(ame)		if at least a character must be supplied
	       *
	       *  followed by the option value type if provided 
	       *
	       *	<type>
	       *
	       *  The two above elements are separated by a blank character if
	       *  the blank must be used to separate the option value and name
	       */
		printf("  -");

		if ( mask & SINGLE )
			printf("(%c)%s",name[0],&name[1]);
		else
		if ( mask & PREFIX )
			printf("%c(%s)",name[0],&name[1]);
		else
			printf("%s",name);

		if ( mask & BLANK )
			printf(" ");

		if ( (mask & OPTVAL_S) && type != NONE)
			printf("[");

		switch ( type ) {

		  case DSTR :
		  case STRING :
			printf("%-8s","<string>");
			break;
		  case LONGI :
			printf("%-8s","<long>");
			break;
		  case INT :
			printf("%-8s","<int>");
			break;
		  case SHORTI :
			printf("%-8s","<short>");
			break;
		  case CHAR :
			printf("%-8s","<char>");
			break;
		  case NONE :
			printf("%-8s","");
			break;

		} /* switch - type */

		if ( (mask & OPTVAL_S) && type != NONE)
			printf("]");

	      /*
	       *  Align the cursor position to the 10-characters field length
	       *  used for the option name
	       */ 
		printf("%s",mkfield("",len > 14 ? 0 : 14-len));


	      /*
	       *  Print the environment variable name if provided
	       */
		if ( mask & ENVIR )
		    if ( mask & NAMENV )
			printf("%-14s  ",namenv);
		    else
			printf("%10s%6s","yes","");
		else
			printf("%16s","");

	      /*
	       *  Print the text description
	       */
		if ( LEN_string(&list->item[k].description) > 0 )
		    printf("%s",GET_string(&list->item[k].description));

		newline;
	}
}

/*
 *
 */
void LIST_options()
{
  option_list   *list = OPTIONS_NAME;
  int  		k;

    if ( list->max > 0 )

        for ( k=0; k<list->size; k++ )

	   if ( list->item[k].selected ) {

		printf("%13s is ",GET_string(&list->item[k].name));

		if ( list->item[k].type != NONE && (list->item[k].mask & OPTVAL_S) 
				&& !list->item[k].isvalue )
			printf("selected\n");
		else
		switch ( list->item[k].type ) {

		  case DSTR :
		  case STRING :
			printf("<%s>\n",GET_string(&list->item[k].value.s));
			break;
		  case LONGI :
			printf("%ld\n",list->item[k].value.l);
			break;
		  case INT :
			printf("%d\n",list->item[k].value.i);
			break;
		  case SHORTI :
			printf("%d\n",list->item[k].value.h);
			break;
		  case CHAR :
			printf("%c\n",list->item[k].value.c);
			break;
		  case NONE :
			printf("selected\n");
			break;

		} /* switch - type */
	   }
}

/*
 *
 */
int INIT_options()
{
  option_list  *list = OPTIONS_NAME;

    list->max = INIT_OPTIONS_LEN ;

    if ( list->max > 0 )
        if ( (list->item=(option_item*)malloc(sizeof(option_item)*list->max)) == NULL ) {
	    list->max = -1;
	    return( ERROR );
	}

    list->size = 0;

    return( TRUE );
}

/*
 *
 */
void FREE_options()
{
  option_list  *list = OPTIONS_NAME;
  int  		   k;

    if ( list->max > 0 ) {

        for ( k=0; k<list->size; k++ )
	    FREE_option_item(&list->item[k]);

	list->max = -1;
	free(list->item);
    }
}

/*
 *
 */
static void ERR_ambiguous(name,opti1,opti2)
char *name;
int  opti1,opti2;
{
  static option_list  *list 	= OPTIONS_NAME;

    vperror(0,"%s option ambiguous: %s %s",
		name,
		GET_string(&list->item[opti1].name),
		GET_string(&list->item[opti2].name));
}

static void ERR_unknown(name)
char *name;
{
    vperror(0,"unknown %s option",name);
}

static void ERR_novalue(name)
char *name;
{
    vperror(0,"no value for -%s option",name);
}

static void ERR_internal(name,msg)
char  *name;
char  *msg;
{
    vperror(0,"internal error - option %s: %s",name,msg);
}

static void ERR_extrargument(argc,argv,argi)
int  argc;
char *argv[];
int  argi;
{
  string emsg;

    INIT_string(&emsg,0);
    while( argi < argc ) {
	CAT_string(&emsg," ");
        CAT_string(&emsg,argv[argi++]);
    }
    vperror(0,"extra argument:%s",GET_string(&emsg));
}

static void ERR_dashargument(name)
char  *name;
{
    vperror(0,"option %s in the argument list",name);
}

static void ERR_optargument(name,opti)
char  *name;
int   opti;
{
  static option_list  *list 	= OPTIONS_NAME;

    vperror(0,"argument %s matches option %s",name,GET_string(&list->item[opti].name));
}

static void ERR_doublematch(name,opti)
char  *name;
int   opti;
{
  static option_list  *list 	= OPTIONS_NAME;

    vperror(0,"argument %s matches an option (%s) already assigned",
		name,GET_string(&list->item[opti].name));
}

static void ERR_boolenv(oname,ename)
char  *oname;
char  *ename;
{
    vperror(0,"%s option: environment variable %s requires no value",oname,ename);
}

static void ERR_missenv(oname,ename)
char  *oname;
char  *ename;
{
    vperror(0,"%s option: missing value for the environment variable %s",oname,ename);
}

/*
 *
 */
static int VERIFY_mask(name,mask)
char *name;
int  mask;
{
  int  k;

    if ( ((mask & (mask>>0x10)) | (~mask & ~(mask>>0x10))) & SPACE_MASK ) {
	ERR_internal(name,"incompatible flag and its opposite"); 
	return( ERROR );
    }

    if ( mask & PREFIX  &&  mask & NOBLANK ) {
	ERR_internal(name,"incompatible PREFIX and NOBLANK flags");
	return( ERROR );
    }

    if ( mask & PREFIX  &&  mask & SINGLE ) {
	ERR_internal(name,"incompatible PREFIX and SINGLE flags");
	return( ERROR );
    }

    return( TRUE );
}

/*
 *
 */
int DEFAULT_mask(mask)
int  mask;
{
  int p_mask = default_mask;

    default_mask = default_mask | (mask & 0xFFFF);
    default_mask = default_mask & ~(((unsigned)mask & 0xFFFF0000) >> 0x10);
    default_mask = default_mask | (mask & 0xFFFF0000);
    default_mask = default_mask & ~((mask & 0xFFFF) << 0x10);

    if ( VERIFY_mask("",default_mask) < 0 )
	return( ERROR );

    return( p_mask );
}

/*
 *  INPUT
 *
 *	variable list of parameters ( see to_option_item() )
 *
 *	- option name
 *	- option mask
 *	- option value type
 *	- environment variable name ( if ENVIR is defined in the mask )
 *	- address of an external variable that must be set if the option will be
 *	  selected ( if EXTSEL is defined in the mask )
 *	- address of an external variable where the option value must be stored
 *	  ( if EXTVAL is defined in the mask )
 *	- text description of the option ( if DESCR is defined in the mask )
 *
 */
static option_item *to_option_item(args)
va_list  args;
{
  static option_item	ptr;
  static boolean	init	= TRUE;
  int			mask;

    if ( !init )
	FREE_option_item(&ptr);
    else
	init = FALSE;

    INIT_option_item(&ptr);

    CPY_string(&ptr.name,va_arg(args,char*));		/* OPTION NAME      */

    mask = va_arg(args,int);				/* OPTION MASK      */
    ptr.mask = default_mask;
    ptr.mask = ptr.mask | (mask & 0xFFFF);
    ptr.mask = ptr.mask & ~(((unsigned)mask & 0xFFFF0000) >> 0x10);
    ptr.mask = ptr.mask | (mask & 0xFFFF0000);
    ptr.mask = ptr.mask & ~((mask & 0xFFFF) << 0x10);

    ptr.type = va_arg(args,int);			 /* OPTION TYPE      */

    if ( ptr.mask & ENVIR  && ptr.mask & NAMENV )
	CPY_string(&ptr.namenv,va_arg(args,char*));	 /* ENVIRONMENT NAME */

    ptr.selected = FALSE;
    if ( ptr.mask & EXTSEL ) {
	ptr.extselected = va_arg(args,char*);		 /* EXTERNAL SELECT  */
	*(boolean*)ptr.extselected = FALSE;
    }

    if ( ptr.mask & EXTVAL )
	ptr.extvalue = va_arg(args,char*);		 /* EXTERNAL VALUE   */

    if ( ptr.mask & DESCR )
	CPY_string(&ptr.description,va_arg(args,char*)); /* TEXT DESCRIPTION */

    if ( VERIFY_mask(GET_string(&ptr.name),ptr.mask) < 0 )
	return( (option_item*)NULL );

    return( &ptr );
}

/*
 *  Add a new option in the option list
 *
 *  INPUT
 *
 *	variable list of parameters ( see to_option_item() )
 *
 *	
 *  RETURN
 *
 *	ERROR		it couldn't add the option
 *	TRUE		option added
 *
 */
int ADD_options(va_alist)
  va_dcl
{
  option_list	*list 		= OPTIONS_NAME;
  option_item	*ptr,*item;
  va_list	args;

    if ( list->max < 0 )
	return( ERROR );

    va_start(args);

    if ( (item = to_option_item(args)) == NULL ) {
	va_end(args);
	return( ERROR );
    }

    va_end(args);

    if ( list->size >= list->max ) {
	if ( list->max == 0 ) {
	    list->max += STEP_OPTIONS_LEN ;
	    if ( (list->item=(option_item*)malloc(sizeof(option_item)*list->max)) == NULL ) {
	        list->max = -1;
	        return( ERROR );
	    }
	}
	else {
	    list->max += STEP_OPTIONS_LEN ;
	    if ( !(ptr=(option_item*)realloc(list->item,sizeof(option_item)*list->max)) ) {
	        list->max -= STEP_OPTIONS_LEN ;
	        FREE_options(list->item);
	        list->max = -1;
	        return( ERROR );
	    }
	    list->item = ptr;
	}
    }

    DUP_option_item(&list->item[list->size++],item);

    return( TRUE );
}

/*
 *  Look for the given option in the option list structure build by ADD_option
 *
 *  INPUT
 *
 *	name		option name previously added via ADD_option() call
 *
 *  RETURN
 *
 *	address		address of the option structure
 *	NULL		the option is not in the list build by ADD_option()
 *
 */
static option_item *GET_options(name)
char	  *name;
{
  option_list   *list 	= OPTIONS_NAME;
  int		k;

    for ( k=0; k<list->size; k++ ) {
	if ( !strcmp(name,GET_string(&(list->item[k].name))) )
		return( &(list->item[k]) );
    }

    fprintf(stderr,"internal error (%s): option %s not defined\n",__FILE__,name);
    exit(1);
}

/*
 *  Check if the current command line argument matches the current option
 *
 *  INPUT
 *
 *      opti		the index in the option list of the current option
 *	str		the current command line argument
 *
 *  OUTPUT
 *
 *	value		the option value if provided into the argument itself
 *
 *  RETURN
 *
 *	TRUE		the command line argument 'str' matches the option 'opti'
 *      FALSE		the command line argument 'str' doesn't match the option 'opti'
 *
 */
static int MATCH_option(opti,str,value)
int     opti;
char    *str;
string  *value;
{
  option_list  *list	= OPTIONS_NAME;
  option_types  type	= list->item[opti].type;
  int		mask	= list->item[opti].mask;
  char		*name	= GET_string(&list->item[opti].name);

    CPY_string(value,"");

    if ( mask & SINGLE ) {
	if ( (int)strlen(str) > 1 ) {
	    if ( mask & BLANK  ||  type == NONE )
		return( FALSE );
	    CPY_string(value,str+1);
	}
	if ( mask & CASESN )
	    return( str[0] == name[0] );
	else
	    return( uppercase(str[0]) == uppercase(name[0]) );
    }
    else {
	if ( mask & CASESN )
	    if ( mask & PREFIX )
		return( !strncmp(name,str,strlen(str)) );
	    else
		if ( mask & BLANK || type == NONE )
		    return( !strcmp(name,str) );
		else {
		    CPY_string(value,str+strlen(name));
		    return( !strncmp(name,str,strlen(name)) );
		}
	else
	    if ( mask & PREFIX )
		return( !strncasecmp(name,str,strlen(str)) );
	    else
		if ( mask & BLANK || type == NONE )
		    return( !strcasecmp(name,str) );
		else {
		    CPY_string(value,str+strlen(name));
		    return( !strncasecmp(name,str,strlen(name)) );
		}
    }

}

/*
 *  Check the correctness of the option value and store it in both the
 *  extern and intern variable
 *
 *  INPUT
 *
 *	opti		index of the option 
 *	value		option value in string format
 *
 *  RETURN
 *
 *	TRUE		ok
 *	ERROR		the value doesn't match the declared type
 *
 */
static int CONVERT_option(opti,value)
int     opti;
string  *value;
{
  option_list	*list 	= OPTIONS_NAME;
  option_item	*item	= &list->item[opti];
  option_types  type	= item->type;
  int		mask	= item->mask;
  char		*name	= GET_string(&item->name);
  long		inum;

    switch ( type ) {

      case DSTR :
      case STRING :
	if ( mask & EXTVAL )
	    if ( mask & ALLOC )
		if ( type == STRING )
		    *(char**)item->extvalue = strdup(GET_string(value));
		else {
		    INIT_string((string*)item->extvalue,0);
		    CPY_string((string*)item->extvalue,GET_string(value));
		}
	    else
		if ( type == STRING )
			strcpy((char*)item->extvalue,GET_string(value));
		else
			CPY_string((string*)item->extvalue,GET_string(value));
	CPY_string(&item->value.s,GET_string(value));
	break;


      case INT :
      case SHORTI :
      case LONGI :
	if ( atoinum(GET_string(value),&inum) < 0 ) {
		vperror(0,"-%s option: integer value",name);
		return( ERROR );
	}
	if ( type == INT ) {
	    if ( mask & EXTVAL )
	        *(int*)item->extvalue = inum;
	    item->value.i = inum;
	}
	else
	if ( type == SHORTI ) {
	    if ( mask & EXTVAL )
	        *(short*)item->extvalue = (short)inum;
	    item->value.h = inum;
	}
	else {
	    if ( mask & EXTVAL )
	        *(long*)item->extvalue = (long)inum;
	    item->value.l = inum;
	}
	break;


      case CHAR :
	if ( (int)strlen(GET_string(value)) > 1 ) {
		vperror(0,"-%s option: one single character value",name);
		return( ERROR );
	}
	if ( mask & EXTVAL )
	   *(char*)item->extvalue = *GET_string(value);
	item->value.c = *GET_string(value);
	break;


      case NONE :
      default : ;

    } /* switch - value type */

    return( TRUE );
}

/*
 *  Get the value of the option from the command line
 *
 *  INPUT
 *
 *	argc		the argument counter passed to the main
 *	argv		the argument list passed to the main
 *	argi		the index of the current argument		
 *	opti		the index in the option list structure
 *      value		string possibly containing the option value
 *
 *  OUTPUT
 *
 *	skip		
 *
 *  RETURN
 *
 *	TRUE		ok
 *	ERROR		value not found - the value doesn't match the declared type
 *
 */
static int VALUE_option(argc,argv,argi,opti,value,skip)
int     argc;
char    *argv[];
int	argi;
int     opti;
string  *value;
int	*skip;
{
  option_list	*list 	= OPTIONS_NAME;
  option_item   *item	= &list->item[opti];

    item->isvalue = TRUE;

    if ( item->type == NONE ) {
	*skip = 0;
	return( TRUE );
    }

    if ( (int)strlen(GET_string(value)) > 0 ) {
	*skip = 0;
	return( CONVERT_option(opti,value) );
    }
    else
	if ( argi+1 < argc ) {
	    if ( (item->mask & OPTVAL_S) && argv[argi+1][0] == '-' ) {
		item->isvalue = FALSE;
		return( TRUE );
	    }
	    *skip = 1;
	    CPY_string(value,argv[argi+1]);
	    return( CONVERT_option(opti,value) );
	}
	else
	if ( item->mask & OPTVAL_S ) {
	    item->isvalue = FALSE;
	    return( TRUE );
	}
	else {
	    ERR_novalue(GET_string(&item->name));
	    return( ERROR );
	}
}

/*
 *  Check if the given option could have its value defined in the environments
 *
 *  INPUT
 *
 *	opti		index in the option list structure
 *
 *  RETURN
 *
 *	TRUE		the value was found
 *	FALSE		the option hasn't a value in the environment or it was
 *			already defined in the command line
 *	ERROR		the environment value doesn't math the declared type
 *
 */
static int ENVIR_option(opti)
int  opti;
{
  option_list  	*list 		= OPTIONS_NAME;
  option_item	*item		= &list->item[opti];
  char		*envalue;
  char		*envname;
  string	value;

    INIT_string(&value,0);

    if ( item->mask & NOENVIR )
	return( FALSE );

    if ( (item->mask & NOPRENV)  &&  item->selected )
	return( FALSE );

    envname = item->mask & NONAMENV ? GET_string(&item->name):GET_string(&item->namenv) ;

    envalue = (char*)getenv(envname);

    if ( envalue == NULL )
	return( FALSE );

    if ( item->type == NONE )
	if ( (int)strlen(envalue) > 0 ) {
	    ERR_boolenv(GET_string(&item->name),envname);
	    return( ERROR );
	}
	else
	    return( TRUE );
    else
	if ( strlen(envalue) == 0 ) {
	    ERR_missenv(GET_string(&item->name),envname);
	    return( ERROR );
	}
	else {
	    CPY_string(&value,envalue);
	    return( CONVERT_option(opti,&value) );
	}

}

/*
 *  Analyse the command line and the environment in order to get the option values
 *
 *  INPUT
 *
 *	argc		the argument counter passed to the main
 *	argv		the argument list passed to the main
 *
 */
int READ_options(argc,argv)
int   argc;
char  *argv[];
{
  option_list	*list 	= OPTIONS_NAME;
  int  		argi;
  int		opti;
  string	value;
  string	savevalue;
  int		skip;

    INIT_string(&value,0);
    INIT_string(&savevalue,0);

/*
 *  Analyse the command line options
 */

    skip = 0;

    list->no_opts = 0;

    for ( argi=1; argi<argc; argi++ ) {

	int  matched,doublematch,errormatch;

	if ( skip ) {
	    skip -- ;
	    continue;
	}

	if ( argv[argi][0] != '-' )   break;

	if ( !strcmp(argv[argi],"--") ) {
	    argi++;
	    break;
	}

	matched = 0;
	doublematch = 0;
	errormatch = 0;

	for ( opti=0; opti<list->size; opti++ ) {

	    if ( MATCH_option(opti,argv[argi]+1,&value) )
		if ( SELECT_option(opti) )
			doublematch = opti+1;
		else
		    if ( matched > 0 ) {
		        ERR_ambiguous(argv[argi],matched-1,opti);
		        return( ERROR );
		    }
		    else {
			ctrl_vperror(VPDISABLE);
		        if ( VALUE_option(argc,argv,argi,opti,&value,&skip) < 0 ) {
			    errormatch = opti+1;
			    CPY_string(&savevalue,GET_string(&value));
			}
		        else
			    matched = opti+1;
			ctrl_vperror(VPENABLE);
		    }

	} /* for - scan option list */

	if ( !matched ) 
	    if ( doublematch ) {
		ERR_doublematch(argv[argi],doublematch-1);
		return( ERROR );
	    }
	    else
	    if ( errormatch ) {
		VALUE_option(argc,argv,argi,errormatch-1,&savevalue,&skip);
		return( ERROR );
	    }
	    else {
		ERR_unknown(argv[argi]);
		return( ERROR );
	    }

	list->no_opts ++ ;
	SELECT_option(matched-1) = TRUE;
	if ( MASK_option(matched-1) & EXTSEL )
		EXTSELECT_option(matched-1) = TRUE;

    } /* for - scan command line options */


/*
 *  Analise the last part of the command line that doesn't contains options
 */

    list->no_args = argc - argi;


    if ( default_mask & NOFILIST  &&  list->no_args > 0 ) {
	ERR_extrargument(argc,argv,argi);
	return( ERROR );
    }

    if ( default_mask & NODSLIST  ||  default_mask & NONMLIST )

        while ( argi < argc ) {

	    if ( default_mask & NODSLIST  &&  argv[argi][0]=='-' ) {
		ERR_dashargument(argv[argi]);
		return( ERROR );
	    }

	    if ( default_mask & NONMLIST &&  argv[argi][0]=='-' )

		for ( opti=0; opti<list->size; opti++ )
		    if ( MATCH_option(opti,argv[argi]+1,&value) ) {
			ERR_optargument(argv[argi],opti);
			return( ERROR );
		    }

	    argi ++ ;
        }


/*
 *  Analise the environments
 */

    list->no_envs = 0;

    for ( opti=0; opti<list->size; opti++ )

	switch ( ENVIR_option(opti) ) {

	    case ERROR : 
		return( ERROR );

	    case FALSE :
		break;

	    case TRUE :
		list->no_envs++;
		SELECT_option(opti) = TRUE;
		if ( MASK_option(opti) & EXTSEL )
		    EXTSELECT_option(opti) = TRUE;
		break;

	} /* switch */


    return( TRUE );
}

/*
 *  INPUT
 *
 *	name		option name previously added via ADD_option() call
 *
 *  RETURN
 *
 *	TRUE		if the option was selected ( after a READ_option() call )
 *	FALSE		if the option wasn't selected
 *	ERROR		the option is not in the list build by ADD_option()
 *
 */
int IS_option(name)
char *name;
{
  option_item  *ptr;

    if ( (ptr=GET_options(name)) == NULL )
	return( ERROR );

    return( ptr->selected );
}

/*
 *  INPUT
 *
 *	name		option name previously added via ADD_option() call
 *
 *  RETURN
 *
 *	address		address of the internal option value
 *	NULL		the option is not in the list build by ADD_option()
 *
 */
char *VAL_option(name)
char *name;
{
  option_item  *ptr;

    if ( (ptr=GET_options(name)) == NULL )
	return( (char*)NULL );

    return( VALUE_option_item(ptr) );
}

/*
 *  NO_opts returns the number of options specified in the command line
 *
 *  NO_args returns the number of arguments in the command line listed after
 *  the last option ( argc - NO_args is the index in the command line of the first
 *  of this arguments )
 *
 *  NO_envs returns the number of options specified in the environment
 *
 */
int NO_opts()
{
    return( (OPTIONS_NAME)->no_opts );
}

int NO_args()
{
    return( (OPTIONS_NAME)->no_args );
}

int NO_envs()
{
    return( (OPTIONS_NAME)->no_envs );
}

/*
 *
 */
#define OPTION_DESCRIPTION  "show the options value"
#define HELP_DESCRIPTION    "get this help"

int ADD_OPT_options(show_list,show_value)
boolean  *show_list;
boolean  *show_value;
{
  long  previous_mask;
  int   reply;

    previous_mask = DEFAULT_mask(0);

    if ( previous_mask & SINGLE )
        DEFAULT_mask(NOSINGLE);
    else
        DEFAULT_mask(VMS_MASK);

    ADD_options("options",EXTSEL | DESCR,NONE,show_value,OPTION_DESCRIPTION);
    reply = ADD_options("help",EXTSEL | DESCR,NONE,show_list,HELP_DESCRIPTION);

    DEFAULT_mask(previous_mask);

    return( reply );
}

