/*
 * Copyright (C) 1990,1991,1992 by CERN/CN/SW/DC
 * All rights reserved
 */

#ifndef lint
static char sccsid[] = "@(#)sockutil.c	3.7 06/24/94  CERN-SW/DC Fabrizio Cane";
#endif /* not lint */

#include <stdio.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include <netdb.h>
#include <varargs.h>
#include <errno.h>
#include <shift_types.h>
#include <strerror.h>
#include <sockutil.h>
#if defined(SOLARIS)
#include <sys/filio.h>
#endif

static select_timeout_t select_timeout;

/*
 *  INPUT
 *      host            host name
 *      port            port number
 *      name            pointer to a sockaddr_in structure
 *
 *  OUTPUT
 *      name            the sockaddr_in structure contains the given address
 *
 *  RETURN
 *      OK              successful completion
 *      ERROR           gethostbyname() failed
 */
int create_sockaddr_in(host,port,name)
char *host;
port_t port;
struct sockaddr_in *name;
{
  static struct hostent  *host_ent;

 /*
  *  Look for the given host name
  */
    if ( (host_ent = gethostbyname(host)) == NULL ) {
        vperror(1,"gethostbyname(%s)",host);
        return(ERROR);
    }

 /*
  *  Set the sockaddr_in structure ( host address - port number - family )
  */
    memset((char*)name,(int)0,(size_t)sizeof(struct sockaddr_in));
    name->sin_addr.s_addr = ((struct in_addr *)(host_ent->h_addr))->s_addr;
    name->sin_port = htons((u_short)port);
    name->sin_family = AF_INET;

    return(OK);
}

/*
 *  INPUT
 *      addr            pointer to a sockaddr_in structure
 *
 *  OUTPUT
 *      host            host name contained in the sockaddr_in structure
 *      port            port number contained in the sockaddr_in structure
 *
 *  RETURN
 *      OK              successful completion
 *      ERROR 		gethostbyaddr() failed
 */
int get_sockaddr_in(addr,host,port)
struct sockaddr_in *addr;
char *host;
port_t *port;
{
  static struct hostent *host_ent;

 /*
  *  Get the port number
  */
    *port = ntohs(addr->sin_port);

 /*
  *  Look for the host address contained in the sockaddr_in structure
  */
    if ( (host_ent = gethostbyaddr((char*)&(addr->sin_addr),
                                   sizeof(struct in_addr),
                                   (int)addr->sin_family)) == NULL ) {
        vperror(1,"gethostbyaddr(%s)",(char*)&(addr->sin_addr));
        return(ERROR);
    }

 /*
  *  Get the host name
  */
    strcpy(host,host_ent->h_name);

    return(OK);
}

/*
 *  Return the select's timeout previousely set through set_timeout()
 */
struct timeval *get_timeout()
{
    switch ( select_timeout.mode ) {
        case TIME :
        case POLLING :
                return(&select_timeout.tv);
        case BLOCKING :
                return((struct timeval*)NULL);
        default :
                vperror(0,"unknown timeout mode (%d)",select_timeout.mode);
                exit(1);
    } /* switch - timeout mode */
}

/*
 *  Set the select's timeout
 *
 *  INPUT
 *      mode            can be one of the following:
 *                       - POLLING      to set the timeout to zero
 *                       - BLOCKING     to set the timeout to infinite
 *                       - TIME         to set the timeout to a specific value
 *      sec             seconds (required only if mode is TIME)
 *      usec            millisenconds (required only if mode is TIME)
 */
void set_timeout(va_alist)
  va_dcl
{
  static va_list args;

    va_start(args);
    select_timeout.mode = va_arg(args,int);
    switch ( select_timeout.mode ) {
        case TIME :
                select_timeout.tv.tv_sec = va_arg(args,int);
                select_timeout.tv.tv_usec = va_arg(args,int);
                break;
        case POLLING :
                select_timeout.tv.tv_sec = 0;
                select_timeout.tv.tv_usec = 0;
                break;
        case BLOCKING :
                break;
        default :
                vperror(0,"unknown timeout mode (%d)",select_timeout.mode);
                exit(1);
    } /* switch - timeout mode */
    va_end(args);
}

/*
 *  Connect to a peer address
 *
 *  INPUT
 *	sock		socket descriptor (if not valid a new socket will be created)
 *	host		peer address host
 *	port		peer address port number
 *	nonblocking	(1)non-blocking/(0)blocking connect (NOT provided for Apollo)
 *			
 *  RETURN
 *	ERROR		an error occured
 * 	>= 0		socket descriptor given in input or created
 *
 *  ERRORS
 *			socket() errors
 *			ioctl()
 */
int connect_to(sock,host,port,nonblocking)
int sock;
char *host;
port_t port;
boolean nonblocking;
{
  struct sockaddr_in name;

    if ( sock < 0  && (sock = socket(AF_INET,SOCK_STREAM,0)) < 0 ) {
        vperror(1,"socket()");
        return(ERROR);
    }

#if !defined( apollo ) && !defined( _IBMESA )

    if ( ioctl(sock,(int)FIONBIO,&nonblocking) < 0 ) {
	vperror(1,"ioctl()");
	close(sock);
	return(ERROR);
    }

#endif

    if ( create_sockaddr_in(host,port,&name) < 0 ) {
	close(sock);
	return(ERROR);
    }

    if ( connect(sock,(struct sockaddr *)&name,sizeof(struct sockaddr_in)) < 0 && errno != EINPROGRESS ) {
	vperror(1,"connect()");
	close(sock);
	return(ERROR);
    }

    return(sock);
}

/*
 *  Compare two host names
 *
 *  INPUT
 *	name		first host name
 *	hostname	second host name
 *
 *  OUTPUT
 *	official	official name of hostname if not itself
 *
 *  RETURN
 *	ERROR		hostname is unknown ( gethostbyname() failed )
 *	0		different host names
 *	1		host names match
 */
int equal_hosts(name,hostname,official)
char *name,*hostname;
char **official;
{
  static struct hostent *host;
  static int idx;
#if !defined( NOHOSTDOT )
  static int dindex;
#endif

    if ( (host = gethostbyname(hostname)) == NULL ) {
	vperror(0,"gethostbyname(%s): not found",hostname);
	return(ERROR);
    }

#if defined( NOHOSTDOT )
    if ( !strcmp(name,host->h_name) ) {
#else
    dindex = strchr(name,'.') ? (char*)strchr(name,'.')-(char*)name : strlen(name);
    if ( !strncmp(name,host->h_name,dindex) && (
			host->h_name[dindex] == '.' || 
			host->h_name[dindex] == '\0' ) ) {
#endif
	if ( official != NULL ) *official = NULL;
	return(1);
    }

    idx = 0;
    while( host->h_aliases[idx] != NULL ) {
#if defined( NOHOSTDOT )
	if ( !strcmp(name,host->h_aliases[idx]) ) {
#else
	if ( !strncmp(name,host->h_aliases[idx],dindex) && (
				host->h_aliases[idx][dindex] == '.' || 
				host->h_aliases[idx][dindex] == '\0' ) ) {
#endif
	    if ( official != NULL ) *official = host->h_name;
#if !defined( NOHOSTDOT )
	    if ( strchr(host->h_name,'.') ) *(char*)strchr(host->h_name,'.') = '\0';
#endif
	    return(1);
	}
	idx++;
    }

    return(0);
}
