/*
 * Copyright (C) 1990-1998 by CERN/CN/SW/DC
 * All rights reserved
 */

#ifndef lint
static char sccsid[] = "@(#)msg.c	1.14 05/06/98  CERN CN-SW/DC Antoine Trannoy";
#endif /* not lint */

/* msg.c               SHIFT message library.		   */

#include <stdio.h>
#if defined(sun) || defined(ultrix) || defined(sgi)
#include <unistd.h>
#endif	/* sun || ultrix || sgi */
#include "msg.h"

#include <sys/types.h>
#if defined(_WIN32)
#define MAXHOSTNAMELEN 64
#else
#include <sys/param.h>
#endif
#include <sys/stat.h>
#if defined(apollo)
#include <strings.h> 
#else
#include <string.h>
#endif	/* apollo */

/*
 * Networking.
 */
#if defined(_WIN32)
#include <winsock2.h>
#else
#include <netdb.h>
#include <sys/socket.h>
#include <netinet/in.h>
#endif
#include <marshall.h>

#define netread			dorecv
#define netwrite		dosend

#if defined(sun)
#include <sys/termios.h>
#endif
#if defined(sgi) || defined(ultrix)
#include <sys/termio.h>
#endif
/* 
 * Error and message handling.
 */
#include <errno.h>

#include <serrno.h>
#include <log.h>
extern int errno ; 
extern int sys_nerr ;
extern void perror() ; 
#if !defined(linux)
extern char *sys_errlist[] ;
#endif

/*
 * External definition missing in include files.
 */
#if defined(CRAY)
extern void exit() ; 
extern unsigned sleep() ; 
#endif

/*
 * Getting connected to the msg daemon.
 */
int conmsgd()
{
	static char msgdhost[MAXHOSTNAMELEN]="";/* Message Daemon host name	*/
	struct servent 		*sp ; 		/* Service entry pointer.	*/
	struct hostent		*hp ;		/* Host entry pointer.		*/
	struct sockaddr_in	sin ;		/* An Internet socket address.	*/ 
	int 		       sock ; 		/* Socket descriptor.		*/
	extern char      * getenv() ; 		/* Getting env variables	*/
	char 		      * env ;		/* To store env variables 	*/

	/*
	 * Creating socket.
	 */
	if (( sock= socket(AF_INET,SOCK_STREAM,0)) == -1 ) {
		return -1 ;
	}

	/*
	 * Getting message daemon hostname.
	 * First check if specified in an environment variable.
	 * If not, get it from the configuration file.
	 */
	if ( !strcmp(msgdhost,"") ) {
		if ( (env=getenv("MSGHOST"))==NULL) {
			extern char * getconfent() ;
			char 		     * str ;

			if ( (str= getconfent("MSG","HOST",0)) == NULL ) {
				close(sock) ;
				return -1 ;
			}
			(void) strcpy(msgdhost,str) ;
		}		
		else	{
			(void) strcpy(msgdhost,env) ;
		}
	}
	if ((hp= gethostbyname(msgdhost)) == NULL ) {
		serrno= SENOSHOST ;
		close(sock) ;
		return -1 ; 
	}
	
	/*
	 * Building Daemon Internet address.
	 * As for the message daemon hostname, try to find it
	 * in an environment variable before getting it from the 
	 * configuration file.
	 */
	if ( (env=getenv("MSGPORT"))==NULL) {
		if ((sp= getservbyname("msg","tcp")) == NULL) {
			serrno= SENOSSERV ;
			close(sock) ;
			return -1 ; 
		}
		sin.sin_port=  sp->s_port ;
	}
	else	{
		sin.sin_port= htons(atoi(env)) ;
	}
	sin.sin_family= AF_INET ;
	sin.sin_addr.s_addr= ((struct in_addr *)(hp->h_addr))->s_addr ;

	/*
	 * Connecting the socket.
	 */
	if ( connect(sock, (struct sockaddr *) &sin, sizeof(sin))  == -1 ) {
		close(sock) ;
		return -1 ;
	}
	return sock ; 
}

/*
 * Receive data in buffer from sock.
 * The data type is returned.
 */
int rcvmsgd(sock,buffer) 
	int      sock ; 
	char * buffer ;
{
	int  magic ;
	int   code ;
	int   size ; 
	char * ptr ;

	/*
	 * Getting header message.
	 */
	switch( netread(sock,buffer,LONGSIZE*3) ) { 
		case  0:
			serrno= SECONNDROP ;
		case -1:
			return -1 ;
	}
	ptr= buffer ; 
	unmarshall_LONG(ptr,magic) ; 
	unmarshall_LONG(ptr, code) ; 
	unmarshall_LONG(ptr, size) ; 
	if ( magic != MSG_MAGIC ) {
		serrno= SEBADVERSION ; 
		return -1 ; 
	}

	/*
	 * Returns, if no body, the message type.
	 */
	if ( size == 0 ) 
		return code ; 

	/* 
	 * Getting message body.
	 */
	switch( netread(sock,buffer,size) ) {
		case  0:
			serrno= SECONNDROP ;
		case -1:
			return -1 ;
	}

	/*
	 * No problem,
	 * sending back message type.
	 */
	return code ; 
}

/*
 * Opening socket to receive reply from the
 * message daemon.
 * Replies are sent on a special socket to keep
 * a certain interface compatibility with the Cray 
 * message daemon ( using named pipes ).
 */
int opnmsgr(port)
	int * port ;
{
	int 		         sock ;	/* Socket identifier.		*/
	char hostname[MAXHOSTNAMELEN] ;	/* Host name.			*/
	struct hostent         * hent ;	/* Host entity			*/
	struct sockaddr_in        sin ;	/* Internet address		*/
	int       sinlen= sizeof(sin) ;	/* Size of an Internet address	*/

	serrno= 0 ;
	/*
	 * Creating socket.
	 */
	if ( (sock= socket(AF_INET, SOCK_STREAM,0)) == -1 ) {
		return -1 ;
	}

	/*
	 * Getting info on host.
	 */
	if ( gethostname(hostname,MAXHOSTNAMELEN) == -1 ) {
		return -1 ;
	}
	if ( (hent= gethostbyname(hostname)) == NULL ) {
		serrno= SENOSHOST ;
		return -1 ;
	}

	/*
	 * Binding socket.
	 */
	memset((char *)&sin,0,sizeof( struct sockaddr_in)) ;
	sin.sin_addr.s_addr= (( struct in_addr *) hent->h_addr)->s_addr ;
	sin.sin_port= htons(*port) ;
	sin.sin_family= AF_INET ;
	if ( bind(sock, (struct sockaddr *)&sin,sizeof(sin)) == -1 ) {
		return -1 ;
	}

	/*
	 * Getting port number. 
	 */
	if ( getsockname(sock,(struct sockaddr *)&sin,&sinlen) == -1 ) {
		return -1 ;
	}
	*port= ntohs(sin.sin_port) ;

	/*
	 * Listening on socket.
	 */
	listen(sock,1) ;
	
	/*
	 * Returning socket identifier.
	 */
	return sock ;
}

/* 
 * Sending MSGI request.
 */
int sndmsgi(msg) 
	char * msg ; 
{
	char buffer[MSGSIZ+4*LONGSIZE];	/* Buffer		*/
	char 	* ptr ;			/* Pointer to the buffer*/
	int   	 sock ;			/* Socket identifier	*/
	int	 size ;			/* Size of the message	*/
	int     rcode ;			/* Return code 		*/

	/*
	 * Connecting the daemon.
	 */
	if ( (sock= conmsgd()) == -1 ) {	
		return -1 ;
	}

	/*
	 * Checking message size.
	 */
	if ( ((int)strlen(msg)+1) > MSGSIZ ) {
		serrno= SEUMSG2LONG ;
		return -1 ;
	}

	/*
	 * Building the message.
	 */
	ptr= buffer ;
	size= 1 + strlen(msg) ;
	marshall_LONG(ptr,MSG_MAGIC) ; 
	marshall_LONG(ptr,RQST_MSGI) ; 
	marshall_LONG(ptr, size) ; 
	marshall_STRING(ptr,msg) ; 

	/*
 	 * Sending the message.
 	 */
	switch( netwrite(sock,buffer,3*LONGSIZE+size) ) {
		case -1:
			(void) close(sock) ;
			return -1 ;
		case 0:
			serrno= SECONNDROP ;
			(void) close(sock) ;
			return -1 ;
	}
		
	/*
	 * Waiting for acknowledgement
	 */
	switch( rcvmsgd(sock,buffer) ) {
		case -1:
			(void) close(sock) ; 
			return -1 ; 
		case ACKN_MSGI:
			break ;
		default:
			(void) close(sock) ; 
			serrno= SEINTERNAL ;
			return -1 ;
	}
	(void) close(sock) ; 
	return 0 ;
}

/*
 * Sending MSGR request.
 */
int sndmsgr(msg,rport,identifier)
	char * msg ; 			/* Message to send	*/
	int  rport ;			/* Reply port		*/
	int  identifier ;		/* Message identifier	*/
{
	char buffer[MSGSIZ+5*LONGSIZE];	/* Buffer		*/
	char 	* ptr ;			/* Pointer to the buffer*/
	int   	 sock ;			/* Socket identifier	*/
	int	 size ;			/* Size of the message	*/
	int     rcode ;			/* Return code 		*/

	serrno= 0 ;
	/*
	 * Connecting the daemon.
	 */
	if ( (sock= conmsgd()) == -1 ) {	
		return -1 ;
	}

	/*
	 * Checking message size.
	 */
	if ( ((int)strlen(msg)+1) > MSGSIZ ) {
		serrno= SEUMSG2LONG ;
		return -1 ;
	}

	/*
	 * Building the message.
	 */
	ptr= buffer ;
	size= 2*LONGSIZE + 1 + strlen(msg) ;
	marshall_LONG(ptr,MSG_MAGIC) ; 
	marshall_LONG(ptr,RQST_MSGR) ; 
	marshall_LONG(ptr, size) ; 
	marshall_LONG(ptr,rport) ;
	marshall_LONG(ptr,identifier) ; 
	marshall_STRING(ptr,msg) ; 

	/*
 	 * Sending the message.
 	 */
	switch( netwrite(sock,buffer,3*LONGSIZE+size) ) {
		case -1:
			(void) close(sock) ;
			return -1 ;
		case 0:
			(void) close(sock) ;
			serrno= SECONNDROP ;
			return -1 ;
	}
	(void) close(sock) ; 
	return 0 ; 
}
		
/*
 * Sending reply.
 */
int sndrepl(index,msg)
	int  index ;			/* Index of the message	*/
	char * msg ;			/* Reply message	*/
{
	char buffer[MSGSIZ+4*LONGSIZE];	/* Buffer		*/
	char 	* ptr ;			/* Pointer to the buffer*/
	int   	 sock ;			/* Socket identifier	*/
	int	 size ;			/* Size of the message	*/
	int     rcode ;			/* Return code 		*/


	/*
	 * Connecting the daemon.
	 */
	if ( (sock= conmsgd()) == -1 ) {	
		return -1 ;
	}

	/*
	 * Building the message.
	 */
	ptr= buffer ;
	size= LONGSIZE + 1 + strlen(msg) ;
	marshall_LONG(ptr,MSG_MAGIC) ; 
	marshall_LONG(ptr,RQST_REPL) ; 
	marshall_LONG(ptr, size) ; 
	marshall_LONG(ptr,index) ;
	marshall_STRING(ptr,msg) ; 

	/*
 	 * Sending the message.
 	 */
	switch( netwrite(sock,buffer,3*LONGSIZE+size) ) {
		case -1:
			(void) close(sock) ;
			return -1 ;
		case 0:
			serrno= SECONNDROP ;
			(void) close(sock) ;
			return -1 ;
	}
		

	/*
	 * Waiting for acknowledgement
	 */
	switch( rcvmsgd(sock,buffer)) {
		case -1:
			rcode= -1 ;
			break ; 
		case ACKN_REPL:
			ptr= buffer ;
			unmarshall_LONG(ptr,rcode) ;
			if ( rcode < 0 ) {
				serrno= -rcode ;
				rcode= -1 ;
			}
			break ; 
		default:
			serrno= SEINTERNAL ;
			rcode= -1 ;
			break ; 
	}

	/*
	 * Closing the socket,
	 * and returning return code.
	 */
	(void) close(sock) ; 
	return rcode ;
}

/*
 * CANCELLING REQUEST.
 * The reply port is not sent since
 * the msg daemon is not sending back any 
 * acknowledgment.
 */
int canmsgr(num,rport) 
	int   num ; 			/* Index of the request */
	int rport ;			/* Reply port		*/
{
	char buffer[5*LONGSIZE];	/* Buffer		*/
	char 	* ptr ;			/* Pointer to the buffer*/
	int   	 sock ;			/* Socket identifier	*/

	serrno= 0 ;
	/*
	 * Connecting the daemon.
	 */
	if ( (sock= conmsgd()) == -1 ) 	
		return -1 ;

	/*
	 * Building the message.
	 */
	ptr= buffer ;
	marshall_LONG(ptr,MSG_MAGIC) ; 
	marshall_LONG(ptr,RQST_CANC) ; 
	marshall_LONG(ptr, LONGSIZE) ; 
	marshall_LONG(ptr,num) ;

	/*
 	 * Sending the message.
 	 */
	switch( netwrite(sock,buffer,4*LONGSIZE) ) {
		case -1:
			(void) close(sock) ;
			return -1 ;
		case 0:
			serrno= SECONNDROP ;
			(void) close(sock) ;
			return -1 ;
	}
	close(sock);
	return 0;  
}

/*
 * Receiving a reply from the message daemon.
 * It can be:
 * - An acknowledgment to a sndmsgr(), specifying the message index.
 * - A message reply.
 */
int rcvmsgr(sock,type,identifier,index,msg,msglen)
	int    sock ;			/* Socket identifier		*/
	int  * type ;			/* Type of message		*/
	int  * identifier ;		/* Local Message identifier	*/
	int  * index ;		 	/* Message index		*/
	char * msg ;			/* Buffer for the reply		*/
	int msglen ; 			/* Length of this buffer	*/
{
	char buffer[MSGSIZ+4*LONGSIZE];	/* Buffer			*/
	char 		     * ptr ;	/* Pointer to the buffer	*/
	int		     nsock ;	/* Socket identifier		*/
	struct sockaddr_in    from ;	/* Internet address structure	*/
	int  fromlen= sizeof(from) ;	/* Length of such a structure	*/

	serrno= 0 ;
	/*
	 * Accepting request.
	 */
	if ((nsock= accept(sock,(struct sockaddr *)&from,&fromlen)) == -1 ) {
		return -1 ;
	}

	/*
	 * Receiving message.
	 * Closing communications.
	 */
	*identifier= 0 ; 
	*index= 0 ; 
	*type= rcvmsgd(nsock,buffer) ; 
	(void) close(nsock) ; 
	switch( *type ) {
		case -1:
			return -1 ;
		case ACKN_MSGR:
			ptr= buffer ;
			unmarshall_LONG(ptr,*identifier) ; 
			unmarshall_LONG(ptr,*index) ;
			if ( *index < 0 ) {
				serrno= - *index ;
				return -1 ; 
			}
			return 0 ; 	
		case GIVE_REPL:
			ptr= buffer ;
			unmarshall_LONG(ptr,*identifier) ; 
			unmarshall_LONG(ptr,*index) ;
			if ( ((int)strlen(ptr)+1) > msglen ) {
				serrno= SEUBUF2SMALL ;
				return -1 ;
			}
			unmarshall_STRING(ptr,msg) ;
			return strlen(msg) ;
		default:
			serrno= SEINTERNAL ;
			return -1 ;
	}
}
