/*
 * Copyright (C) 1990-1998 by CERN/CN/SW/DC
 * All rights reserved
 */

#ifndef lint
static char sccsid[] = "@(#)oper.c	1.18 12/09/98 CERN CN-SW/DC Antoine Trannoy";
#endif /* not lint */

/* oper.c		SHIFT operator interface to remote message daemon */

#include <stdio.h>
#if ! defined(apollo) 
#include <unistd.h>
#endif 	/* ! apollo */

#include <sys/types.h>
#include <sys/time.h>
#include <sys/param.h>
#if defined(hpux)
#include <sys/file.h>
#include <signal.h>
#endif /* hpux */
#if defined(_AIX) && defined(_IBMR2)
#include <signal.h>
#include <sys/select.h>
#endif /* AIX */
#if defined(SOLARIS) || defined(linux)
#include <signal.h>
#endif /* SOLARIS || linux */
#include <sys/stat.h>
#if defined(apollo)
#include <strings.h>
#else
#include <string.h>
#endif
#include <fcntl.h>

/*
 * Processus control
 */
#if !defined(sgi) 
#include <sys/wait.h>
#endif	/* ! sgi */
#if defined(apollo)
#define SIG_ERR (void(*)())-1
#endif	/* apollo */

/*
 * Blocking and Unblocking signals
 */
#if defined(sgi) || defined(SOLARIS)
#define SIGOFF	( sighold(SIGWINCH) , sighold(SIGALRM) )
#define SIGON	( sigrelse(SIGWINCH), sigrelse(SIGALRM) )
#else
static int mask ; 
#if defined(hpux)
#define SIGOFF	(mask= sigblock(sigmask(SIGWINDOW)|sigmask(SIGALRM)|sigmask(SIGINT))) 
#else
#define SIGOFF	(mask= sigblock(sigmask(SIGWINCH)|sigmask(SIGALRM)|sigmask(SIGINT))) 
#endif 	/* hpux */
#define SIGON	sigsetmask(mask)
#endif 	/* sgi || SOLARIS */

/*
 * Terminal control.
 */
#include <curses.h>
static int 	     termflags ; 	/* Terminal flags		*/
char * tgetstr() ;
#if defined(apollo)
static struct sgttyb termargs  ;	/* Terminal parameters		*/
#define ERASE termargs.sg_erase 	/* Current erase character	*/
#define KILL  termargs.sg_kill  	/* Current kill character	*/
#else
#if defined(HPUX10)
#undef CR0
#endif
#include <termio.h>
static struct termio termargs  ;	/* Terminal parameters		*/
#define ERASE termargs.c_cc[VERASE] 	/* Current erase character	*/
#define KILL  termargs.c_cc[VKILL]  	/* Current kill character	*/
#endif	/* apollo */

/*
 * Error and message handling.
 */
#include <errno.h>
extern int errno ;
extern int sys_nerr ;
extern void perror() ;
#if !defined(linux)
extern char *sys_errlist[] ;
#endif

/*
 * Interrupt handler to catch ^C
 */
void actionSIGINT()
{
	endwin() ;
	(void) fcntl(0,F_SETFL,termflags) ;
	exit(0) ;
}

/*
 * Interrupt handler for signal SIGWINCH.
 * In some system, the signal handler has to be 
 * reset each time it is called (i.e sgi). 
 */
static int winchflag = 1 ;		/* Window changed flag		*/

void actionSIGWINCH()
{
	winchflag= 1 ;
#if defined(sgi) || defined(_AIX) || defined(hpux) || defined(SOLARIS)
	signal(SIGWINCH,(void (*)())actionSIGWINCH) ;
#endif	/* sgi || AIX || hpux || SOLARIS */
}

/*
 * Command to execute, msgd by default.
 * Buffer for the command line.
 */
static char command[20]  ;
static char cmdline[256] ;

/*
 * List of commands to be executed in refreshed mode or once
 */
struct cmda {
	char name[20] ;	/* Command name		*/
	char flag ;	/* 'r' for refresh mode, else '1' */
	struct cmda *next ;
} ;
static struct cmda *cmdlistp ;
static int hit_key_to_continue = 0 ;

/*
 * Window definitions
 *
 * + titlew: To display the command name as well as time and page number.
 * + inputw: Control window for the user.
 */
WINDOW    * titlew ;	/* Title window		*/
WINDOW	  * inputw ;	/* Input window		*/

/*
 * Output of the command is formatted into 
 * several windows stored in a doubly linked list. 
 */
struct qelem {
	struct qelem *q_forw ;
	struct qelem *q_back ;
	WINDOW       * q_win ;
} ;

static struct qelem  * listw = NULL ;	/* List of windows		*/
static struct qelem  * currw = NULL ;	/* Pointer to current window	*/
static int nbpage= 1 ; 			/* Nb of windows in the list	*/
static int   page= 1 ;			/* Window index to be displayed	*/
	
/* 
 * to avoid zombie creation, SIGCHLD signals 
 * have to be caught.
 */
#if !defined(sgi) 
void actionSIGCHLD()
{
#if !defined(_AIX) && !defined(hpux) && !defined(SOLARIS) && !(defined(__osf__) && defined(__alpha))
	union wait	status ;

	for(;;) {
		if ( wait3(&status,WNOHANG,(struct rusage *)0) <= 0 ) 
			break ;
	}
#else
	int	status;

        wait (&status);
        signal (SIGCHLD, (void (*)())actionSIGCHLD);
#endif	/* ! AIX && ! hpux && ! SOLARIS */
}
#endif	/* ! sgi */

/*
 * Is there any input character ?
 */
static int ioflag = 0 ; 

/*
 * Refresh period.
 * Every PERIOD seconds, the current command is called again.
 */
#define PERIOD	10
static unsigned int period = PERIOD ;

/*
 * Interrupt handler for signal SIGALRM.
 * In some system, the signal handler has to be 
 * reset each time it is called (i.e sgi). 
 */
static int alarmflag = 1 ;		/* Refresh flag		*/

void actionSIGALRM()
{
	alarmflag= 1 ;
#if defined(sgi) || defined(_AIX) || defined(hpux) || defined(SOLARIS)
	signal(SIGALRM,(void (*)())actionSIGALRM) ;
#endif
}

/*
 * Deleting all the windows forward
 * in the list.
 */
static void delforw(list) 
	struct qelem * list ;
{

	while  ( list != NULL ) {
		struct qelem * next ;

		delwin(list->q_win) ;
		if ( list->q_back != NULL ) 
			list->q_back->q_forw= list->q_forw ;
		if ( list->q_forw != NULL ) 
			list->q_forw->q_back= list->q_back ;

		next= list->q_forw ;	
		free(list) ;
		list= next ; 
	}
	
}

/*
 * Exiting because of an error.
 */
void wexit(string) 
	char * string ; 
{
	endwin() ;
	(void) fcntl(0,F_SETFL,termflags) ;
	perror(string) ;
	exit(1) ;
}

/*
 * Initializing terminal
 * Called at the beginning of the run, and,
 * each time a signal SIGWINCH is sent.	
 */
void initterm() 
{
	static int first = 1 ; /* Called for the first time.	*/

	/* 
	 * Resetting winchflag flag.
	 */
	winchflag= 0 ;

	/* 
	 * Getting new terminal window size.
	 * Checking its size.
	 */
	if ( termwinsiz(&COLS,&LINES) == -1 ) {
		if ( first ) {
			perror("termwinsiz()") ;
			exit(1) ;   
		}
		else	{
			wexit("termwinsiz()") ; 
		}
	}
	if ( COLS<30 || LINES<8 ) {
		if ( ! first ) {
			endwin() ;
			(void) fcntl(0,F_SETFL,termflags) ;
		}
		fprintf(stderr,"Window too small\n") ;
		fflush(stderr) ; 
		exit(1) ;
	}

	/*
	 * Deleting old window, and,
	 * resetting environment to get
	 * new terminal size.
	 */
	if ( first ) {
		first= 0 ; 
	}
	else	{
		delwin(titlew) ;
		delwin(inputw) ;
		delforw(listw);
#if defined(sgi)
		reset_shell_mode() ;
#else
		resetty() ;
#endif /* sgi */
	}

	/*
	 * Initializing terminal configuration
	 */
	initscr() ;
	cbreak() ;
	noecho() ;
#if defined(sgi) || defined(SOLARIS) || defined(linux)
	nodelay(stdscr,TRUE) ;
#else
	if ( fcntl(0,F_SETFL,FNDELAY) == -1 ) {
		wexit("fcntl(0,F_SETFL,FNDELAY)") ;
	}
#endif	/* sgi */
	/*
	 * Creating new windows.
	 * + Title window,
	 * + Input window, and,
 	 * + First output window.
	 */
	titlew = newwin(3,COLS,0,0) ;
	inputw = newwin(3,COLS,LINES-3,0);
	if ( (listw= ( struct qelem *) malloc( sizeof( struct qelem))) == NULL ) {
		wexit("malloc()") ;
	}
	listw->q_forw= NULL ;
	listw->q_back= NULL ;
	listw->q_win = newwin(LINES-6,COLS,3,0) ;
	currw= listw ;
	nbpage= 1 ;
	page= 1 ;
	scrollok (inputw, TRUE) ;
	mvwaddstr(inputw,2,0,cmdline) ;
}

/*
 * Filling the windows with data on the pipe.
 * Called exclusively from getoutput().
 */
static void getwindows(pfd) 
	int pfd[2] ;		/* Pipe file descriptor	*/
{
	char	        c ;	/* Character buffer	*/
	int		i ;	/* Page index		*/
	int            rc ; 	/* Return code		*/
	int 	      col ;	/* Current column	*/
	int 	      row ;	/* Current row		*/
	int 	     mask ;	/* Signal mask 		*/
	struct qelem * pg ; 	/* Page to fill	pointer	*/

	/* 
	 * Closing writing side of the pipe
	 */
	(void) close(pfd[1]) ;

	/*
 	 * Initializing local variables
	 */
	pg= listw ; 
	wclear(pg->q_win) ;
	i  = 1 ;
	col= 0 ;
	row= 0 ;
	SIGOFF ;
	for (;;) {
		rc= read(pfd[0],&c,1) ;
		if ( rc == -1 )
			if ( errno == EINTR ) continue ;
			else wexit("read()") ;
		if ( rc == 0 ) break;

		/*
		 * If we are on the last line 
		 * of the window.
		 */
		if ( row == (LINES-6-2) ) {
			if ( c == '\n' || col == COLS ) {
				/*
				 * The page #i is full.
				 * Pointing now to the next one.
				 */
				if ( pg->q_forw == NULL ) {
					/*
					 * Creating a new page.
					 */
					pg->q_forw= (struct qelem *) malloc( sizeof(struct qelem)) ;
					if ( pg->q_forw == NULL ) {
						wexit("malloc()") ;
					}
					pg->q_forw->q_forw= NULL ;
					pg->q_forw->q_back= pg ;       
					pg->q_forw->q_win = newwin(LINES-6,COLS,3,0) ; 
					nbpage ++ ;
				}
				pg= pg->q_forw ;
				wclear(pg->q_win) ;
				i ++ ;
				col= 0 ;
				row= 0 ;
			}
			/*
			 * printing the character if it 
			 * is not a carriage return.
			 */
			if ( c != '\n' ) {
				waddch(pg->q_win,c) ;
				col ++ ;
			}
		}
		/*
		 * This is not the last 
		 * line of the page.
		 */
		else 	{
			if ( c == '\n' ) {
				col= 0 ;
				row ++ ;
				wmove(pg->q_win,row,col) ;
			}
			else	{
				if ( col == COLS ) {
					col= 0 ;
					row ++ ;
					wmove(pg->q_win,row,col) ;
				}	
				else 	{
					col ++ ;
				}
				waddch(pg->q_win,c) ;
			}
		}
	}
	SIGON ;  

	/*
	 * If some windows are useless,
	 * they are removed.
	 */
	if ( i < nbpage ) {
		delforw(pg->q_forw) ;	
		nbpage= i ;
	}
	
	/*
	 * Closing pipe.
	 */
	(void) close(pfd[0]) ;
}

/*
 * Displaying title window.
 */
static void displaytitle()
{
	time_t clock ; 		/* To get current time	*/

	wclear(titlew) ;
	wmove(titlew,1,0) ;
	(void) time(&clock) ; 
	wprintw(titlew,"Command: %-4.10s Page:%d/%d [delay %d]",command,page,nbpage,period) ;
	wprintw(titlew," %s",ctime(&clock)) ;
	overwrite(titlew,stdscr) ;	
}

/*
 * Displaying output of the command.
 */
static void display() 
{
	/*
	 * If the page to display does not exist anymore,
	 * the last one replace it.
	 */
	if ( page > nbpage ) {
		page= nbpage ;
		currw= listw ;
		while ( currw->q_forw != NULL ) 
			currw= currw->q_forw ;	
	}
	if ( page < nbpage ) 
		mvwaddstr(currw->q_win,LINES-7,COLS-10,"<MORE>") ;
	else
		mvwaddstr(currw->q_win,LINES-7,COLS-10,"      ") ;
	overwrite(currw->q_win,stdscr) ;

	/*
	 * Displaying title window.
	 */
	displaytitle() ;

	/*
	 * Displaying input window.
	 */
	overwrite(inputw,stdscr) ;

	/*
	 * Refreshing screen.
	 */
	move(LINES-1,strlen(cmdline)) ; 
	refresh() ;
}
/*
 * Spawning a process and executing the command.
 */
static void execcmd(pfd,cmd)
int pfd[2] ;
char *cmd ;
{
	/*
	 * Creating a pipe.
	 */
	if ( pipe(pfd) == -1 ) {
		wexit("pipe()") ;
	}

	/*
	 * Spawning a process to run the command.
	 */
	switch( fork()) {
		case -1:
			wexit("fork()") ;
			/* NOTREACHED */
		case 0:
			/* 
			 * The output of the shell command
			 * is redirected on the pipe.
			 */
			(void) dup2(pfd[1],1) ;
			(void) dup2(pfd[1],2) ;
			execl("/bin/sh","sh","-c",cmd,0) ;
			wexit("exec()") ;
			/* NOTREACHED */
		default:
			break;
	}
}

/*
 * Spawning a process to execute the command.
 * Getting the output of it on a pipe.
 */
static void getoutput()
{
	int pfd[2] ;	/* Pipe file descriptor	*/

	execcmd(pfd,command);
	/* 
	 * The output of the shell command
	 * is read from the pipe, and displayed.
	 */
	getwindows(pfd) ;
	display() ;
			
	/*
	 * The timer is set again.
	 */
	alarmflag= 0 ;
	alarm(period) ;
}

/*
 * Can the command passed by argument
 * be called inside the oper display ?
 */
static int authorized(cmd)
	char * cmd ; 
{
	if ( *cmd != '\0' ) 
		return 1 ; 
	else
		return 0 ; 
}

/*
 * Checks command mode in OPERCONFIG
 * returns 'r' for refresh mode
 *         '1' for one single time
 *         '0' if command not found in OPERCONFIG
 */
static char cmdmode(cmd)
	char * cmd ;
{
	struct cmda *clp ;
	char cmdname[20] ;
	char *p, *q ;
	for (p=cmd,q=cmdname; *p; p++,q++) {
		if (*p == ' ') break ;
		*q = *p ;
	}
	*q = '\0' ;
	clp = cmdlistp ; 
	while (clp) {
		if ( !strcmp(cmdname,clp->name) ) break ;
		clp = clp->next ;
	}
	if ( clp ) return clp->flag ;
	else return '0' ;
}

static int putchr(c)
{
	putchar(c);
}

/*
 * Taking in account command given 
 * by the operator.
 */
static void docmd(cmd) 
	char * cmd ; 
{
	int pfd[2] ;	/* Pipe descriptor	*/
	int	rc ;	/* Return code		*/
	char	 c ;	/* Character buffer	*/ 
	static char hitmsg[] = "[Hit return to continue]" ;
	char sbuf[256];
	char *p, *q ;
	int scrollneeded = 0 ;

	if ( !strcmp(cmd,"ex") || !strcmp(cmd,"exit") ) {
		/*
		 * Exiting the oper command
		 */
		endwin() ;
		fcntl(0,F_SETFL,termflags) ;
		exit(0) ;
	}
	else if ( !strncmp(cmd,"ref ",4) ) {
		/*
		 * Changing the refresh period.
		 * The shortest possible period is 5s.
		 * The title and the input windows are refreshed.
		 */
		int new ;

		if ( sscanf(cmd+4,"%d",&new) ) {
			period= ( new >= 5 ) ? new : 5 ;	
			displaytitle() ;
			wrefresh(titlew) ;
			wmove(inputw,2,strlen(cmdline)) ;
			wrefresh(inputw) ;
		}	
	}
	else if ( authorized(cmd) ) {
		switch (cmdmode(cmd)) {
			case 'r':
				/*
				 * Executing a command in refresh mode
				 */
				(void) strcpy(command,cmd) ;
				getoutput() ;
				break;
			case '1':
				/*
				 * Executing a single time a command
				 * inside the oper windows.
				 */
				execcmd(pfd,cmd) ;
				(void) close(pfd[1]) ; 
				wmove(inputw,2,0) ;
				SIGOFF ;
				for (;;) {
					rc= read(pfd[0],&c,1) ;
					if ( rc == -1 )
						if ( errno == EINTR ) continue ;
						else wexit("read()") ;
					if ( rc == 0 ) break;
					if ( c == '\n' ) break ;
					waddch(inputw,c) ; 	
					scrollneeded = 1 ;
				}
				SIGON ;
				(void) close(pfd[0]) ; 
				if (scrollneeded) {
					scroll(inputw);
					mvwaddstr(inputw,2,0,cmdline) ;
					wmove(inputw,2,2) ;
					wrefresh(inputw) ; 
				}
				break;
			default:
				/*
				 * Executing a single time a command
				 * outside the oper windows
				 */
				alarm(0) ;
				clear() ;
				endwin() ;
				execcmd(pfd,cmd) ;
				(void) close(pfd[1]) ; 
				for (;;) {
					rc= read(pfd[0],&c,1) ;
					if ( rc == -1 )
						if ( errno == EINTR ) continue ;
						else wexit("read()") ;
					if ( rc == 0 ) break;
					putchar(c) ;
				}
				(void) close(pfd[0]) ; 
				q = sbuf ;
				p = tgetstr("so", &q) ;
				if (p == NULL) p = "" ;
				tputs (p, 1, putchr);	/* reverse video */
				for (p=hitmsg; *p; p++)
					putchar(*p) ;	/* hit return to continue */
				q = sbuf ;
				p = tgetstr("se", &q) ;
				if (p == NULL) p = "" ;
				tputs (p, 1, putchr);	/* normal mode */
				fflush(stdout) ;
				hit_key_to_continue = 1 ;
		}
	}
}

/*
 * Getting input from the terminal.
 */
static void getinput()
{
	int      c ; 		/* Character buffer	*/

	/*
	 * Getting characters as long as
	 * some are available.
	 */
	while( (c= getch()) != -1 ) {
		/*
		 * Current column.
		 */
		int	col ;	

		col= strlen(cmdline) ;

		/*
		 * Changing the displayed page ?
		 * If '+', ' '  or '-' are entered as the 
		 * first character of a command.
		 */
		if ( col == 2 ) {
			if ( c == '+' || c == ' ' ) {
				if ( page < nbpage ) {
					page ++ ;
					currw= currw->q_forw ;
					display() ;
				}
				return ;
			}
			if ( c == '-' ) {
				if ( page > 1 ) {
					page -- ;
					currw= currw->q_back ;
					display() ;
				}
				return ;
			}
		}
	
		if ( c == '\n') {
			char cmd[256] ;
			if (hit_key_to_continue) {
				hit_key_to_continue = 0 ;
				cbreak() ;
				noecho() ;
				wrefresh(curscr) ;
				getoutput() ;
				return ;
			}

			(void) strcpy(cmd,cmdline+2) ;
			scroll(inputw) ;
			(void) strcpy(cmdline,"> ") ;
			mvwaddstr(inputw,2,0,cmdline) ;
			wmove(inputw,2,2) ;
			wrefresh(inputw) ; 
			docmd(cmd) ;
		}
		else if ( c == KILL ) {
			(void) strcpy(cmdline,"> ") ;
			mvwaddstr(inputw,2,0,cmdline) ;
			wclrtoeol(inputw) ;
			wmove(inputw,2,2) ;
			wrefresh(inputw) ;
		}
		else if ( c == ERASE ) {
			if ( col > 2 ) {
				cmdline[col-1]= '\0' ;
				mvwaddch(inputw,2,col-1,' ');
				wmove(inputw,2,col-1) ;
				wrefresh(inputw) ;
			}
		}
		else	{
			mvwaddch(inputw,2,col,c) ;
			wrefresh(inputw) ;
			cmdline[col]= c ;	
			cmdline[col+1]= '\0';
		} 
	}
}

main()
{
	char cbuf[81];
	struct cmda *clp, *clpp;
	FILE *fopen(), *s ;
	char obuf[BUFSIZ] ;	/* To speed up screen display	*/

	/*
	 * Saving terminal flags.
	 */
	if ( ( termflags= fcntl(0,F_GETFL,0)) == -1 ) {
		perror("fcntl()") ; 
		exit(1) ; 
	}

	/*
	 * Getting ERASE and KILL characters.
	 */
#if defined(apollo)
	if ( ioctl(0,TIOCGETP,&termargs) == -1 ) {
#else
	if ( ioctl(0,TCGETA,&termargs) == -1 ) {
#endif 	/* apollo */
		perror("ioctl()") ; 
		exit(1) ; 
	}

	/*
	 * Setting buffer
	 */
	setbuf(stdout,obuf) ;		

	/*
	 * Reading configuration file.
	 */
	cmdlistp = 0 ;
	clp = 0 ;
	if ((s = fopen (OPERCONFIG,"r")) != NULL ) {
		while (fgets (cbuf,sizeof(cbuf),s) != NULL) {
			if (cbuf[0] == '#') continue ;	/* comment line */
			clpp = clp ;
			clp = (struct cmda *) calloc (1,sizeof(struct cmda)) ;
			if (cmdlistp)
				clpp->next = clp ;
			else
				cmdlistp = clp ;
			sscanf (cbuf,"%s %c",clp->name,&clp->flag) ;
		}
	}

	/*
	 * Initialization of command line.
	 */
	(void) strcpy(command,"msgd") ;
	(void) strcpy(cmdline, "> " ) ;

	/*
	 * Setting up interrupt handler
	 * to catch SIGINT signals ( ^C ).
	 */
	if ( signal(SIGINT,(void (*)())actionSIGINT) == SIG_ERR ) {
		perror("signal()") ;
		exit(1) ;
	}
	
	/*
	 * Setting up interrupt handler
	 * to avoid zombie creations. 
	 */
#if defined(sgi) 
	if ( signal(SIGCLD,SIG_IGN) == SIG_ERR ) {
		perror("signal()") ;
		exit(1) ;
	}
#else
	if ( signal(SIGCHLD,(void (*)())actionSIGCHLD) == SIG_ERR ) {
		perror("signal()") ;
		exit(1) ;
	}
#endif	/* sgi 	*/

	/*
	 * Setting an interrupt handler 
 	 * to catch SIGWINCH signals.
	 */
#if defined(hpux)
	if ( signal(SIGWINDOW,(void (*)())actionSIGWINCH) == SIG_ERR ) {
		perror("signal()") ;
		exit(1) ;
	}
#else
	if ( signal(SIGWINCH,(void (*)())actionSIGWINCH) == SIG_ERR ) {
		perror("signal()") ;
		exit(1) ;
	}
#endif	/* hpux	*/

	/*
	 * Setting an interrupt handler 
 	 * to catch SIGALRM signals.
	 */
	if ( signal(SIGALRM,(void (*)())actionSIGALRM) == SIG_ERR ) {
		perror("signal()") ;
		exit(1) ;
	}

	for(;;) {
		int      rc ;	/* Return code	*/
		fd_set rfds ;	/* Set of fds	*/

		/*
		 * The terminal size has changed or 
		 * it is the first time.
		 */
		if ( winchflag ) {
			initterm() ;
			getoutput() ;
		}
	
		/*
		 * Periodically the command is executed
		 * in order to get fresh data.
		 */
		if ( alarmflag ) {
			getoutput() ;
		}

		/*
		 * Getting input.
		 */
		FD_ZERO(&rfds) ;
		FD_SET(0,&rfds) ;
		if ( select(FD_SETSIZE,&rfds,(fd_set *) 0,(fd_set *)0,NULL) == -1 ) {
			if ( errno == EINTR ) 
				continue ; 
			else
				wexit("select()") ;
		}
		if ( FD_ISSET(0,&rfds) ) 
			getinput() ;
	}
}
