/*
 * Copyright (C) 1990-1999 by CERN/CN/SW/DC
 * All rights reserved
 */

#ifndef lint
static char sccsid[] = "@(#)tpio.c	1.42 11/10/99  CERN CN-SW/DC Antoine Trannoy";
#endif /* not lint */

/* tpio.c	Tape I/O.					*/

#include <stdio.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#if (defined(_AIX) && defined(_IBMR2)) || defined (SOLARIS) || ( defined(__osf__) && defined(__alpha) )
#include <sys/stat.h>
#endif /* _AIX && _IBMR2  ||  SOLARIS  || ( __alpha__ && __osf ) */

#if defined(CRAY)
#include <sys/bmxctl.h>
#else
#include "../tape/taperr.h"
#if ! defined(_AIX)
#include <sys/mtio.h>
#endif	/* ! _AIX */
#endif	/* CRAY	*/
#if SONYRAW
#include "../h/scsictl.h"
#endif

#include "../tape/tape.h"
#include "rtcopy.h"
#ifndef linux
extern char *sys_errlist[];
#endif /* linux */

/*
 * Defining page size.
 */
#define PAGE	4096

/*
 * To handle tape list.
 */
#if defined(CRAY)
struct bmxlist list ; 
struct bmxctl	 cb ; 
#endif	/* CRAY	*/

/*
 * Keeping information on current
 * tape file to handle labels.
 */
static int     trec ;
static int    tmode ;
static char * tfile ;
static int skiponerr = 0; 	/* If error occurs in read on a block, skip to next block */
static int Eflag = 0 ;		/* Was -E specified ? */
#if SONYRAW
static int negotiate;
int sonyraw  = 0;
#endif

extern int tpon ;
extern char tpdevtyp[];

#if ! defined(CRAY)
static char driver_name[7];
char *dvrname;
/*
 * For end of volume processing
 */
static int  eovflag ;
static char labelid[2][4] = {"EOF","EOV"} ;
#endif	/* ! CRAY */

void  setpopt(option)
int option ;
{
	if ( option == SKIP_ON_ERR )		 
		skiponerr = 1 ;
	if ( option == SET_EFLAG )
		Eflag = 1 ;
#if defined(TRACE1)
	fprintf(stderr,"Eflag set to %d\n",Eflag);
	fprintf(stderr,"skiponerr set to %d\n",skiponerr);
#endif
}

/*
 * Error writing tape.
 */
static void
twerror(fd)
	int	fd ;
{
#if !defined(CRAY)
	int errcat;
	char *msgaddr;
#endif
#if defined(CRAY)
	(void) fprintf (stderr, RT116, "CPDSKTP", sys_errlist[errno], trec+1);
	exit(UNERR) ;
#else
	switch(errno) {
		case ENXIO:
			(void) fprintf (stderr, RT140, "CPDSKTP");
			exit(SYERR) ;
		case EIO:
			errcat = gettperror (fd, &msgaddr);
			(void) fprintf (stderr, RT126, "CPDSKTP", msgaddr, trec+1);
			if ( fd >= 0 ) tcloserr(fd) ;
			switch (errcat) {
				case ETHWERR:
					rtyexit("HWERROR",SYERR);
				case ETUNREC:
				case ETBLANK:             /* should not happen on write... */
					exit(USERR);
				case ETPARIT:
				        rtyexit("SYRETRYUSERR",USERR) ;
   			        case ETNOSNS:
				default :
					rtyexit("DEVRETRY",UNERR) ;
			}
#if defined(_AIX) && defined(_IBMR2)
		case EMEDIA:
		  /* media surface error on IBM */
			(void) fprintf (stderr, RT125, "CPTPDSK", sys_errlist[errno], trec+1);
			rtyexit("SYRETRYUSERR",USERR) ;
#endif
		case EINVAL:
			(void) fprintf (stderr, RT119, "CPDSKTP", trec+1);
			exit(USERR) ; 
#if defined(sun)
			/* 
			 * Should not happen :EACCES only occurs when the driver 
			 * gave a wrong information to tpmnt. Then exit with SYRETRYD
			 */
		case EACCES :
			(void) fprintf (stderr, RT116, "CPDSKTP", sys_errlist[errno], trec+1);
			exit(SYRETRYD) ;
#endif /* sun */
		default:
			(void) fprintf (stderr, RT116, "CPDSKTP", sys_errlist[errno], trec+1);
			exit(UNERR) ;
	}
#endif	/* CRAY	*/
}

/*
 * Error reading tape.
 */
static int
trerror(fd,doskip)
	int	fd ; 
	int doskip ;
{
#if !defined(CRAY)
	int errcat;
	char *msgaddr;
#endif

#if defined(CRAY)
	(void) fprintf (stderr, RT113, "CPTPDSK", sys_errlist[errno], trec+1);
	exit(UNERR) ;
#else
	switch(errno) {
		case EIO:
			errcat = gettperror (fd, &msgaddr);
			(void) fprintf (stderr, RT125, "CPTPDSK", msgaddr, trec+1);
#if defined(TRACE1)
			fprintf(stderr,"skiponerr = %d, doskip= %d, errcat=%d, ETPARIT=%d\n",skiponerr,doskip,errcat,ETPARIT);
#endif
			if ( (skiponerr == 0 && doskip == 1) || doskip == 0 || 
					( skiponerr && doskip && errcat != ETPARIT ) ) {
				switch ( errcat ) {
					case ETPARIT: 
						if ( ! Eflag ) {
							rtyexit("SYRETRYUSERR",USERR) ;
						}
						else
							rtyexit("SYRETRYPARITY",MNYPARY) ;
					case ETHWERR:
						rtyexit("HWERROR",SYERR) ;
					case ETBLANK:
					case ETUNREC:
						if (trec && Eflag && ! skiponerr)
							exit (MNYPARY) ;
						else
							exit (USERR) ;
					default:
						rtyexit("DEVRETRY",UNERR);
				}
			}
				/* Option to skip bad block is set and fits the error */
			else {
				(void) fprintf (stderr, RT211, "CPTPDSK");
				return NEXTREC ;
			}
			break ;
#if defined(_AIX) && defined(_IBMR2)
		case EMEDIA:
			(void) fprintf (stderr, RT125, "CPTPDSK", sys_errlist[errno], trec+1);
			if ( skiponerr ) {	/* -E skip */
				(void) fprintf (stderr, RT211, "CPTPDSK");
				return NEXTREC ;
			}
			if ( ! Eflag ) {
				rtyexit("SYRETRYUSERR",USERR) ;
			}
			else	/* -E keep */
				rtyexit("SYRETRYPARITY",MNYPARY) ;
#endif
		case ENOSPC:
			(void) fprintf (stderr, RT137, "CPTPDSK", trec+1);
			exit(USERR) ;
		case ENXIO:
			(void) fprintf (stderr, RT140, "CPTPDSK");
			exit(SYERR) ; 
		case ENOMEM:
		case EINVAL:
			(void) fprintf (stderr, RT103, "CPTPDSK", trec+1);
			if ( skiponerr ) {
				(void) fprintf (stderr, RT211, "CPTPDSK");
				return NEXTREC ;
			}
			exit(USERR) ;
		default:
			(void) fprintf (stderr, RT113, "CPTPDSK", sys_errlist[errno], trec+1);
			exit(UNERR) ; 
	}
#endif	/* CRAY	*/
return(0);
}

/*
 * Opening tape file.
 */
int topen(file,mode)
	char * file ;
	int    mode ;
{
	int fd ; 

	/*
	 * Keeping information on current file.
	 */
	trec = 0 ;
	tfile= file ;
	tmode= mode ;
#if ! defined(CRAY)
	eovflag = 0 ;
#endif	/* ! CRAY */

	/*
	 * Opening file.
	 */
#if SONYRAW
	if (! sonyraw) {
#endif
		fd= open(file,mode);
#if SONYRAW
	} else {
#if SOLARIS
		fd= open(file,mode);
#else
		fd = open (file, O_RDWR);
#endif
		negotiate = 0;
    }
#endif
	if ( fd == -1 ) {
		if ( mode == O_RDWR ) 
			fprintf (stderr, RT111, "CPDSKTP", sys_errlist[errno]);
		else
			fprintf (stderr, RT111, "CPTPDSK", sys_errlist[errno]);
#if defined(CRAY)
		exit(UNERR) ; 
#else
		exit(SYERR) ; 
#endif	/* CRAY	*/
	}
	/*
	 * Using tapelist I/O
	 */
#if defined(CRAY)
	cb.bxc_address= (long) &list ;
	cb.bxc_count= 1 ;
	if ( ioctl(fd,BXC_LIST,&cb) == -1 ) {
		if ( mode == O_RDWR ) 
			fprintf (stderr, RT114, "CPDSKTP", sys_errlist[errno]);
		else
			fprintf (stderr, RT114, "CPTPDSK", sys_errlist[errno]);
		exit(SYERR) ;
	}
#endif	/* CRAY	*/
#if defined(_AIX) && defined(_IBMR2)
	if (getdvrnam (file, driver_name) < 0)
		strcpy (driver_name, "tape");
	dvrname = driver_name;
#endif
#if defined(linux)
        clear_compression_stats(fd,tfile,tpdevtyp);
#endif /* linux */
	/*
	 * Returning file descriptor.
	 */
	return fd ; 
}

/*
 * Closing tape file.
 */
int tclose(fd) 
{
#if !defined(CRAY)
#if defined(linux)
        float   compress ;
        COMPRESSION_STATS compstats ;
#endif
#if SONYRAW
    if (! sonyraw) {
#endif
	if ( tmode == O_RDWR && trec ) {
		if ( wrttrllbl(fd,tfile,labelid[eovflag],trec) < 0 ) {
#if defined(_IBMR2)
			if ( errno == ENOSPC || errno == ENXIO)
#else
			if ( errno == ENOSPC )
#endif
				exit(ENOSPC);
			else
				twerror(fd) ;
		}
	}
#if defined(linux)
	if ( trec && !get_compression_stats(fd,tfile,tpdevtyp,&compstats) ) {
		if ( tmode == O_RDWR ) {
			(void) fprintf(stderr, RT215, "CPDSKTP", compstats.from_host) ;
			(void) fprintf(stderr, RT216, "CPDSKTP", compstats.to_tape) ;
			if (compstats.to_tape) {
				compress=(float)compstats.from_host/(float)compstats.to_tape ;
				(void) fprintf(stderr, RT217, "CPDSKTP", compress) ;
			}
		} else {
			(void) fprintf(stderr, RT218, "CPTPDSK", compstats.to_host) ;
			(void) fprintf(stderr, RT219, "CPTPDSK", compstats.from_tape) ;
			if (compstats.from_tape) {
				compress=(float)compstats.to_host/(float)compstats.from_tape ;
				(void) fprintf(stderr, RT217, "CPTPDSK", compress) ;
			}
		}
	}
#endif /* linux */
#if SONYRAW
    }
#endif
#endif	/* ! CRAY */
	(void) close(fd) ; 
	tpon=0 ;
	return 0 ;
}

/*
 * Closing tape file on error.
 */
int tcloserr(fd) 
{
#if !defined(CRAY)
	if ( trec && deltpfil(fd,tfile) < 0 ) {
		(void) fprintf (stderr, RT141, "CPDSKTP");
	}
#endif	/* ! CRAY */
	(void) close(fd) ;
	/* Just in case the SIGPIPE handler is raised */
	tpon=0 ;
	return 0 ;
}

/*
 * Writing tape block.
 */
int twrite(fd,ptr,len)
	int     fd ;
	char * ptr ;
	int    len ; 
{
	int 	rc ;
#if !defined(CRAY)
	char nextvol[100] ;	/* Nextvol command      */
	int    tfd ;

writeblock:
#if SONYRAW
    if (! sonyraw) {
#endif
	/*
	 * Writing header labels.
	 * (The CRAY is smart enough to do it by itself)
	 */
	if ( trec == 0 ) {
		if ( wrthdrlbl(fd,tfile) < 0 )
#if defined(_IBMR2)
			if ( errno == ENOSPC || errno == ENXIO)
#else
			if ( errno == ENOSPC )
#endif
				exit(ENOSPC);
			else
				twerror(fd) ;
	}
#endif	/* ! CRAY */
	
	/*
	 * Writing data.
	 * trec is incremented to know how many BLOCKs
	 * have been written (needed for trailing labels).
	 */
	trec ++ ;
#if defined(CRAY)
	list.flags= 0 ; 
	list.bytes= len ;
	list.state= BMS_EOR ;
	rc= write(fd,ptr,((len+PAGE-1)/PAGE)*PAGE) ;
#else
	rc= write(fd,ptr,len) ;
#endif	/* CRAY	*/
	/*
	 * There is an error.
	 */
	if ( rc == -1 ) 
#if ! defined(CRAY)
#if defined(_IBMR2)
		if ( errno == ENOSPC || errno == ENXIO)
#else
		if ( errno == ENOSPC )
#endif
			eovflag= 1;	/* tape volume overflow */
		else
#endif	/* ! CRAY */
			twerror(fd) ;

#if defined(sun) || defined(sgi)
	if ( rc == 0 ) {
		struct mtget mt_info ;

		if ( ioctl(fd,MTIOCGET,&mt_info) == -1 ) {
			fprintf (stderr, RT109, "CPDSKTP", sys_errlist[errno]);
			exit(SYERR) ; 
		}
#if defined(sun)
		if ( mt_info.mt_erreg == 19 ) {	/* SC_EOT */
#else
		if ( mt_info.mt_erreg == 0 ) {
#endif	/* sun */
			eovflag= 1;	/* tape volume overflow */
		}
		else	{
			(void) fprintf (stderr, RT117, "CPDSKTP", mt_info.mt_erreg, trec+1);
#if defined(sun)
                        rtyexit("DEVRETRY",UNERR) ;
#else
			exit(UNERR) ;
#endif  /* sun */
		}
	}
#endif	/* sun || sgi */
#if ! defined(CRAY)
	/*
	 * Switch volumes if necessary
	 */
	if ( eovflag ) {
		trec-- ;
		tclose(fd) ;
		sprintf(nextvol,"nextvol -P %s",tfile) ;
		rc= system(nextvol) ;
		if ( rc & 0xFF ) {
			(void) fprintf (stderr, RT122, "CPDSKTP"); /* interrupted */
			exit(UNERR) ;
		}
		rc= (rc >> 8) & 0xFF ;
		if ( rc == 214 ) {
			(void) fprintf (stderr, RT124, "CPDSKTP"); /* vol overflow */
			exit(ENOSPC) ;
		}
		if ( rc != 0 ) {	/* error switching tapes */
			exit(UNERR) ;
		}
		tfd= topen(tfile,O_RDWR) ;
		if ( tfd != fd ) {
			fcntl(tfd,F_DUPFD,fd) ;	/* forces same decriptor number */
			close(tfd) ;
		}
		goto writeblock;
	}
#endif	/* ! CRAY */
#if SONYRAW
    } else {
	return (write_sony(fd, ptr, len));
    }
#endif
  return(rc);
}

/*
* Reading tape block.
*/
int tread(fd,ptr,len)
	int     fd ;
	char * ptr ;
	int    len ; 
{
	int 	rc ; 
	int status ;
#if !defined(CRAY)
	char nextvol[100] ;	/* Nextvol command      */
	int    tfd ;
#endif	/* ! CRAY */

#if defined(CRAY)
	list.flags= 0 ; 
	list.bytes= len ;
	list.state= BMS_EOR ;
#else
readblock:
#endif	/* CRAY	*/

	/*
	 * Reading block.
	 */
#if SONYRAW
    if (! sonyraw) {
#endif
	if ( (rc= read(fd,ptr,len)) == -1 )  {
		/* 
		 * trerror returns a code when -E option is set to 
		 * skip when an error on the tape is encountered
		 */
		status = trerror(fd,1) ; 				
		if (status == NEXTREC ) {
			trec ++ ;
			return NEXTREC ;
		}
	}

	/*
	 * Count the number of blocks
	 */
	trec ++ ;

	/*
	 * Blank tape may have been encountered.
	 */
#if defined(sun)
	if ( rc == 0 ) {
		char *msgaddr;

		if (gettperror (fd, &msgaddr) == ETBLANK) {
			(void) fprintf (stderr, RT118, "CPTPDSK", trec);
			if (trec && Eflag && ! skiponerr)
				exit (MNYPARY) ;
			else
				exit (USERR) ;
		}
	}
#endif	/* sun	*/
	
	/*
	 * Returning number of bytes read.
	 */
#if defined(CRAY)
	if ( list.state == BMS_EOD ) 
		return 0 ;
	else
		return list.bytes ;
#else
	if ( rc == 0 ) {
		if ((rc = checkeofeov(fd, tfile)) < 0) {
			if (rc == -ETLBL) {
				(void) fprintf (stderr, RT128, "CPTPDSK");
				exit (USERR);
			} else
				exit (UNERR);
		}
		if ( rc == 0 ) return 0 ;	/* end of file has been reached */
		tclose(fd) ;
		sprintf(nextvol,"nextvol -P %s",tfile) ;
		rc= system(nextvol) ;
		if ( rc & 0xFF ) {
			(void) fprintf (stderr, RT122, "CPTPDSK"); /* interrupted */
			exit(UNERR) ;
		}
		rc= (rc >> 8) & 0xFF ;
		if ( rc == 214 ) return 0 ;	/* list of tapes is exhausted */
		if ( rc != 0 ) {	/* error switching tapes */
			exit(UNERR) ;
		}
		tfd= topen(tfile,O_RDONLY) ;
		if ( tfd != fd ) {
			fcntl(tfd,F_DUPFD,fd) ;	/* forces same decriptor number */
			close(tfd) ;
		}
		goto readblock ;
	} else
		return rc ; 
#endif	/* CRAY	*/
#if SONYRAW
    } else {
	return (read_sony(fd, ptr, len));
    }
#endif
}

#if SONYRAW
/*	tpio_sony - sends scsi commands to support RAW mode on SONY DIR1000
	using a DFC-1500/1700 controller */

read_sony(fd, buf, len)
int fd;
char *buf;
int len;
{
	unsigned char cdb[6];
	int errcat;
	int flags;
	char *msgaddr;
	int n;
	int nb_sense_ret;
	int ntracks;
	int offset;
	int rc;
	char sense[MAXSENSE];
	int tpt;

	ntracks = len / 144432;
#if IRIX6
	tpt = 29;
#else
#if HPUX10
	tpt = 7;	/* Tracks per scsi transfer */
#else
	tpt = 1;
#endif
#endif
	offset = 0;
	trec++;
	flags = SCSI_IN | SCSI_SEL_WITH_ATN;
	if (! negotiate) {	/* Set Fast/Wide on first xfer */
		flags |= (SCSI_SYNC | SCSI_WIDE);
		negotiate = 1;
	}
	while (ntracks > 0) {
		memset (cdb, 0, sizeof(cdb));
		cdb[0] = 0x08;		/* Read */
		cdb[1] = 0x01;		/* Fixed blocks */
		n = (ntracks > tpt) ? tpt : ntracks;
		cdb[4] = n;
		rc = send_scsi_cmd (fd, "", 1, cdb, 6, buf + offset, n * 144432,
			sense, 38, 30000, flags, &nb_sense_ret, &msgaddr);
		if (rc == -1 || rc == -2) {
			(void) fprintf (stderr, RT113, "CPTPDSK", msgaddr, trec);
			exit (UNERR);
		}
		if (rc < 0) {
			(void) fprintf (stderr, RT125, "CPTPDSK", msgaddr, trec);
			if (rc == -4 && nb_sense_ret >= 14 &&
			    get_sk_msg (sense[2] & 0xF, sense[12], sense[13], &msgaddr) == ETPARIT)
				exit (USERR);
			else
				exit (SYERR);
		}
		ntracks -= n;
		offset += n * 144432;
	}
	return (len);
}
		
write_sony(fd, buf, len)
int fd;
char *buf;
int len;
{
	unsigned char cdb[6];
	int errcat;
	int flags;
	char *msgaddr;
	int n;
	int nb_sense_ret;
	int ntracks;
	int offset;
	int rc;
	char sense[MAXSENSE];
	int tpt;

	ntracks = len / 144432;
#if IRIX6
	tpt = 29;
#else
#if HPUX10
	tpt = 7;	/* Tracks per scsi transfer */
#else
	tpt = 1;
#endif
#endif
	offset = 0;
	trec++;
	flags = SCSI_OUT | SCSI_SEL_WITH_ATN;
	if (! negotiate) {	/* Set Fast/Wide on first xfer */
		flags |= (SCSI_SYNC | SCSI_WIDE);
		negotiate = 1;
	}
	while (ntracks > 0) {
		memset (cdb, 0, sizeof(cdb));
		cdb[0] = 0x0A;		/* Write */
		cdb[1] = 0x01;		/* Fixed blocks */
		n = (ntracks > tpt) ? tpt : ntracks;
		cdb[4] = n;
		rc = send_scsi_cmd (fd, "", 1, cdb, 6, buf + offset, n * 144432,
			sense, 38, 30000, flags, &nb_sense_ret, &msgaddr);
		if (rc == -1 || rc == -2) {
			(void) fprintf (stderr, RT116, "CPDSKTP", msgaddr, trec);
			exit (UNERR);
		}
		if (rc < 0) {
			(void) fprintf (stderr, RT126, "CPDSKTP", msgaddr, trec);
			if (rc == -4 && nb_sense_ret >= 14 &&
			    get_sk_msg (sense[2] & 0xF, sense[12], sense[13], &msgaddr) == ETPARIT)
				exit (USERR);
			else
				exit (SYERR);
		}
		ntracks -= n;
		offset += n * 144432;
	}
	return (len);
}
#endif
