/*
 * Copyright (C) 1990-1998 by CERN/IT/PDP/IP
 * All rights reserved
 */

#ifndef lint
static char sccsid[] = "@(#)stream.c	1.16 12/14/98  F. Hemmer, A. Trannoy, F. Hassine";
#endif /* not lint */

/* stream.c       Remote File I/O - Version 3 streaming routines        */

#define RFIO_KERNEL     1       /* system part of Remote File I/O       */

#include <syslog.h>             /* system logger                        */
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include "rfio.h"               /* remote file I/O definitions          */
#include "rfcntl.h"             /* remote file control mapping macros   */
#include <pwd.h>
#include <signal.h>

#if !defined(_WIN32)
#include <sys/time.h>
#endif
#if defined(_AIX) && defined(_IBMR2)
#include <sys/select.h>
#endif
#include <stdlib.h>

#if defined(_WIN32)
#include <winsock2.h>
#else
#include <netinet/in.h>
#include <netinet/tcp.h>
#endif

#ifndef linux
extern char *sys_errlist[];     /* system error list                    */
#endif

static int data_rfio_connect();

static void rfio_setup_ext_v3(iop,uid,gid,passwd)
	RFILE   *iop;
	int 	uid;
	int 	gid;
	int	passwd;
{
	extern char * getenv() ; 	/* External declaration		*/
	char * cp ; 			/* Character pointer		*/
	int v ;

	if ( cp= getenv("RFIO_READOPT") ) {
		v = atoi(cp) ;
		rfiosetopt(RFIO_READOPT, &v , 4) ; 
	}
	/*
	 * We don't allow READAHEAD without buffering 
	 * until the Ultra bug ( or mine ) is fixed.
	if ( rfioreadopt(RFIO_READOPT) == 2 ) {
			v = 3 ;
			rfiosetopt( RFIO_READOPT, &v ,4) ; 
	}
	 */

	iop->magic = RFIO_MAGIC;
	iop->s = -1;
	if (uid || gid)
		iop->mapping=0;
	else
		iop->mapping=1;
	iop->passwd=passwd; /* used only if mapping == 0 */
	iop->uid = (uid==0 ? geteuid(): uid);
	iop->gid = (gid==0 ? getegid(): gid);
	INIT_TRACE("RFIO_TRACE");
	TRACE ( 1,"rfio","rfio_setup_ext(%d,%d,%d)",iop,uid,gid);
	TRACE ( 2,"rfio","rfio_setup_ext: owner s uid is %d",iop->uid);
	TRACE ( 2,"rfio","rfio_setup_ext: owner s gid is %d",iop->gid);
	END_TRACE();
	(void) umask(iop->umask=umask(0));
	iop->ftype = FFTYPE_C;
#if !defined(VM) && !defined(MVS)
	iop->binary = 0;                 /* no translation needed        */
#endif /* VM */
	iop->eof = 0;
	iop->unit = 0;
	iop->access = 0;
	iop->format = 0;
	iop->recl = 0;
	iop->blank = 0;
	iop->opnopt = 0;
	iop->offset= 0 ; 
	iop->_iobuf.base = NULL;
	iop->_iobuf.ptr = NULL;
	iop->_iobuf.count = 0;
	iop->_iobuf.hsize = 0;
	iop->_iobuf.dsize = 0;
	iop->lseekhow= -1 ; 
	iop->ahead= rfioreadopt(RFIO_READOPT) & RFIO_READAHEAD ;
	iop->eof= 0 ; 
	iop->readissued= 0 ; 
	iop->preseek = 0 ; 
	iop->nbrecord= 0 ; 
	strcpy(iop->host,"????????");
}

int     rfio_cleanup_v3(s)         /* cleanup rfio descriptor              */
int     s;
{
	INIT_TRACE("RFIO_TRACE");
	TRACE(1, "rfio", "rfio_cleanup_v3(%d)", s);

	if (rfilefdt[s] != NULL) {
		if (rfilefdt[s]->magic != RFIO_MAGIC && rfilefdt[s]->magic != B_RFIO_MAGIC) {
			serrno = SEBADVERSION ; 
			END_TRACE();
			return(-1);
		}
		if (rfilefdt[s]->_iobuf.base != NULL)    {
			TRACE(2, "rfio", "freeing I/O buffer at 0X%X", rfilefdt[s]->_iobuf.base);
			(void) free(rfilefdt[s]->_iobuf.base);
		}
		TRACE(2, "rfio", "closing %d",s) ;
		(void) close(s) ;
		TRACE(2, "rfio", "freeing RFIO descriptor at 0X%X", rfilefdt[s]);
		(void) free((char *)rfilefdt[s]);
		rfilefdt[s] = NULL;
	}
	END_TRACE();
	return(0);
}
	
int 	rfio_open_v3(filepath, flags, mode)
        char    * filepath ;
        int     flags,mode ;
{
	char rh[1] ;
	rh[0]='\0' ;

	return(rfio_open_ext_v3(filepath, flags, mode,(uid_t)0,(gid_t)0,0,rh,rh));
}


/*
 * Remote file open.
 */
int	rfio_open_ext_v3(filepath, flags, mode,uid,gid,passwd,reqhost,vmstr) 
	char    * filepath ;
	int 	flags,mode ;
	uid_t	uid;
	gid_t	gid;
	int 	passwd ;
	char 	* reqhost; /* In case of a Non-mapped I/O with uid & gid 
				sepcified, which host will be contacted
				for key check ? */
	char  	*vmstr ;
{
	/*
	 * TODO: support special filemode for binary/nobinary. This only
	 *       applies to non-ascii machines (e.g. IBM VM/MVS)
	 */
	int     status ;	/* Return code 			*/
	int	 rcode ;	/* Remote errno			*/
	int        len ;
	char    * host ; 
	char * filename;
	char  * account;
	char       * p ;	/* Pointer to rfio buffer	*/
	RFILE    * rfp ;	/* Remote file pointer          */
	WORD	   req ;
	struct passwd *pw;
	int 	    rt ; 	/* daemon in site(0) or not (1) */	
	int    bufsize ; 	/* socket buffer size 		*/	
	struct sockaddr_in      to;
	int                     tolen;
	struct  hostent *hp;
	extern void rfio_setup_ext_v3();
	extern char * getifnam() ;
	int data_port;
	int rem;
	int rb;
	int yes;

	INIT_TRACE("RFIO_TRACE");
	TRACE(1,"rfio","rfio_open_ext(%s, %d, %d, %d, %d, %d, %s, %s)",filepath,flags,mode,uid,gid,passwd,reqhost,vmstr ) ;

#if defined (CLIENTLOG)
        /* Client logging */
        rfio_logst();
#endif /* CLIENTLOG */

	/*
	 * The file is local.
	 */
	if ( ! rfio_parse(filepath,&host,&filename) ) {
		status= open(filename, flags, mode) ;
		END_TRACE() ; 
		rfio_errno = 0;
#if defined (CLIENTLOG)
          /* Client logging */
          	rfio_logop(status,filename,host,flags);
#endif /* CLIENTLOG */
		return status ;
	}

	
	/*
	 * Allocate and initialize a remote file descriptor.
	 */
	if ((rfp = (RFILE *)malloc(sizeof(RFILE))) == NULL)        {
		TRACE(2, "rfio", "rfio_open: malloc(): ERROR occured (errno=%d)", errno);
		END_TRACE();
		return(-1);
	}
	rfio_setup_ext_v3(rfp,(int)uid,(int)gid,passwd) ;
	TRACE(2, "rfio", "RFIO descriptor allocated at 0x%X", rfp);
	/*
	 * Connecting server.
	 */
	rfp->s = rfio_connect(host,&rt);
	if (rfp->s < 0)      {
		TRACE(2, "rfio", "Failing Doing first connect");

		TRACE(2, "rfio", "freeing RFIO descriptor at 0X%X", rfp);
		(void) free(rfp);
		END_TRACE();
		return(-1);
	}
	tolen=sizeof(to);
	if (getpeername(rfp->s,(struct sockaddr *)&to, &tolen)<0)        {
		syslog(LOG_ALERT, "rfio: open: getpeername: %s\n",sys_errlist[errno]);
	}
	if ((hp = gethostbyaddr((char *) (&to.sin_addr), sizeof(struct in_addr), to.sin_family)) == NULL){
		strncpy(rfp->host, (char *)inet_ntoa(to.sin_addr), RESHOSTNAMELEN );
	}
	else    {
		strncpy(rfp->host,hp->h_name, RESHOSTNAMELEN );
	}


	if ( !rt && !rfp->mapping ) {
		rfp->uid=geteuid() ;
		rfp->gid=getegid() ;
		TRACE(3,"rfio", "re-setting (uid,gid) to %d,%d",rfp->uid,rfp->gid) ;
		rfp->mapping = 1 ;
	}
		
	/*
	 * Remote file table is not large enough.
	 */
	if ( rfp->s >= MAXRFD ) {
		TRACE(2, "rfio", "freeing RFIO descriptor at 0X%X", rfp);
		(void) free(rfp);
		END_TRACE();
		errno= EMFILE ;
		return -1 ;
	}
	rfilefdt[rfp->s]=rfp;
	bufsize= DEFIOBUFSIZE ;

	/* reset error code */
	serrno=0;

	/* Initialization needed for RFIO version 3 write and read requests */
	rfp->first_write = 1;
	rfp->byte_written_to_network = 0;
	rfp->first_read = 1;
	rfp->byte_read_from_network = 0;

	/* Set the keepalive option on socket */
	yes = 1;
	if (setsockopt(rfp->s,SOL_SOCKET,SO_KEEPALIVE,(char *)&yes, sizeof(yes)) < 0) {
	  TRACE(2,"rfio","setsockopt keepalive on ctrl: %s",sys_errlist[errno]);
	}
	TRACE(2,"rfio","setsockopt keepalive on ctrl done");

#if (defined(__osf__) && defined(__alpha) && defined(DUXV4))    	    
	/* Set the keepalive interval to 20 mns instead of the default 2 hours */
	yes = 20 * 60;
	if (setsockopt(rfp->s,IPPROTO_TCP,TCP_KEEPIDLE,(char *)&yes,sizeof(yes)) < 0) {
	  TRACE(2,"rfio","setsockopt keepidle on ctrl: %s",sys_errlist[errno]);
	}
	TRACE(2,"rfio","setsockopt keepidle on ctrl done (%d s)",yes);
#endif
	    
#if !(defined(__osf__) && defined(__alpha) && defined(DUXV4))
	yes = 1;
	if (setsockopt(rfp->s,IPPROTO_TCP,TCP_NODELAY,(char *)&yes,sizeof(yes)) < 0) {
	  TRACE(2,"rfio","setsockopt nodelay on ctrl: %s",sys_errlist[errno]);
	}
	TRACE(2,"rfio","setsockopt nodelay option set on ctrl socket");
#endif /* !(defined(__osf__) && defined(__alpha) && defined(DUXV4)) */

	/*
	 * Allocate, if necessary, an I/O buffer.
	 */
	rfp->_iobuf.hsize= 3*LONGSIZE + WORDSIZE ;
	if ( rfioreadopt(RFIO_READOPT) & RFIO_READBUF ) {
                rfp->_iobuf.dsize = bufsize - rfp->_iobuf.hsize;
		rfp->_iobuf.count = 0;
		rfp->_iobuf.ptr = iodata(rfp) ;
	}

	if ( (pw = getpwuid(geteuid()) ) == NULL ) {
			TRACE(2, "rfio" ,"rfio_open_v3: getpwuid() error %s",sys_errlist[errno]);
			rfio_cleanup_v3(rfp->s);
                        END_TRACE();
                        return -1 ;
	}
	/*
	 * Building and sending request.
	 */
	if ((account = getacct()) == NULL) account = "";
	TRACE(2,"rfio","rfio_open: uid %d gid %d umask %o ftype %d, mode %d, flags %d",
		rfp->uid,rfp->gid,rfp->umask,rfp->ftype,mode,flags) ;
	TRACE(2,"rfio","rfio_open: account: %s",account) ;
	TRACE(2,"rfio","rfio_open: filename: %s",filename) ;
	if (reqhost != NULL && strlen(reqhost) )
		TRACE(2,"rfio","rfio_open: requestor's host: %s",reqhost) ;
	p= rfio_buf ;
	len= 5*WORDSIZE + 3*LONGSIZE + strlen(account) + strlen(filename) +strlen(pw->pw_name) + strlen(reqhost) + strlen(vmstr) + 5 ;
	marshall_WORD(p,B_RFIO_MAGIC) ;
	marshall_WORD(p,RQST_OPEN_V3) ;
	marshall_LONG(p,len) ;
	p= rfio_buf + RQSTSIZE ;
	marshall_WORD(p,rfp->uid) ;
	marshall_WORD(p,rfp->gid) ;
	marshall_WORD(p,rfp->umask) ;
	marshall_WORD(p,FFTYPE_C) ;
	marshall_LONG(p,htonopnflg(flags)) ;
	marshall_LONG(p,mode) ;
	marshall_STRING(p,account) ;
	marshall_STRING(p,filename) ;
	marshall_STRING(p,pw->pw_name) ;
	marshall_STRING(p,reqhost) ;
	marshall_LONG(p,rfp->passwd);
	marshall_WORD(p,rfp->mapping);
	marshall_STRING(p, vmstr) ;
	TRACE(2,"rfio","rfio_open_v3: sending %d bytes",RQSTSIZE+len) ;
	if (netwrite(rfp->s,rfio_buf,RQSTSIZE+len) != RQSTSIZE+len) {
		TRACE(2,"rfio","rfio_open_v3: write(): ERROR occured (errno=%d)", errno) ;
		syslog(LOG_ALERT, "rfio: open_v3: %s (error %d with %s) [uid=%d,gid=%d,pid=%d] in netwrite(%d,0X%X,%d)",
			sys_errlist[errno], errno, rfp->host, rfp->uid, rfp->gid, getpid(), rfp->s, rfio_buf, RQSTSIZE+len);
		rfio_cleanup_v3(rfp->s) ;
		END_TRACE() ;
		return -1 ;
	}
	/*
	 * Getting status and current offset.
	 */
	TRACE(1, "rfio", "rfio_open_v3: reading %d bytes",rfp->_iobuf.hsize) ;
	serrno = 0;
	if ((rb = netread(rfp->s,rfio_buf,RQSTSIZE)) != RQSTSIZE ) {
	  if (rb == 0)
	    {
	      TRACE(2, "rfio", "rfio_open_v3: Server doesn't support V3 protocol");
	      serrno = SEPROTONOTSUP;
	      return(-1);
	    }
	  else
	    {
	      TRACE(2, "rfio", "rfio_open_v3: read(): ERROR occured (errno=%d)", errno);
	      syslog(LOG_ALERT, "rfio: open_v3: %s (error %d with %s) [uid=%d,gid=%d,pid=%d] in netread(%d,0X%X,%d)",
		     sys_errlist[errno], errno, rfp->host, rfp->uid, rfp->gid, getpid(), rfp->s, rfio_buf, RQSTSIZE+len);
	      rfio_cleanup_v3(rfp->s);
	      END_TRACE();
	      return(-1);
	    }
	}
	p= rfio_buf ;
	unmarshall_WORD(p,req) ; 
	unmarshall_LONG(p,status) ;
	unmarshall_LONG(p, rcode) ; 
	unmarshall_LONG(p, data_port);
	TRACE(1,"rfio","rfio_open_v3: return status(%d), rcode(%d), fd: %d",status,rcode,rfp->s) ;

	/* will have to check if server doesn't support OPEN_V3 */
	if ( status < 0 ) {
		if ( rcode >= SEBASEOFF)
			serrno = rcode ;
		else
			rfio_errno= rcode ;
		/* Operation failed but no error message was sent */
		if ( rcode == 0 )
			serrno = SENORCODE ;
		rfio_cleanup_v3(rfp->s) ;
		END_TRACE() ;
		return -1 ;
	}
	else {
		rfp->offset= status ;
	}

	/* lseekhow field contains the fd of the data socket */
	/* Add error checking here ! (-1)*/
	rfp->lseekhow = data_rfio_connect(host,&rem,data_port,flags);

	/* In case of network errors, abort */
	if (rfp->lseekhow == -1) {
	        rfio_cleanup_v3(rfp->s) ;
                END_TRACE() ;
                return -1 ;
	}

	/* Set the keepalive option on both sockets */
	yes = 1;
	if (setsockopt(rfp->lseekhow,SOL_SOCKET,SO_KEEPALIVE,(char *)&yes, sizeof(yes)) < 0) {
	  TRACE(2,"rfio","setsockopt keepalive on data: %s",sys_errlist[errno]);
	}
	TRACE(2,"rfio","setsockopt keepalive on data done");

#if (defined(__osf__) && defined(__alpha) && defined(DUXV4))    	    	    
	/* Set the keepalive interval to 20 mns instead of the default 2 hours */
	yes = 20 * 60;
	if (setsockopt(rfp->lseekhow,IPPROTO_TCP,TCP_KEEPIDLE,(char *)&yes,sizeof(yes)) < 0) {
	  TRACE(2,"rfio","setsockopt keepidle on data: %s",sys_errlist[errno]);
	}
	TRACE(2,"rfio","setsockopt keepidle on data done (%d s)",yes);
#endif	    
#if !(defined(__osf__) && defined(__alpha) && defined(DUXV4))
	yes = 1;
	if (setsockopt(rfp->lseekhow,IPPROTO_TCP,TCP_NODELAY,(char *)&yes,sizeof(yes)) < 0) {
	  TRACE(2,"rfio","setsockopt nodelay on data: %s",sys_errlist[errno]);
	}
	TRACE(2,"rfio","setsockopt nodelay option set on data socket");
#endif /* !(defined(__osf__) && defined(__alpha) && defined(DUXV4)) */
	    
#if defined (CLIENTLOG)
        /* Client logging */
        rfio_logop(rfp->s,filename,host,flags);
#endif
	/*
	 * The file is open, update rfp->fp
	 */
#if defined(hpux)
        rfp->fp.__fileL = rfp->s;
#else
#if defined(linux)
        rfp->fp._fileno = rfp->s;
#else
#if defined(__Lynx__)
        rfp->fp._fd = rfp->s;
#else
        rfp->fp._file = rfp->s;
#endif  /* __Lynx__ */
#endif  /* linux */
#endif  /* hpux */
#if defined(CRAY)
	rfp->fp._file50= rfp->s ;
#endif	/* CRAY	*/
	END_TRACE() ;
	return (rfp->s) ;
}

void rfio_setup_v3(iop)
        RFILE   *iop;
{
        (void)rfio_setup_ext_v3(iop,0,0,0);
}

/*
 * Remote file read
 */
int rfio_read_v3(ctrl_sock, ptr, size) 
char    *ptr;
int     ctrl_sock, size;
{
	int status ;	/* Return code of called func	*/
	char   * p ; 	/* Pointer to buffer		*/
	fd_set fdvar;
	struct timeval t;
	int req;
	char *iobuffer; 
	int byte_in_buffer;
	int n;
	char rqstbuf[BUFSIZ];

	INIT_TRACE("RFIO_TRACE");
	TRACE(1, "rfio", "rfio_read_v3(%d, %x, %d)", ctrl_sock, ptr, size) ;

#if defined (CLIENTLOG)
	/* Client logging */
	rfio_logrd(ctrl_sock,size);
#endif

	/*
	 * The file is local.
	 */
	if ( ctrl_sock<0 || ctrl_sock>= MAXRFD || rfilefdt[ctrl_sock] == NULL) {
		TRACE(2, "rfio", "rfio_read_v3: using local read(%d, %x, %d)", ctrl_sock, ptr, size);
		status = read(ctrl_sock, ptr, size);
		END_TRACE();
		rfio_errno = 0;
		return(status);
	}

	/*
	 * Checking magic number.
	 */
	if (rfilefdt[ctrl_sock]->magic != RFIO_MAGIC) {
		serrno = SEBADVERSION ; 
		(void) close(ctrl_sock) ;
		free((char *)rfilefdt[ctrl_sock]);
		rfilefdt[ctrl_sock] = NULL ;
		END_TRACE();
		return(-1);
	}

	if (rfilefdt[ctrl_sock]->first_read)
	  {
	    rfilefdt[ctrl_sock]->first_read = 0;
	    rfilefdt[ctrl_sock]->eof_received = 0;
	    /*
	     * Sending request using control socket.
	     */
	    p = rfio_buf;
	    marshall_WORD(p,RFIO_MAGIC);
	    marshall_WORD(p,RQST_READ_V3);
	    
	    TRACE(2, "rfio", "rfio_read_v3: sending %d bytes",RQSTSIZE) ;
	    if (netwrite(ctrl_sock, rfio_buf, RQSTSIZE) != RQSTSIZE) {
	      TRACE(2,"rfio","rfio_read_v3: write(): ERROR occured (errno=%d)",errno) ;
	      END_TRACE() ;
	      return -1 ;
	    }

	    TRACE(2,"rfio", "rfio_read_v3: reading %d bytes",RQSTSIZE) ;
	    if ((n = netread(ctrl_sock,rqstbuf,RQSTSIZE)) != RQSTSIZE) {
	      if (n == 0)
		{
		  return(-1);
		}
	      else
		{
		  TRACE(2, "rfio","read ctrl socket: read(): %s\n", sys_errlist[errno]);
		  END_TRACE();
		  return -1 ;
		}
	    }
	    p = rqstbuf;
	    unmarshall_WORD(p,req) ;			
	    unmarshall_LONG(p,rfilefdt[ctrl_sock]->filesize) ;
	    TRACE(2,"rfio", "rfio_read_v3: filesize is %d bytes",rfilefdt[ctrl_sock]->filesize) ;
	  }


	iobuffer = ptr;
	byte_in_buffer = 0;

	while (1)
	  {
	    /* EOF received previously but not all data read */
	    if (rfilefdt[ctrl_sock]->eof_received) {
	      if (rfilefdt[ctrl_sock]->filesize == rfilefdt[ctrl_sock]->byte_read_from_network)
		{
		  /* End of file and all data has been readby the client */
		  TRACE(2,"rfio","rfio_read: request satisfied eof encountered (read returns %d)",byte_in_buffer);
		  END_TRACE();
		  return(byte_in_buffer);
		}
	      else
		{
		  int to_be_read;

		  to_be_read = rfilefdt[ctrl_sock]->filesize - rfilefdt[ctrl_sock]->byte_read_from_network;

		  TRACE(2,"rfio","filesize=%d,byte_read=%d,size=%d,buffer=%d,toberead=%d",rfilefdt[ctrl_sock]->filesize,rfilefdt[ctrl_sock]->byte_read_from_network,size,byte_in_buffer,to_be_read);
		  if (size-byte_in_buffer <= to_be_read)
		    {
		      /* Receiving data using data socket */
		      TRACE(2,"rfio","datarfio_read_v3: reading %d bytes from datasocket filedesc=%d",size-byte_in_buffer,rfilefdt[ctrl_sock]->lseekhow) ;
		      if ((n = netread(rfilefdt[ctrl_sock]->lseekhow, iobuffer, size-byte_in_buffer)) != size-byte_in_buffer) {
			if (n == 0)
			  {
			    return(-1);
			  }
			else
			  {
			    TRACE(2,"rfio","datarfio_read_v3: read(): ERROR occured (errno=%d)",errno) ;
			    END_TRACE() ;
			    return -1 ;
			  }
		      }
		      rfilefdt[ctrl_sock]->byte_read_from_network += (size - byte_in_buffer);
		      byte_in_buffer = size;
		      TRACE(2,"rfio","rfio_read_v3: request satisfied after eof met"); ;
		      END_TRACE();
		      return(size);
		    }
		  else {
		    if (size-byte_in_buffer > to_be_read)
		      {
			/* Receiving data using data socket */
			TRACE(2,"rfio","datarfio_read_v3: reading %d bytes from datasocket (to_be_read)",to_be_read,rfilefdt[ctrl_sock]->lseekhow) ;
			if ((n = netread(rfilefdt[ctrl_sock]->lseekhow, iobuffer, to_be_read)) != to_be_read) {
			  if (n == 0)
			    {
			      return(-1);
			    }
			  else
			    {
			      TRACE(2,"rfio","datarfio_read_v3: read(): ERROR occured (errno=%d)",errno) ;
			      END_TRACE() ;
			      return -1 ;
			    }
			}
			rfilefdt[ctrl_sock]->byte_read_from_network += to_be_read;
			byte_in_buffer += to_be_read;
			END_TRACE();
			TRACE(2,"rfio","rfio_read_v3: request partially satisfied : %d bytes",byte_in_buffer);
			return(byte_in_buffer);
		      }
		  }
		}
	    }
	    else
	      {
		FD_ZERO(&fdvar);
		FD_SET(ctrl_sock,&fdvar);
		FD_SET(rfilefdt[ctrl_sock]->lseekhow,&fdvar);
		t.tv_sec = 10;
		t.tv_usec = 0;

		TRACE(2,"rfio","read_v3: doing select") ;
		if (select(FD_SETSIZE,&fdvar,NULL,NULL,&t) < 0)
		  {
		    TRACE(2,"rfio","read_v3: select failed (errno=%d)",errno) ;
		    END_TRACE() ;
		    return -1 ;
		  }
	    
		if (FD_ISSET(ctrl_sock,&fdvar))
		  {
		    int cause,rcode;
		    int n;
		    char rqstbuf[BUFSIZ];
		    
		    /* Something received on the control socket */
		    TRACE(2,"rfio", "ctrl socket: reading %d bytes",RQSTSIZE) ;
		    if ((n = netread(ctrl_sock,rqstbuf,RQSTSIZE)) != RQSTSIZE) {
		      if (n == 0)
			{
			  return(-1);
			}
		      else
			{
			  TRACE(2, "rfio","read ctrl socket: read(): %s", sys_errlist[errno]);
			  END_TRACE();
			  return -1 ;
			}
		    }
		    p = rqstbuf;
		    unmarshall_WORD(p,cause) ;			
		    unmarshall_LONG(p,status) ;
		    unmarshall_LONG(p,rcode) ;
		    if (cause == REP_ERROR)
		      {
			TRACE(2,"rfio", "read_v3: reply error status %d, rcode %d", status, rcode) ;
			rfio_errno = rcode;

			TRACE(2,"rfio","read_v3: sending ack for error") ;
			if (netwrite(ctrl_sock, rqstbuf, RQSTSIZE) != RQSTSIZE) {
			  TRACE(2,"rfio","read_v3: write(): ERROR occured (errno=%d)",errno) ;
			  END_TRACE() ;
			  return -1 ;
			}

			END_TRACE();
			return(-1);
		      }
		    if (cause == REP_EOF)
		      {
			rfilefdt[ctrl_sock]->eof_received = 1;
			TRACE(2,"rfio", "read_v3: eof received") ;
		      }
		  }

		if (FD_ISSET(rfilefdt[ctrl_sock]->lseekhow,&fdvar))
		  {
		    /* Receiving data using data socket */
			/* Do not use read here because NT doesn't support that with socket fds */
		    if ((n = s_nrecv(rfilefdt[ctrl_sock]->lseekhow, iobuffer, size-byte_in_buffer)) <= 0) {
		      if (n == 0)
			{
			  return(-1);
			}
		      else
			{
			  TRACE(2,"rfio","datarfio_read_v3: read(): ERROR occured (errno=%d)",errno) ;
			  END_TRACE() ;
			  return -1 ;
			}
		    }
		    byte_in_buffer += n;
		    rfilefdt[ctrl_sock]->byte_read_from_network += n;
		    iobuffer += n;

		    TRACE(2,"rfio","rfio_read: receiving datasocket=%d,buffer=%d,req=%d",n,byte_in_buffer,size) ;

		    if (byte_in_buffer == size)
		      {
			TRACE(2,"rfio","rfio_read: request satisfied completely"); ;
			END_TRACE();
			return(size);
		      }

		  }
	      }
	  }
      }



/*
 * Remote file write
 */
int rfio_write_v3(ctrl_sock, ptr, size) 
char    *ptr;
int     ctrl_sock, size;
{
	int status ;	/* Return code of called func	*/
	char   * p ; 	/* Pointer to buffer		*/
	fd_set fdvar;
	struct timeval t;
	
	INIT_TRACE("RFIO_TRACE");
	TRACE(1, "rfio", "rfio_write_v3(%d, %x, %d)", ctrl_sock, ptr, size) ;

#if defined (CLIENTLOG)
	/* Client logging */
	rfio_logwr(ctrl_sock,size);
#endif

	/*
	 * The file is local.
	 */
	if ( ctrl_sock<0 || ctrl_sock>= MAXRFD || rfilefdt[ctrl_sock] == NULL) {
		TRACE(2, "rfio", "rfio_write_v3: using local write(%d, %x, %d)", ctrl_sock, ptr, size);
		status = write(ctrl_sock, ptr, size);
		END_TRACE();
		rfio_errno = 0;
		return(status);
	}

	/*
	 * Checking magic number.
	 */
	if (rfilefdt[ctrl_sock]->magic != RFIO_MAGIC) {
		serrno = SEBADVERSION ; 
		(void) close(ctrl_sock) ;
		free((char *)rfilefdt[ctrl_sock]);
		rfilefdt[ctrl_sock] = NULL ;
		END_TRACE();
		return(-1);
	}

	if (rfilefdt[ctrl_sock]->first_write)
	  {
	    rfilefdt[ctrl_sock]->first_write = 0;

	    /*
	     * Sending request using control socket.
	     */
	    p = rfio_buf;
	    marshall_WORD(p,RFIO_MAGIC);
	    marshall_WORD(p,RQST_WRITE_V3);
	    
	    TRACE(2, "rfio", "rfio_write_v3: sending %d bytes",RQSTSIZE) ;
	    if (netwrite(ctrl_sock, rfio_buf, RQSTSIZE) != RQSTSIZE) {
	      TRACE(2,"rfio","rfio_write_v3: write(): ERROR occured (errno=%d)",errno) ;
	      END_TRACE() ;
	      return -1 ;
	    }

	  }
	
	FD_ZERO(&fdvar);
	FD_SET(ctrl_sock,&fdvar);

	t.tv_sec = 0; 
	t.tv_usec = 0;

	TRACE(2,"rfio","write_v3: doing select");
	/* Immediate return here to send data as fast as possible */
	if (select(FD_SETSIZE,&fdvar,NULL,NULL,&t) < 0)
	  {
	    TRACE(2,"rfio","write_v3: select failed (errno=%d)",errno) ;
	    END_TRACE() ;
	    return -1 ;
	  }
	    
	if (FD_ISSET(ctrl_sock,&fdvar))
	  {
	    int cause,rcode;
	    int n;
	    char rqstbuf[BUFSIZ];

	    /* Something received on the control socket */
	    TRACE(2,"rfio", "ctrl socket: reading %d bytes",RQSTSIZE) ;
	    if ((n = netread(ctrl_sock,rqstbuf,RQSTSIZE)) != RQSTSIZE) {
	      if (n == 0)
		{
		  return(-1);
		}
	      else
		{
		  TRACE(2, "rfio","read ctrl socket: read(): %s\n", sys_errlist[errno]);
		  END_TRACE();
		  return -1 ;
		}
	    }
	    p = rqstbuf;
	    unmarshall_WORD(p,cause) ;			
	    unmarshall_LONG(p,status) ;
	    unmarshall_LONG(p,rcode) ;
	    if (cause == REP_ERROR)
	      TRACE(2,"rfio", "write_v3: reply error status %d, rcode %d", status, rcode) ;
	    else
	      TRACE(2,"rfio", "write_v3: unknown error status %d, rcode %d", status, rcode) ;
	    rfio_errno = rcode;

	    TRACE(2,"rfio","rfio_write: sending ack for error") ;
	    if (netwrite(ctrl_sock, rqstbuf, RQSTSIZE) != RQSTSIZE) {
	      TRACE(2,"rfio","rfio_write_v3: write(): ERROR occured (errno=%d)",errno) ;
	      END_TRACE() ;
	      return -1 ;
	    }

	    END_TRACE();
	    return(-1);
	  }

	/* Sending data using data socket */
	TRACE(2,"rfio","rfio_write: sending %d bytes to datasocket filedesc=%d",size,rfilefdt[ctrl_sock]->lseekhow) ;
	if (netwrite(rfilefdt[ctrl_sock]->lseekhow, ptr, size) != size) {
	  TRACE(2,"rfio","datarfio_write_v3: write(): ERROR occured (errno=%d)",errno) ;
	  END_TRACE() ;
	  return -1 ;
	}

	rfilefdt[ctrl_sock]->byte_written_to_network += size;
	END_TRACE();
	return(size);
}

/* Close part */
/*
 * remote file close
 */
int rfio_close_v3(s)    
int     s;
{
  int req;
  char   * p  ; 
  struct {
    unsigned int    rcount; /* read() count                 */
    unsigned int    wcount; /* write() count                */
    unsigned int    rbcount;/* byte(s) read                 */
    unsigned int    wbcount;/* byte(s) written              */
  } iostatbuf ;
  int rcode,status;
  struct timeval t;
  fd_set fdvar;
  char dummy[128 * 1024];
  int n;

  INIT_TRACE("RFIO_TRACE");
  TRACE(1, "rfio", "rfio_close_v3(%d)", s);

  /*
   * The file is local
   */
  if ( s < 0 || s >= MAXRFD || rfilefdt[s] == NULL ) {
    TRACE(2, "rfio", "rfio_close_v3: using local close(%d)",s) ; 
    status= close(s) ; 
#if defined (CLIENTLOG)
    /* Client logging */
    rfio_logcl(s);
#endif
    END_TRACE() ; 
    rfio_errno = 0;
    return status ;
  }
#if defined (CLIENTLOG)
  /* Client logging */
  rfio_logcl(s);
#endif
  /*
   * Checking magic number
   */
  if ( rfilefdt[s]->magic != RFIO_MAGIC ) {
    serrno = SEBADVERSION ;
    (void) close(s) ;
    free((char *)rfilefdt[s]);
    rfilefdt[s] = 0;
    END_TRACE();
    return(-1);
  }
  
  /*
   * Sending request.
   */
  p= rfio_buf ;
  marshall_WORD(p, RFIO_MAGIC);
  marshall_WORD(p, RQST_CLOSE_V3);
  marshall_LONG(p, rfilefdt[s]->byte_written_to_network);
  TRACE(2, "rfio", "rfio_close_v3: sending %d bytes",RQSTSIZE) ;
  if (netwrite(s, rfio_buf,RQSTSIZE) != RQSTSIZE) {
    TRACE(2, "rfio", "rfio_close_v3: write(): ERROR occured (errno=%d)", errno);
    (void) rfio_cleanup_v3(s) ;
    END_TRACE() ;
    return -1 ;
  }

  while (1) 
    {
      FD_ZERO(&fdvar);
      FD_SET(s,&fdvar);
      FD_SET(rfilefdt[s]->lseekhow,&fdvar);
      
      t.tv_sec = 10;
      t.tv_usec = 0;

      TRACE(2,"rfio","close_v3: doing select") ;
      if (select(FD_SETSIZE,&fdvar,NULL,NULL,&t) < 0)
	{
	  TRACE(2,"rfio","close_v3: select failed (errno=%d)",errno) ;
	  END_TRACE() ;
	  return -1 ;
	}
      
      /* CLOSE confirmation received from server */
      if (FD_ISSET(s,&fdvar))
	{
	  /*
	   * Getting data from the network.
	   */
	  
	  TRACE(2, "rfio", "rfio_close: reading %d bytes",RQSTSIZE) ; 
	  if ((n = netread(s,rfio_buf,RQSTSIZE)) != RQSTSIZE) {
	    if (n == 0)
	      {
		return(-1);
	      }
	    else
	      {
		TRACE(2, "rfio", "rfio_close_v3: read(): ERROR occured (errno=%d)", errno);
		(void)rfio_cleanup_v3(s) ; 
		END_TRACE() ;
		return -1 ; 
	      }
	  }
	  
	  /* Closing data socket after the server has read all the data */
	  TRACE(2, "rfio", "rfio_close_v3 closing data socket, fildesc=%d",rfilefdt[s]->lseekhow) ; 
	  if (s_close(rfilefdt[s]->lseekhow) < 0)
	    TRACE(2, "rfio", "rfio_close_v3: close(): ERROR occured (errno=%d)", errno);
	  
	  p = rfio_buf ;
	  unmarshall_WORD(p,req) ;
	  unmarshall_LONG(p,status) ;
	  unmarshall_LONG(p, rcode) ;
	  
	  rfio_errno = rcode ;
	  switch(req) {
	  case RQST_CLOSE_V3:
	    (void) rfio_cleanup_v3(s) ; 
	    TRACE(1, "rfio", "rfio_close_v3: return status=%d, rcode=%d",status,rcode) ;
	    END_TRACE() ; 
	    return status ;
          case REP_ERROR: /* It is possible to receive an error reply from the server write_v3 routine */
			  /* This is a bug fix for the internal rfio error returned by close_v3 when
			     write_v3 on the server side returns 'not enough space' while the client
		             has executing the close_v3 routine */
            (void) rfio_cleanup_v3(s) ;
            TRACE(1, "rfio", "rfio_close_v3: return status=%d, rcode=%d",status,rcode) ;
            END_TRACE() ;
            return status ;
	  default:
	    TRACE(1,"rfio","rfio_close_v3(): Bad control word received\n") ; 
	    rfio_errno= SEINTERNAL ;
	    (void) rfio_cleanup_v3(s) ; 
	    END_TRACE() ; 
	    return -1 ;
	  }
	}

      /* Server is still sending data through data socket */
      /* Reading data socket to unstuck the server which then will
	 be able to handle the CLOSE request we have sent previously */
      if (FD_ISSET(rfilefdt[s]->lseekhow,&fdvar))
	{	    
	  TRACE(2, "rfio", "rfio_close_v3: emptying data socket") ;

	  /*  Do not use read here as NT doesn't support this with socket fds */
	  if ((n = s_nrecv(rfilefdt[s]->lseekhow,dummy,sizeof(dummy))) <= 0) {
	    if (n < 0) /* 0 = continue (data socket closed, waiting for close ack */
	      {
		TRACE(2,"rfio","close_v3: read failed (errno=%d)",errno) ;
		END_TRACE() ;
		return -1 ;
	      }
	    else
	       TRACE(2, "rfio", "rfio_close_v3: emptying data socket, %d bytes read from data socket",n) ;

	  }
	}
    }
}


extern char     *getenv();      /* get environmental variable value     */

static char     last_host[256]; /* to hold last connect host name       */

static int     data_rfio_connect(node,remote,port,flags)       /* Connect <node>'s rfio server */
char    *node;                  /* remote host to connect               */
int     *remote  ;              /* connected host is remote or not      */
int port;
{
  
	register int    s;      /* socket descriptor                    */
	struct hostent  *hp;    /* host entry pointer                   */
	struct sockaddr_in sin; /* socket address (internet) struct     */
	char    *host;          /* host name chararcter string          */
	char    *p, *cp;        /* character string pointers            */
	register int    retrycnt; /* number of NOMORERFIO retries       */
	register int    retryint; /* interval between NOMORERFIO retries*/
	register int    crtycnt = 0; /* connect retry count             */
	register int    crtyint = 0; /* connect retry interval          */
	register int    crtyattmpt = 0; /* connect retry attempts done  */
	register int    crtycnts = 0 ;
	struct    stat statbuf ; /* NOMORERFIO stat buffer              */
	char    nomorebuf1[BUFSIZ], nomorebuf2[BUFSIZ]; /* NOMORERFIO buffers */
	int setsock_ceiling = 1 * 1024 * 1024;
	int max_rcvbuf;
	int max_sndbuf;
	int i;
#if !defined(_WIN32)
	int optlen,maxseg;
#endif
/*	int max_rcvlow; */

	INIT_TRACE("RFIO_TRACE");
/*
 * Should we use an alternate name ?
 */

/*
 * Under some circumstances (heavy load, use of inetd) the server fails
 * to accept a connection. A simple retry mechanism is therefore
 * implemented here.
 */
	if ( rfioreadopt(RFIO_NETRETRYOPT) != RFIO_NOTIME2RETRY ) {
		if ((p = getconfent("RFIO", "CONRETRY", 0)) != NULL)        {
			if ((crtycnt = atoi(p)) <= 0)     {
				crtycnt = 0;
			}
		}
		serrno = 0 ;
		if ((p = getconfent("RFIO", "CONRETRYINT", 0)) != NULL)        {
			if ((crtyint = atoi(p)) <= 0)     {
				crtyint = 0;
			}
		}
	}
	crtycnts = crtycnt ;
/*
 * When the NOMORERFIO file exists, or if NOMORERFIO.host file exists,
 * the RFIO service is suspended. By default it will retry for ever every
 * DEFRETRYINT seconds.
 */
	if ((p=getconfent("RFIO", "RETRY", 0)) == NULL) {
		retrycnt=DEFRETRYCNT;
	}
	else    {
		retrycnt=atoi(p);
	}
	if ((p=getconfent("RFIO", "RETRYINT", 0)) == NULL) {
		retryint=DEFRETRYINT;
	}
	else    {
		retryint=atoi(p);
	}

	if ( rfioreadopt(RFIO_NETOPT) != RFIO_NONET ) {
		if ((host = getconfent("NET",node,1)) == NULL)  {
			host = node;
		}
		else {
			TRACE(3,"rfio","set of hosts: %s",host);
		}
	}
	else    {
		host = node;
	}

	serrno = 0; /* reset the errno could be SEENTRYNFND */
	rfio_errno = 0;

	TRACE(1, "rfio", "datarfio_connect: connecting(%s) using port %d",host,port);

	memset(&sin,0,sizeof(sin));

	sin.sin_family = AF_INET;
	sin.sin_port = htons(port);

	cp = strtok(host," \t") ;
        if (cp == NULL ) {
               TRACE(1,"rfio","host specified incorrect");
               serrno = SENOSHOST;
               END_TRACE();
               return(-1);
        }

conretryall:
	TRACE(2, "rfio", "datarfio_connect: gethostbyname(%s)", cp);
	hp = gethostbyname(cp);
	if (hp == NULL) {
		if (strchr(cp,'.') != NULL) { /* Dotted notation */
			TRACE(2, "rfio", "datarfio_connect: using %s", cp);
			sin.sin_addr.s_addr = htonl(inet_addr(cp));
		}
		else    {
			TRACE(2, "rfio", "datarfio_connect: %s: no such host",cp);
			serrno = SENOSHOST;     /* No such host                 */
			END_TRACE();
			return(-1);
		}
	}
	else    {
		sin.sin_addr.s_addr = ((struct in_addr *)(hp->h_addr))->s_addr;
	}

#if defined(_WIN32)
	if (strncmp (NOMORERFIO, "%SystemRoot%\\", 13) == 0 &&
	    (p = getenv ("SystemRoot")))
		sprintf (nomorebuf1, "%s%s", p, strchr (NOMORERFIO, '\\'));
#else
	strcpy(nomorebuf1, NOMORERFIO);
#endif
	sprintf(nomorebuf2, "%s.%s", nomorebuf1, cp);
retry:
	if (!stat(nomorebuf1,&statbuf)) {
		if (retrycnt-- >=0)  {
			syslog(LOG_ALERT, "rfio: dataconnect: all RFIO service suspended (pid=%d)\n", getpid());
			sleep(retryint);
			goto retry;
		} else {
			syslog(LOG_ALERT, "rfio: dataconnect: all RFIO service suspended (pid=%d), retries exhausted\n", getpid());
			serrno=SERTYEXHAUST;
			return(-1);
		}
	}
	if (!stat(nomorebuf2, &statbuf)) {
		if (retrycnt-- >=0)  {
			syslog(LOG_ALERT, "rfio: dataconnect: RFIO service to <%s> suspended (pid=%d)\n", cp, getpid());
			sleep(retryint);
			goto retry;
		} else {
			syslog(LOG_ALERT, "rfio: dataconnect: RFIO service to <%s> suspended (pid=%d), retries exhausted\n", cp, getpid());
			serrno=SERTYEXHAUST;
			return(-1);
		}
	}

conretry:
	TRACE(2, "rfio", "rfio_dataconnect: socket(%d, %d, %d)",
				AF_INET, SOCK_STREAM, 0);
	if ((s = socket(AF_INET, SOCK_STREAM, 0)) < 0)  {
		TRACE(2, "rfio", "datarfio_connect: socket(): ERROR occured (errno=%d)", errno);
		END_TRACE();
		return(-1);
	}

	max_rcvbuf = setsock_ceiling;
	max_sndbuf = setsock_ceiling;

/* A bit ugly but this is the only solution because the posix
   flag O_ACCMODE is not defined under NT with MS C++ compiler */
#if defined(_WIN32)
#ifndef O_ACCMODE
#define O_ACCMODE 3
#endif
#endif
        /* Do the RCV_BUF setsockopt only if we are reading a file */
        /* This is a fix for observed performance problems */
        /* between DECs and SGIs using HIPPI */
        if ((flags & O_ACCMODE) == O_RDONLY)
          {
            for (i = setsock_ceiling ; i >= 16 * 1024 ; i >>= 1)
              {
                if (set_rcv_sockparam(s,i) == i)
                  {
                    max_rcvbuf = i;
                    break;
                  }
              }
            TRACE(2,"rfio","setsockopt maxrcv=%d",max_rcvbuf);
          }

        /* Do the SND_BUF setsockopt only if we are reading a file */
        /* This is a fix for observed performance problems */
        /* between DECs and SGIs using HIPPI */
        if ((flags & O_ACCMODE) == O_WRONLY)
          {
            for (i = setsock_ceiling ; i >= 16 * 1024 ; i >>= 1)
              {
                if (set_snd_sockparam(s,i) == i)
                  {
                    max_sndbuf = i;
                    break;
                  }
              }
            TRACE(2,"rfio","setsockopt maxsnd=%d",max_sndbuf);
          }

	TRACE(2, "rfio", "rfio_dataconnect: connect(%d, %x, %d)", s, &sin, sizeof(struct sockaddr_in));
	if (connect(s, (struct sockaddr *)&sin, sizeof(struct sockaddr_in)) < 0)   {
		TRACE(2, "rfio", "rfio_dataconnect: connect(): ERROR occured (errno=%d)", errno);
#if defined(_WIN32)
		if (errno == WSAECONNREFUSED)	{
#else
		if (errno == ECONNREFUSED)      {
#endif
			syslog(LOG_ALERT, "rfio: dataconnect: %d failed to connect %s", getpid(), cp);
			if (crtycnt-- > 0)       {
				if (crtyint) sleep(crtyint);
				syslog(LOG_ALERT, "rfio: dataconnect: %d retrying to connect %s", getpid(), cp);
/*
 * connect() returns "Invalid argument when called twice,
 * so socket needs to be closed and recreated first
 */

				(void) close(s);
				crtyattmpt ++ ;
				goto conretry;
			}
			if ( ( cp = strtok(NULL," \t")) != NULL ) 	{
				crtycnt =  crtycnts ;
				syslog(LOG_ALERT, "rfio: dataconnect: after ECONNREFUSED, changing host to %s", cp) ;
				TRACE(3,"rfio","rfio: dataconnect: after ECONNREFUSED, changing host to %s", cp) ;
				(void) close(s);
				goto conretryall;
			}
		}
#if defined(_WIN32)
		if (errno==WSAEHOSTUNREACH || errno==WSAETIMEDOUT)	{
#else
		if (errno==EHOSTUNREACH || errno==ETIMEDOUT)	{
#endif
			if ( ( cp = strtok(NULL," \t")) != NULL ) 	{
				crtycnt =  crtycnts ;
#if defined(_WIN32)
				if (errno == WSAEHOSTUNREACH)
#else
				if (errno == EHOSTUNREACH)
#endif
					syslog(LOG_ALERT, "rfio: dataconnect: after EHOSTUNREACH, changing host to %s", cp) ;
				else
					syslog(LOG_ALERT, "rfio: dataconnect: after ETIMEDOUT, changing host to %s", cp) ;
					
				(void) close(s);
				goto conretryall;
			}
				
		}
		(void) close(s);
		END_TRACE();
		return(-1);
	}
	if (crtyattmpt) {
		syslog(LOG_ALERT, "rfio: dataconnect: %d recovered connection after %d secs with %s",getpid(), crtyattmpt*crtyint,cp) ;
	}
	*remote = isremote( sin.sin_addr, node ) ;
	strcpy(last_host,cp); /* remember to fetch back remote errs     */
	TRACE(2, "rfio", "rfio_dataconnect: connected");
	TRACE(2, "rfio", "rfio_dataconnect: calling setnetio(%d)", s);
	if (setnetio(s) < 0)    {
		close(s);
		END_TRACE();
		return(-1);
	}


#if !defined(_WIN32)
	optlen = sizeof(maxseg);
	if (getsockopt(s,IPPROTO_TCP,TCP_MAXSEG,(char *)&maxseg,&optlen) < 0) {
	  TRACE(2,"rfio","srwrite_v3: getsockopt: ERROR occured (errno=%d)",errno) ;
	  END_TRACE() ;
	  return -1 ;
	}
	TRACE(2,"rfio","rfio_dataconnect: max TCP segment: %d",maxseg);
#endif
/*
	maxseg = 4312 * 30;
	optlen = sizeof(maxseg);
	if (setsockopt(s,SOL_SOCKET,SO_SNDLOWAT,(char *)&maxseg,optlen) < 0) {
	  TRACE(2,"rfio","dataconnect: setsockopt snd lowat: ERROR occured (errno=%d)",errno) ;
	}
	TRACE(2,"rfio","rfio_dataconnect: set snd low water: %d\n",maxseg);

	max_rcvlow = maxseg * (max_rcvbuf / maxseg);
	optlen = sizeof(max_rcvlow);
	if (setsockopt(s,SOL_SOCKET,SO_RCVLOWAT,(char *)&max_rcvlow,optlen) < 0) {
	  TRACE(2,"rfio","dataconnect: setsockopt rcv lowat: ERROR occured (errno=%d)",errno) ;
	}
	TRACE(2,"rfio","rfio_dataconnect: set rcv low water: %d\n",max_rcvlow);

	optlen = sizeof(max_rcvlow);
	if (getsockopt(s,SOL_SOCKET,SO_RCVLOWAT,(char *)&max_rcvlow,&optlen) < 0) {
	  TRACE(2,"rfio","srwrite_v3: getsockopt: ERROR occured (errno=%d)",errno) ;
	  END_TRACE() ;
	  return -1 ;
	}
	TRACE(2,"rfio","rfio_dataconnect: rcv low water: %d\n",max_rcvlow);

	optlen = sizeof(max_rcvlow);
	if (getsockopt(s,SOL_SOCKET,SO_RCVTIMEO,(char *)&max_rcvlow,&optlen) < 0) {
	  TRACE(2,"rfio","srwrite_v3: getsockopt: ERROR occured (errno=%d)",errno) ;
	  END_TRACE() ;
	  return -1 ;
	}
	TRACE(2,"rfio","rfio_dataconnect: rcv timeout water: %d\n",max_rcvlow);

	optlen = sizeof(max_rcvlow);
	if (getsockopt(s,SOL_SOCKET,SO_SNDTIMEO,(char *)&max_rcvlow,&optlen) < 0) {
	  TRACE(2,"rfio","srwrite_v3: getsockopt: ERROR occured (errno=%d)",errno) ;
	  END_TRACE() ;
	  return -1 ;
	}
	TRACE(2,"rfio","rfio_dataconnect: snd timeout water: %d\n",max_rcvlow);
*/	
	TRACE(1, "rfio", "rfio_dataconnect: return socket %d", s);
	END_TRACE();
	return(s);
}

static char *rfio_lasthost() /* returns last succesfully connected host     */
{
	return(last_host);
}


int set_rcv_sockparam(s,value)
int s,value;
{
  if (setsockopt(s,SOL_SOCKET,SO_RCVBUF,(char *)&value, sizeof(value)) < 0) {
#if defined(_WIN32)
    if (errno != WSAENOBUFS)
#else
    if (errno != ENOBUFS)
#endif
      {
	TRACE(2,"rfio", "setsockopt rcvbuf(): %s\n",sys_errlist[errno]);
	exit(1);
      }
    else
      return(-1);
  }
  else
    return(value);
}


int set_snd_sockparam(s,value)
int s,value;
{
  if (setsockopt(s,SOL_SOCKET,SO_SNDBUF,(char *)&value, sizeof(value)) < 0) {
#if defined(_WIN32)
    if (errno != WSAENOBUFS)
#else
    if (errno != ENOBUFS)
#endif
      {
	TRACE(2,"rfio", "setsockopt sndbuf(): %s\n",sys_errlist[errno]);
	exit(1);
      }
    else
      return(-1);
  }
  else
    return(value);
}
