/*
* Copyright (C) 1996-1997 by CERN/CN/PDP/DM
* All rights reserved
*/

#ifndef lint
static char sccsid[] = "@(#)rtstat.c	1.21 10/27/99 CERN CN-PDP/DM Claire Redmond";
#endif /* not lint */

#include <errno.h>
#include <fcntl.h>
#include <grp.h>
#include <malloc.h>
#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <time.h>
#include "../h/sacct.h"

#define MAXDGP 20		/* Maximum number of device groups */
#define MAXLEN 7		/* Current max length of device group names */
#define MAXSERVNAME 8		/* Current maximum length of server name */
#define MAXSERVS 40		/* Maximum number of servers */

/* Macro to swap byte order */

#define swap_it(a) {int __a; \
	swab((char *)&a, (char *)&__a, sizeof(a)); \
	a = (unsigned int)__a<<16 | ((unsigned int)__a>>16); }

extern char *optarg;	/* Optional command line arguments */
extern int optind;
#ifndef linux
extern char *sys_errlist[];
#endif /* linux */

struct stats {                  /* Structure to store the statistics info */
    int dev_succ;               /* number of successful developers requests */
    int dev_unsucc;             /* number unsuccessful */
    int usr_succ;               /* number of successful user requests */
    int usr_unsucc;             /* number unsuccessful */
    int usr_errors;		/* number of user errors */
    int tape_errors;		/* numver of tape errors */
    int shift_errors;		/* number of shift errors */
    int robot_errors;		/* number of robot errors */
    int network_errors;		/* number of network errors */
    int other_errors;		/* number of other errors */
    int unknown_errors;		/* number of unidentified errors */
    int no_mess;		/* failed requests with no matching error message */
};
struct stats global;

struct dev_grp {		/* Structure to store device group info */
    char dgn[MAXLEN];		/* device group name */
    char type[9];		/* device group type */
    struct stats stat;		/* statistics record */
    int *err_messages;		/* error messages for device group */
    struct dev_grp *next;	/* pointer to next record in list */
};
struct dev_grp *dev_first = NULL;
struct dev_grp *dev_prev = NULL;

struct failed {			/* Store info on failed requests with no message */
    time_t complete_time;	/* time of completion */
    char dgn[MAXLEN];		/* device group */
    char type[9];		/* device type */
    int jid;			/* job id */
    char server[MAXSERVNAME];	/* server name */
    struct failed *next;	/* pointer to next record */
};
struct failed *fail_first = NULL;
struct failed *fail_prev = NULL;

struct requests {		/* Structure to store request details */
    time_t complete_time;	/* time of completion */
    char dgn[MAXLEN];		/* device group */
    int err_lineno;		/* error line number */
    int exitcode;		/* exit code for command */
    char gid[9];		/* group id */
    int jid;			/* job id */
    int aut;			/* set to number of matches found in err file */
    struct requests *next;	/* next record in list */
    struct requests *prev;	/* pointer to previous record */
    time_t req_time;		/* time of request */
    int retry;			/* retry number for command */
    char server[MAXSERVNAME];	/* server name */
    char type[9];		/* device type */
    char unm[9];		/* unit name */
    char vid[7];		/* volume id */
};
struct requests *req_first = NULL;
struct requests *req_prev = NULL;

struct tape_errs {		/* structure to store tape error info */
    time_t first_time;		/* time of request */
    time_t last_time;		/* last error time */
    char unm[9];		/* unit name */
    char vid[7];		/* voulme id */
    int first_jid;		/* job id of first request with this error */
    int last_jid;		/* job id of last request with this error */
    int num_occs;		/* number of occurances */
    int mess_line;		/* line of error message */
    char server[MAXSERVNAME];	/* server name */
    struct tape_errs *next;	/* next record */
};
struct tape_errs *tape_first = NULL;
struct tape_errs *tape_prev = NULL;

struct tp_sort {
    time_t first_time;            /* time of request */
    time_t last_time;		/*last time of error */
    int first_jid;		/* job id of first request with this error */
    int last_jid;		/* job id of last request with this error */
    int num_occs;		/* number of occurances */
    int mess_line;		/* line of error message */
    char server[MAXSERVNAME];   /* server name */
    char unm[9];                /* unit name */
    char vid[7];                /* voulme id */
};
struct tp_sort *tperr_sort;      

struct server {			/* Structure to store information by server */
    int failed;			/* set if server read failed */
    char dgn[MAXLEN];		/* first device group on server */
    char type[9];		/* type running on server */
    struct server *next;   	/* pointer to next record in list */
    char server_name[MAXSERVNAME];/* server name */
    struct stats stat;		/* statistics record */
    int *err_messages;		/* error messages for server */
};
struct server *serv_first = NULL;
struct server *serv_prev = NULL;

struct store_dgn {		/* Structure to store the dgn for a jid */
    char dgn[MAXLEN];		/* device group name */
    int jid;			/* job id */
    struct store_dgn *next;	/* pointer to next record */
    struct store_dgn *prev;	/* pointer to previous record */
    char unm[9];		/* unit name */
    char vid[7];		/* voulme id */
};
struct store_dgn *store_first = NULL;
struct store_dgn *store_prev = NULL;

struct type {			/* Structure to store info by device type */
    int *err_messages;		/* error messages for device type */
    char type[9];		/* device type */
    struct stats stat;		/* statistics record */
    struct type *next;		/* pointer to next record in list */
};
struct type *type_first = NULL;
struct type *type_prev = NULL;

struct tpconfrec {		/* structure to store data from TPCONFIG file */
    char unm[9];		/* unit name */
    char dgn[9];		/* device group name */
    char dev_name[32];		/* device name */
    char dens[6];		/* density */
    char instat[5];		/* initial status */
    char loader[9];		/* manual or robot */
    char type[9];		/* device type */
};

struct err_mess {		/* structure to store error messages */
    char *message;		/* error message */
    int error_code;		/* type of error */
    int num_occs;		/* number of times the error occurs */
};
struct err_mess *emp;

struct unknown_errs {		/* structure to store unknown messages and details */
    char dgn[MAXLEN];		/* device group */
    char type[9];		/* device type */
    char *message;		/* error message */
    int num_occs;		/* number of occurences */
    char server_name[MAXSERVNAME];/* server */
    struct unknown_errs *next;	/* pointer to next record */
};
struct unknown_errs *unknown_first = NULL;
struct unknown_errs *unknown_prev = NULL;

int num_lines = 0;		/* number of lines in error messages file */
int num_tperrs = 0;		/* number of tape errors */

main (argc, argv) 
int argc;
char **argv;
{
    char acctfile[80];		/* accounting file path and name */
    char acctweek[80];		/* week number.. optional */
    char acctfile2[80];		/* input accounting file */
    struct accthdr accthdr;	/* accthdr record */
    char buf[256];		/* buffer */
    int c;			/* return from getopt function */
    time_t current_time;	/* current time */
    time_t cvt_datime();	/* date function */
    char dev_group[MAXDGP];	/* device group name */
    int eflag = 0;		/* set if endtime given */
    time_t endtime = 0;		/* time to end reading records */
    int errflg = 0;		/* error flag */
    int fd_acct;		/* file descriptor */
    int gflag = 0;		/* set if device group specified */
    int i;			/* counter */
    int nrec = 0;		/* number of records read */
    int num_serv_errs = 0;	/* number of servers on which open failed */
    struct accttape *rp;	/* pointer to accttape record */
    struct acctrtcp *rtcp;	/* pointer to acctrtcp record */
    char server_errors[MAXSERVS][9];/* list of servers on which open failed */
    int server_failed = 0;	/* set if server open has failed */
    struct server *serv_list;	/* pointer to server structure */
    char server_name[MAXSERVNAME];/* server name if specified */
    int Sflag = 0;		/* set if server specified */
    int sflag = 0;		/* set if starttime specified */
    time_t starttime = 0;	/* time to start reading records from */
    int sub_time = 0;		/* seconds to subtract */
    int swapped = 0;		/* set if byte order needs swapping */
    int vflag = 0;		/* set if verbose requested */
    int wflag = 0;		/* set if week number given */

/* set acctfile = NULL */

    acctfile[0] = '\0';
    acctfile2[0] = '\0';
    acctweek[0] = '\0';

/* Read in command line options */

    while ((c = getopt (argc, argv, "e:f:g:S:s:vw:")) != EOF) {
	switch (c) {
	    case 'e':		/* endtime */
		if ((endtime = cvt_datime (optarg)) < 0) {
		    fprintf (stderr, "incorrect time value %s\n", optarg);
		    errflg = 1;
		}
	        eflag++;
		break;
	    case 'f':		/* accounting file */
		strcpy (acctfile2, optarg);
		break;
	    case 'g':		/* device group */
		strcpy (dev_group, optarg);
		gflag++;
		break;
	    case 'S':		/* server */
		strcpy (server_name, optarg);
		Sflag++;
		break;
	    case 's':
		if (strncmp (optarg, "-", 1) == 0) {
		    sub_time = atoi (optarg+1);
		    time (&current_time);
		    if ((starttime = current_time - sub_time) <= 0) {
			fprintf (stderr, "starttime is a not valid %d\n", starttime);
			errflg = 1;
		    }
	    	}
		else if ((starttime = cvt_datime (optarg)) < 0) {
		    fprintf (stderr, "incorrect time value %s\n", optarg);
		    errflg = 1;
		}
		sflag++;
		break;
	    case 'v':
		vflag++;
		break;
	    case 'w':
		strcpy (acctweek, optarg);
		wflag++;
		break;
	    case '?':
		errflg = 1;
		break;
	}
    }

/* if invalid options or additional parameters given exit program with message */

    if (argc > optind) {	/* additional parameters */
	printf ("Additional parameters given\n");
	errflg = 1;
    }

    read_mess2mem();

/* create lists of required servers and device groups */

    create_dg_s_list (gflag, Sflag, dev_group, server_name);

    if (gflag && dev_first == NULL) {
	printf ("Incorrect device group name given :  %s\n", dev_group);
	errflg = 1;
    }
    else if (Sflag && serv_first == NULL) {
	printf ("Incorrect server given : %s\n", server_name);
	errflg = 1;
    }

    if (errflg) {			
	usage (argv[0]);
	exit (1);
    }

/* Determine the type of each device group by reading the TPCONFIG file and */
/* extracting the type field for each given device group. Create list of types */
/* sort server list into device type alphanumeric order */

    read_tpfile ();
    create_type();
    sort_servlist();

/* For each server in the list open the accounting file and read in the data */

    serv_list = serv_first;
    while (serv_list) {
	strcpy (acctfile, serv_list->server_name);
	strcat (acctfile, ":");
	if (acctfile2[0] != '\0')
	    strcat (acctfile, acctfile2);
	else
	    strcat (acctfile, ACCTFILE);
	if (wflag) {
	    strcat (acctfile, ".");
	    strcat (acctfile, acctweek);
	}

	server_failed = 0;
		
	if (vflag)
	    fprintf (stderr, "\nopening file : %s\n", acctfile);
	if ((fd_acct = rfio_open (acctfile, O_RDONLY)) < 0) {
	    fprintf (stderr, "%s : open error : %s\n", acctfile, rfio_serror());
	    server_failed = 1;
	    serv_list->failed = 1; 
	    strcpy (server_errors[num_serv_errs], serv_list->server_name);
	    num_serv_errs++;
	}
	nrec = 0;
	if (! server_failed) {
	    while (getacctrec (fd_acct, &accthdr, buf, &swapped)) { 
		if (accthdr.package != ACCTRTCOPY && accthdr.package != ACCTTAPE) 
		    continue;
		nrec++;
		if (!starttime &&nrec == 1) starttime = accthdr.timestamp;
		if (!endtime && nrec == 1) endtime = accthdr.timestamp;
		if (accthdr.timestamp < starttime) {
		    if (!sflag) starttime = accthdr.timestamp;
		    else continue;
		}
		if (endtime && accthdr.timestamp > endtime) {
		    if (!eflag) endtime = accthdr.timestamp;
		    else continue;
		}
 		if (accthdr.package == ACCTTAPE) {
		    rp = (struct accttape *) buf;	
		    if (swapped) {
			swap_fields (rp);
			swapped = 0;
		    }
		    store_dgn (rp);
		}
		else {
		   rtcp = (struct acctrtcp *) buf;
		   if (swapped) {
			swap_fields2 (rtcp);
			swapped = 0;
		   }
		   if (rtcp->subtype == RTCPCMDR) 
		   	create_reqrec (rtcp, accthdr.timestamp, serv_list);
		   else if (rtcp->subtype == RTCPCMDC)
		       match_reqrec (rtcp, accthdr.timestamp, serv_list, gflag , dev_group);
		}
	    }
	    clear_store();
	    rfio_close (fd_acct);
	    if (req_first != NULL)
		read_errfile (serv_list, vflag, wflag, acctweek); 
	}
	serv_list = serv_list->next;
    }

    print_stats (starttime, endtime, gflag, Sflag, wflag, acctweek); 
    if (tape_first != NULL)
	sort_tperrs();

    if (num_serv_errs > 0) {
	printf ("\n\n");
	for (i = 0; i < num_serv_errs; i++) 
	    printf ("Server %s failed, no information read\n",
		server_errors[i]);
    }
    

    if (errflg) exit(2);
    else        exit(0);
}

/********************************************************/
/* Function to read errfile and flag errors with retrys */
/********************************************************/

check_retry(fp)
int fp;
{
    char buf[500];              /* buffer */
    int c;                      /* return from read function */
    char *p;                    /* char pointer */
    char *q;                    /* char pointer */
    char savebuf[500];          /* buffer */
    int saveflag = 0;           /* flag set if part line in buffer */

    while ((c = rfio_read (fp, buf, sizeof (buf) - 1)) > 0) {
	buf[c] = 0;
	p = buf;
	if (saveflag) {
	    q = strchr (buf, '\n');
	    if (!q) {
		strcat (savebuf, p);
		continue;
 	    }

	    *q = '\0';
	    strcat (savebuf, p);
	    if (*savebuf != '\0' && *savebuf != ' ') 
		check_req (savebuf);
            p = q + 1;
	    saveflag = 0;
        }
        while (q = strchr (p, '\n')) {
            *q = '\0';
            if (*p == '\0' || *p == ' ') {
                p = q + 1;
                continue;
            }
            check_req (p);
            p = q + 1;
        }
        if (strlen (p)) {
            strcpy (savebuf, p);
            saveflag = 1;
        }
    }
    if (saveflag && (*p != '\0' || *p != ' '))
        check_req  (savebuf);
}

/*****************************************************/
/* Function to read line and match to request record */
/*****************************************************/

check_req(buf)
char *buf;
{
    time_t current_time;        /* current time */
    char day[4];                /* day of the week */
    int i = 0;                  /* counter */
    int jid;                    /* job id */
    int len = 0;                /* length of message */
    time_t mess_time;           /* time of error message */
    char month[4];              /* month */
    static char *months[] = {"Jan","Feb","Mar","Apr","May","Jun","Jul",
			     "Aug","Sep","Oct","Nov","Dec"};
    char *p;                    /* char pointer */
    char *p1;                   /* char pointer */
    struct tm t;                /* time structure */
    int year;                   /* year */
    struct requests *rq;	/* requests record */

    memset ((char *)&t, 0, sizeof (struct tm));
    time (&current_time);
    if (strstr (buf, "Automatic retry") != NULL) {
        p = strchr (buf, ' ');
        len = p - buf;
        strncpy (day, buf, len);
	day[3] = '\0';
        p1 = strchr (p + 1, ' ');
        len = p1 - (p + 1);
        strncpy (month, p + 1, len);
	month[3] = '\0';
        if (p1[1] == ' ') {
            if (p1[2] == ' ')
                p = p1 + 2;
            else
                p = p1 + 1;
        }
        else p = p1;
        t.tm_mday = atoi (p + 1);
        p1 = strchr (p + 1, ' ');
        t.tm_hour = atoi (p1 + 1);
        p = strchr (buf, ':');
        t.tm_min = atoi (p + 1);
        p1 = strchr (p + 1, ':');
        t.tm_sec = atoi (p1 + 1);
        p = strchr (p1, ' ');
        year = atoi (p + 1);
        p1 = strchr (p, '[');
        jid = atoi (p1 + 1);
        for (i = 0; i < 12; i++) {
            if (strcmp (month, months[i]) == 0) {
                t.tm_mon = i;
                break;
            }
        }
       t.tm_year = year - 1900;

#if defined(sun) && ! defined(SOLARIS)
    mess_time = timelocal (&t);
#else
    t.tm_isdst = -1;
    mess_time = mktime (&t);
#endif

    	rq = req_first;
	while (rq != NULL) {
	    if (jid == rq->jid && 
		(mess_time >= rq->req_time && mess_time <= rq->complete_time)) {
		rq->retry = 1;        
		break;
	    }
	    rq = rq->next;
	}
    }
}


/*******************************************************/
/* Function to clear jid and dgn store for each server */
/*******************************************************/

clear_store()
{
    struct store_dgn *sd;	/* pointer to store_dgn record */
    struct store_dgn *sd2;	/* pointer to store_dgn record */

    sd = store_first;
    while (sd) {
	sd2 = sd->next;
	free (sd);
	sd = sd2;
    }
    store_first = NULL;
    store_prev = NULL;
}

/***************************************/
/* Function to create a dev_grp record */
/***************************************/

create_devrec(d)
char *d;
{
    int dev_found = 0;          /* flag set if device found */
    struct dev_grp *dt;         /* pointer to device types record */

/* create device records for each device group that is not already in list */

    dt = dev_first;
    if (dt == NULL) {
	dt = (struct dev_grp *) calloc (1, sizeof (struct dev_grp));
	strcpy (dt->dgn, d);
        dt->err_messages = (int *) calloc (num_lines, sizeof (int));
	dt->next = NULL;
	dev_first = dt;
	dev_prev = dt;
    }
    else {
	while (dt) {
	    if (strcmp (d, dt->dgn) == 0) {
		dev_found = 1;
		break;
	    }
	    dt = dt->next;
	}
	if (!dev_found) {
	    dt = (struct dev_grp *) calloc (1, sizeof (struct dev_grp)) ;
	    strcpy (dt->dgn, d);
            dt->err_messages = (int *) calloc (num_lines, sizeof (int));
	    dt->next = NULL;
	    dev_prev->next = dt;
	    dev_prev = dt;
	}
    }
}

/*********************************************************/
/* Function to create lists of device groups and servers */
/*********************************************************/

create_dg_s_list (gflag, Sflag, dev_group, server_name)
int gflag;
int Sflag;
char dev_group[MAXLEN];
char server_name[MAXSERVNAME];
{
    char buf[128];              /* buffer */
    char *d;                    /* token pointer */
    char devgroup[MAXDGP];      /* current device group name */
    FILE *fp;                   /* file pointer */

/* open and read configuration file and extract the names of all devices currently */
/* in use, read the device group names into device_types list , and server names */
/* into server list . If device group specified on create record for that group */
/* and for the servers on which they run.  If server specified only create records */
/* for that server and the devices which run on it */

    if ((fp = fopen (CONFIGFILE, "r")) != NULL ) {
	fgets (buf, sizeof (buf), fp);
	do {
	    d = strtok (buf, " \t\n");
	    if (strcmp (d, "TPSERV") == 0 || strcmp (d, "TPSERVR") == 0 ||
	      strcmp (d, "TPSERVW") == 0) {
		d = strtok (NULL, " \t");
		if (strcmp (d, "CT1") == 0) continue;
		if (gflag && (strcmp (d, dev_group)) != 0) continue;

		if (!Sflag) create_devrec(d);
		
		strcpy (devgroup, d);
		d = strtok (NULL, " \t\n");
		while (d) {
		    if (Sflag && (strcmp (d, server_name)) != 0) {
			d = strtok (NULL, " \t\n");
			continue;
		    }

		    if (Sflag) {
			create_servrec(d, devgroup);
			create_devrec(devgroup);
		    }
		    else 
			create_servrec(d, devgroup);

		    d = strtok (NULL, " \t\n");
		}
	    }
	} while (fgets (buf, sizeof (buf), fp) != NULL);
    }
   fclose (fp);
}

/*****************************************************************************/
/* Function to create list of failed requests with no matching error message */
/*****************************************************************************/

create_failed (rq, server)
struct requests *rq;
struct server *server;
{
    struct failed *fd;		/* pointer to failed sturcture */

    fd = fail_first;
    if (fd == NULL) {
	fd = (struct failed *) calloc (1, sizeof (struct failed));
	fd->jid = rq->jid;
	strcpy (fd->server, server->server_name);
 	strcpy (fd->dgn, rq->dgn);
	strcpy (fd->type, rq->type);
	fd->complete_time = rq->complete_time;
	fd->next = NULL;
	fail_first = fd;
	fail_prev = fd;
    }
    else {
	fd = (struct failed *) calloc (1, sizeof (struct failed));
        fd->jid = rq->jid;
        strcpy (fd->server, server->server_name);
 	strcpy (fd->dgn, rq->dgn);
	strcpy (fd->type, rq->type);
        fd->complete_time = rq->complete_time;
	fd->next = NULL;
	fail_prev->next = fd;
	fail_prev = fd;
    }
}
	
/*************************************/
/* Functions to create request record */
/*************************************/

create_reqrec (rtcp, timestamp, serv)
struct acctrtcp *rtcp;
time_t timestamp;
struct server *serv;
{
    struct requests *rq;	/* pointer to requests record */
    struct group *grp;		/* pointer to group record */

    rq = req_first;
    if (rq == NULL) {
	rq = (struct requests *) calloc (1, sizeof (struct requests));
	rq->prev = NULL;
	req_first = rq;
	req_prev = rq;
    }
    else {
	rq = (struct requests *) calloc (1, sizeof (struct requests));
	rq->prev = req_prev;
	req_prev->next = rq;
	req_prev = rq;
    }
    rq->next = NULL;
    rq->jid = rtcp->jid;
   
    if ( (grp = getgrgid (rtcp->gid)) != NULL )
    	strcpy (rq->gid, grp->gr_name);
    else sprintf(rq->gid,"%d",rtcp->gid);

    strcpy (rq->server, serv->server_name);

    if (rtcp->subtype == RTCPCMDR) rq->req_time = timestamp;
    else rq->complete_time = timestamp;
}

/************************************/
/* Function to create server record */
/************************************/

create_servrec(d, devgroup)
char *d;
char devgroup[MAXDGP];
{
    int serv_found = 0;		/* set if server already in list */
    struct server *svd;		/* pointer to server datat record */

/* create server_data record for each server that is not already in list */
    
    svd = serv_first;
    if (svd == NULL) {
	svd = (struct server *) calloc (1, sizeof (struct server));
	strcpy (svd->server_name, d);
	strcpy (svd->dgn, devgroup);
	svd->next = NULL;
	serv_first = svd;
	serv_prev = svd;
    }
    else {
	while (svd) {
	    if (strcmp (svd->server_name,d) == 0) {
		serv_found = 1;
		break;
	    }
	    svd = svd->next;
	}
	if (!serv_found) {
	    svd = (struct server *) calloc (1, sizeof (struct server));
	    strcpy (svd->server_name, d);
	    strcpy (svd->dgn, devgroup); 
	    svd->next = NULL;
	    serv_prev->next = svd;
	    serv_prev = svd;
	}
    }
}

/**********************************/
/* Function to create type record */
/**********************************/

create_type()
{
    struct dev_grp *dg;		/* pointer to dev_grp record */
    int found = 0;		/* flag set if record exists */
    struct server *sr;		/* pointer to server record */
    struct type *ty;		/* pointer to type record */

    dg = dev_first;
    ty = type_first;

/* for each type found in device group list, create a record */

    while (dg) {
	if (strcmp (dg->type, "") != 0) {
	    if (ty == NULL) {
	    	ty = (struct type *) calloc (1, sizeof (struct type));
	    	strcpy (ty->type, dg->type);
            	ty->err_messages = (int *) calloc (num_lines, sizeof (int));
	    	ty->next = NULL;
	    	type_first = ty;
	    	type_prev = ty;
	    }
	    else {
	    	ty = type_first;
	    	found = 0;
	    	while (ty) {
	   	    if (strcmp (ty->type, dg->type) == 0) {
		        found = 1;
		    	break;
		    }
		    ty = ty->next;
	    	}

	        if (!found) {
		    ty = (struct type *) calloc (1, sizeof (struct type));
		    strcpy (ty->type, dg->type);
                    ty->err_messages = (int *) calloc (num_lines, sizeof (int));
		    ty->next = NULL;
		    type_prev->next = ty;
		    type_prev = ty;
	        }
	    }
	}
	dg = dg->next;
    }

    sr = serv_first;
    while (sr) {
	dg = dev_first;
	while (dg) {
	    if (strcmp (dg->dgn, sr->dgn) == 0) {
	        strcpy (sr->type, dg->type);
		break;
	    }
	    dg = dg->next;
	}
	sr = sr->next;
    }
}

/****************************************************/
/* Function to create unknown error message  record */
/****************************************************/

create_unknown (rq, message, sr)
struct requests *rq;
char *message;
struct server *sr;
{
    int found = 0;		/* flag set if record exists */ 
    int len = 0;		/* length of message */
    struct unknown_errs *uk;	/* pointer to unknown error message structure */

    uk = unknown_first;
    if (uk == NULL) {
	uk = (struct unknown_errs *) calloc (1, sizeof (struct unknown_errs));	
	len = strlen (message) + 1;
	uk->message = malloc (len);
	strcpy (uk->message, message);
	strcpy (uk->server_name, sr->server_name);
	strcpy (uk->dgn, rq->dgn);
	strcpy (uk->type, rq->type);
	uk->num_occs++;
	uk->next = NULL;
	unknown_first = uk;
	unknown_prev = uk;
    }
    else {
	while (uk != NULL) {
	    if ((strcmp (uk->message, message) == 0) &&
		 (strcmp (uk->server_name, sr->server_name) == 0) &&
		 (strcmp (uk->dgn, rq->dgn) == 0) &&
		 (strcmp (uk->type, rq->type) == 0)) {
		uk->num_occs++;
		found = 1;
		break;
	    }
	    uk = uk->next;
	}
	if (!found) {
            uk = (struct unknown_errs *) calloc (1, sizeof (struct unknown_errs));  
            len = strlen (message) + 1;
            uk->message = malloc (len);
            strcpy (uk->message, message);
	    strcpy (uk->server_name, sr->server_name);
	    strcpy (uk->dgn, rq->dgn);
	    strcpy (uk->type, rq->type);
            uk->num_occs++;
            uk->next = NULL;
            unknown_prev->next = uk;
            unknown_prev = uk;
        }
    }
}

/**************************************************/
/* Function to convert date and time into seconds */
/**************************************************/

time_t
cvt_datime(arg)
char *arg;
{
    time_t current_time;
    static int lastd[12] = {31,29,31,30,31,30,31,31,30,31,30,31};
    int n;
    struct tm t;
    struct tm *tm;

    memset ((char *) &t, 0, sizeof(struct tm));
    time (&current_time);                /* Get current time */
    tm = localtime (&current_time);
    n = sscanf (arg, "%2d%2d%2d%2d%2d", &t.tm_mon, &t.tm_mday, &t.tm_hour,
		&t.tm_min, &t.tm_year);
    if (n < 4) return (-1);
    if (n == 4)
	t.tm_year = tm->tm_year;
    else if (t.tm_year >= 0 && t.tm_year <= 37)
	t.tm_year += 100;
    else if (t.tm_year >= 38 && t.tm_year <= 69)
	return (-1);
    if ((t.tm_mon < 1) || (t.tm_mon > 12)) return (-1);
    if ((t.tm_mon == 2) && (t.tm_mday == 29) && (t.tm_year % 4 != 0)) return (-1);
    if ((t.tm_mday < 1) || (t.tm_mday > lastd[t.tm_mon-1])) return (-1);
    t.tm_mon--;

#if defined(sun) && ! defined(SOLARIS)
    return (timelocal (&t));
#else
    t.tm_isdst = -1;
    return (mktime (&t));
#endif
}

/****************************************/
/* Function to delete a request record  */
/****************************************/

delete_reqrecord (rq)
struct requests *rq;
{
    if (rq->prev == NULL && rq->next == NULL) {
	req_first = NULL;
	req_prev = NULL;
    }
    else if (rq->prev == NULL && rq->next != NULL) {
	req_first = rq->next;
	rq->next->prev = NULL;
    }
    else if (rq->prev != NULL && rq->next == NULL) {
	req_prev = rq->prev;
	rq->prev->next = NULL;
    }
    else if (rq->prev != NULL && rq->next != NULL) {
	rq->prev->next = rq->next;
	rq->next->prev = rq->prev;
    }
    free (rq);
}

/*******************************************/
/* Function to delete a store_dgn  record  */
/*******************************************/

delete_storerecord (sd)
struct store_dgn *sd;
{
    if (sd->prev == NULL && sd->next == NULL) {
	store_first = NULL;
	store_prev = NULL;
    }
    else if (sd->prev == NULL && sd->next != NULL) {
	store_first = sd->next;
	sd->next->prev = NULL;
    }
    else if (sd->prev != NULL && sd->next == NULL) {
	store_prev = sd->prev;
	sd->prev->next = NULL;
    }
    else if (sd->prev != NULL && sd->next != NULL) {
	sd->prev->next = sd->next;
	sd->next->prev = sd->prev;
    }
    free (sd);
}

/********************************************************/
/* Function to read in records from the accounting file */
/********************************************************/

getacctrec (fd_acct, accthdr, buf, swapped)
int fd_acct;
struct accthdr *accthdr;
char *buf;
int *swapped;
{
   int c;

   c = rfio_read (fd_acct, accthdr, sizeof (struct accthdr));

   if (c != sizeof (struct accthdr)) {
      if (c == 0) return (0);
      if (c > 0) fprintf (stderr, "read returns %d\n", c);
      else fprintf (stderr, "read error : %s\n", sys_errlist[errno]);
      exit (2);
   }

/* if required swap byte order */

   if (accthdr->package > 255) {
	swap_it (accthdr->package);
	swap_it (accthdr->len);
	swap_it (accthdr->timestamp);
	*swapped = 1;
   }

   if (accthdr->len > 256) {
      fprintf (stderr, "corrupted accounting file\n");
      return (0);
   }
   c = rfio_read (fd_acct, buf, accthdr->len);

   if ( c != accthdr->len) {
      if (c >= 0) fprintf (stderr, "read returns %d\n", c);
      else fprintf (stderr, "read error : %s\n", sys_errlist[errno]);
      exit (2);
   }

   return(accthdr->len);
}

/********************************/
/* Function to get a stored dgn */
/********************************/

get_dgn (rq)
struct requests *rq;
{
    struct store_dgn *sd;
    struct dev_grp *dg;

    sd = store_first;
    while (sd) {
	if (sd->jid == rq->jid) {
	    strcpy (rq->dgn, sd->dgn);
	    strcpy (rq->unm, sd->unm);
	    strcpy (rq->vid, sd->vid);
	    delete_storerecord (sd);
	    break;
	}
	sd = sd->next;
    }

    dg = dev_first;
    while (dg) {
	if (strcmp (rq->dgn, dg->dgn) == 0 ) {
	    strcpy (rq->type, dg->type);
	    break;
	}
       dg = dg->next;
    }
}

/******************************************/
/* Function to get statistics return code */
/******************************************/

get_statcode (rq)
struct requests *rq;
{
    int code;		/* code for exit status */

    if (strcmp (rq->gid, "c3") == 0 || strcmp (rq->gid, "ct") == 0) {
	if (rq->exitcode == 0 || rq->exitcode == 193 || rq->exitcode == 194
                || rq->exitcode == 195 || rq->exitcode == 197)
	    code = 7;
	else
	    code = 8;
    }
    else {
	if (rq->exitcode == 0 || rq->exitcode == 193 || rq->exitcode == 194
		|| rq->exitcode == 195 || rq->exitcode == 197)
	    code = 9;
	else {
	    rq->exitcode = -1;
	    code = 0;
	}
    }
    return (code);
} 

/********************************************/
/* Function to insert statistics into lists */
/********************************************/

insert_stats (rq, sr, errs)
struct requests *rq;
struct server *sr;
int errs;
{
    int code;			/* code set for statistic evaluation */
    struct dev_grp *dg;		/* pointer to device group record */
    int group_match = 0;	/* set if no group match found */
    struct type *ty;		/* pointer to type record */
    int type_match = 0;		/* set if no type match found */

/* for each list group find then one that matches the current record */

    if (strcmp (rq->dgn, "") == 0) {
	group_match = 1;
  	strcpy (rq->type, sr->type);
    }
	   
   
    if (!group_match) {
	dg = dev_first;
	while (dg) {
	    if (strcmp (dg->dgn, rq->dgn) == 0) 
		break;
	    dg = dg->next;
	}
    }

    if (dg == NULL)
	group_match = 1;
 
    if (!type_match) {
	ty = type_first;
	while (ty) {
	    if (strcmp (ty->type, rq->type) == 0 )
		break;
	    ty = ty->next;
	}
    }

    if (ty == NULL)
	type_match = 1;

    if (rq->err_lineno != -1) {
	sr->err_messages[rq->err_lineno]++;
	if (!group_match)
	    dg->err_messages[rq->err_lineno]++;
	if (!type_match)
	    ty->err_messages[rq->err_lineno]++;
    }

/* get the code for this requests record */

    if (!errs)
	code = get_statcode (rq);
    else
	code = rq->exitcode;

/* set relevant statistics fields depending on return code */

    switch (code) {
	case 1:
	    global.usr_unsucc++;
	    global.usr_errors++;
	    sr->stat.usr_unsucc++;
	    sr->stat.usr_errors++;
	    if (!group_match) {
		dg->stat.usr_unsucc++;
		dg->stat.usr_errors++;
	    }
	    if (!type_match) {
		ty->stat.usr_unsucc++;
		ty->stat.usr_errors++;
	    }
	    break;
	case 2:
	    global.usr_unsucc++;
	    global.tape_errors++;
	    sr->stat.usr_unsucc++;
	    sr->stat.tape_errors++;
	    if (!group_match) {
		dg->stat.usr_unsucc++;
		dg->stat.tape_errors++;
	    }
	    if (!type_match) {
		ty->stat.usr_unsucc++;
		ty->stat.tape_errors++;
	    }
	    break;
	case 3:
	    global.usr_unsucc++;
	    global.shift_errors++;      
	    sr->stat.usr_unsucc++;
	    sr->stat.shift_errors++;
	    if (!group_match) {
		dg->stat.usr_unsucc++;
		dg->stat.shift_errors++;   
	    }
	    if (!type_match) {
		ty->stat.usr_unsucc++; 
		ty->stat.shift_errors++;   
	    }
	    break;     
	case 4:
	    global.usr_unsucc++;
	    global.robot_errors++;      
	    sr->stat.usr_unsucc++;
	    sr->stat.robot_errors++;
	    if (!group_match) { 
		dg->stat.usr_unsucc++;
		dg->stat.robot_errors++;   
	    }
	    if (!type_match) {
		ty->stat.usr_unsucc++; 
		ty->stat.robot_errors++;   
	    }
	    break;     
	case 5:
	    global.usr_unsucc++;
	    global.network_errors++;      
	    sr->stat.usr_unsucc++;
	    sr->stat.network_errors++;
	    if (!group_match) { 
		dg->stat.usr_unsucc++;
		dg->stat.network_errors++;   
	    }
	    if (!type_match) {
		ty->stat.usr_unsucc++; 
		ty->stat.network_errors++;   
	    }
	    break;     
	case 6:
	    global.usr_unsucc++;
	    global.other_errors++;      
	    sr->stat.usr_unsucc++;
	    sr->stat.other_errors++;
	    if (!group_match) { 
		dg->stat.usr_unsucc++;
		dg->stat.other_errors++;   
	    }
	    if (!type_match){   
		ty->stat.usr_unsucc++;
		ty->stat.other_errors++;   
	    }
	    break;     
	case 7:
	    global.dev_succ++;
	    sr->stat.dev_succ++;
	    if (!group_match) 
		dg->stat.dev_succ++;
	    if (!type_match)
		ty->stat.dev_succ++;
	    break;
	case 8:
	    global.dev_unsucc++;
	    sr->stat.dev_unsucc++;
	    if (!group_match) 
		dg->stat.dev_unsucc++;
	    if (!type_match )
		ty->stat.dev_unsucc++;
	    break;
	case 9:
	    global.usr_succ++;
	    sr->stat.usr_succ++;
	    if (!group_match) 
		dg->stat.usr_succ++;
	    if (!type_match)
		ty->stat.usr_succ++;
	    break;
        case 10:
            global.usr_unsucc++;
	    global.unknown_errors++;
            sr->stat.usr_unsucc++;
	    sr->stat.unknown_errors++;
            if (!group_match) {
                dg->stat.usr_unsucc++;
	 	dg->stat.unknown_errors++;
	    }
            if (!type_match) {
                ty->stat.usr_unsucc++;
		ty->stat.unknown_errors++;
	    }
            break;
	case 11:
            global.usr_unsucc++;
            global.no_mess++;
            sr->stat.usr_unsucc++;
            sr->stat.no_mess++;
            if (!group_match) {
                dg->stat.usr_unsucc++;
                dg->stat.no_mess++;
            }
            if (!type_match) {
                ty->stat.usr_unsucc++;
                ty->stat.no_mess++;
            }
            break;
   }


    if (code == 7 || code == 8 || code == 9) 
	delete_reqrecord (rq);
}

/*********************************************************************/
/* function to insert the device group type into the relevant record */
/*********************************************************************/

insert_type (p, serv)
char *p;
struct server *serv;
{
    struct dev_grp *dev_list;   /* pointer to device list */
    char *q;
    struct tpconfrec *tprec;    /* tpconfrec record */

/* Read the line into tprec, match dgn and insert type into record */

    tprec = (struct tpconfrec *) calloc (1, sizeof (struct tpconfrec));
    sscanf (p, "%s%s%s%s%s%s%s", tprec->unm, tprec->dgn, tprec->dev_name,
			  tprec->dens, tprec->instat, tprec->loader, tprec->type);
    if (q = strstr (tprec->type, "/VB")) *q = '\0';
    if (strcmp (tprec->type, "8505") == 0)
	strcpy (tprec->type, "8500");

    dev_list = dev_first;
    while (dev_list) {
	if (strcmp (dev_list->dgn, tprec->dgn) == 0 &&
	   (strcmp (dev_list->type, "") == 0)) {
	    strcpy (dev_list->type, tprec->type);
	    break;
	}
	else if (strcmp (dev_list->dgn, tprec->dgn) == 0 &&
		(strcmp (dev_list->type, "") != 0))
		break;
	dev_list = dev_list->next;
    }

	strcpy (serv->type, tprec->type);

    free (tprec);
}

/****************************/
/* Function to match errors */
/****************************/

match_errors (buf, rq, message, stop)
char *buf;
struct requests *rq;
char **message;
int *stop;
{
    time_t current_time;	/* current time */
    char day[4];		/* day of the week */
    int i = 0;			/* counter */
    int jid;			/* job id */
    int len = 0;		/* length of message */
    int match = 0;		/* set if matching record found */
    time_t mess_time;		/* time of error message */
    char month[4];		/* month */
    static char *months[] = {"Jan","Feb","Mar","Apr","May","Jun","Jul",
			     "Aug","Sep","Oct","Nov","Dec"};
    char *p;			/* char pointer */
    char *p1;			/* char pointer */
    struct tm t;		/* time structure */
    int year;			/* year */

    memset ((char *)&t, 0, sizeof (struct tm));
    time (&current_time);
    if ((strstr (buf, "rtcopyd") == NULL) && ((strstr (buf, "Mon") != NULL) ||
	(strstr (buf, "Tue") != NULL) || (strstr (buf, "Wed") != NULL) || 
	(strstr (buf, "Thu") != NULL) || (strstr (buf, "Fri") != NULL) || 
	(strstr (buf, "Sat") != NULL) || (strstr (buf, "Sun") != NULL))) {
	p = strchr (buf, ' ');
	len = p - buf;
	strncpy (day, buf, len);
	day[3] = '\0';
	p1 = strchr (p + 1, ' ');
	len = p1 - (p + 1);
	strncpy (month, p + 1, len);
 	month[3] = '\0';
	if (p1[1] == ' ') {
	    if (p1[2] == ' ')
		p = p1 + 2;
	    else
		p = p1 + 1;
	}
	else p = p1;
	t.tm_mday = atoi (p + 1);
	p1 = strchr (p + 1, ' ');
	t.tm_hour = atoi (p1 + 1);
    	p = strchr (buf, ':');
    	t.tm_min = atoi (p + 1);
    	p1 = strchr (p + 1, ':');
    	t.tm_sec = atoi (p1 + 1);
    	p = strchr (p1, ' ');
    	year = atoi (p + 1);
    	p1 = strchr (p, '[');
    	jid = atoi (p1 + 1);
	if (jid == rq->jid) {
    	    p = strchr (buf, ']');
    	    p1 = strchr (buf, '\0');
    	    len = p1 - p;
    	    *message = malloc (len);
    	    strncpy (*message, p + 1, len);

    	    for (i = 0; i < 12; i++) {
	    	if (strcmp (months[i], month) == 0) {
	    	    t.tm_mon = i;
	    	    break;
	    	}
    	    }
           t.tm_year = year - 1900;

#if defined(sun) && ! defined(SOLARIS)
    mess_time = timelocal (&t);
#else
    t.tm_isdst = -1;
    mess_time = mktime (&t);
#endif

    	    if (len <= 3)
		match = 0;
    	    else if (strstr (*message, "TMS not responding") != NULL) 
		match = 0;
    	    else if (strstr (*message, "Volume busy") != NULL) 
		match = 0;
	    else if (strstr (*message, "Volume in use") != NULL)
	    	match = 0;
	    else if (strstr (*message, "New path") != NULL)
		match = 0;
	    else if (strstr (*message, "EEI status=") != NULL)
		match = 0;
	    else if (strstr (*message, "sensekey=") != NULL)
		match = 0;
	    else if (rq->retry == 1 && rq->aut == 0 && 
			(strstr (*message, "Automatic retry") != NULL)) {
		rq->aut = 1;
		match = 0;
	    }
	    else if (rq->retry == 1 && rq->aut == 0 && 
                         (strstr (*message, "Automatic retry") == NULL)) 
		match = 0;
	    else if (mess_time >= rq->req_time && mess_time <= rq->complete_time) 
	        match = 1;
            else if (mess_time > rq->complete_time)
	    	*stop = 1;
	    if (! match) {
		free (*message);
		*message = NULL;
	    }
	}
	else 
	    match = 0;	  
    }
    else
	match = 0;

    return (match);
} 

/**************************************/
/* Function to match requests records */
/**************************************/

match_reqrec (rtcp, timestamp, serv, gflag, dev_group)
struct acctrtcp *rtcp;
time_t timestamp;
struct server *serv;
int gflag;
char dev_group[MAXDGP];
{
    int found = 0;		/* flag set if matching record found */
    struct requests *rq;	/* pointer to request structure */
    int skip_insert = 0;	/* set if record deleted */

    rq = req_first;

    while (rq) {
	if ((rq->jid == rtcp->jid) && rq->complete_time == 0) {
	    found = 1;
	    rq->complete_time = timestamp;
	    break;
	}
	rq = rq->next;
    }

    if (!found) {
	create_reqrec (rtcp, timestamp, serv);
	rq = req_prev;
    }

    get_dgn (rq);
    if (gflag && strcmp (rq->dgn, dev_group) != 0) {
	delete_reqrecord (rq);
	skip_insert = 1;
    } 
    
    if (rtcp->exitcode == 205 || rtcp->exitcode == 222) {
	delete_reqrecord (rq);
	skip_insert = 1;
    }
    else {
    	rq->exitcode = rtcp->exitcode;
    	rq->retry = rtcp->retryn;
    }
    
    if (!skip_insert) {
	rq->err_lineno = -1;
	insert_stats (rq, serv, 0);
    }
}

/************************************/
/* Function to print out statistics */
/************************************/

print_stats (starttime, endtime, gflag, Sflag, wflag, week)
time_t starttime;
time_t endtime;
int gflag;
int Sflag;
int wflag;
char week[80];
{
    struct dev_grp *dg;		/* pointer to device group record */
    struct failed *fd;		/* failed record */
    int i = 0;			/* counter */
    struct server *sr;		/* pointer to server record */
    int set = 0;		/* set for new error type */
    struct type *ty;		/* pointer to type record */
    struct unknown_errs *uk;	/* pointer to unknown_err structure */
    int total_reqs = 0;		/* total number of user requests made */
    int ty_totalreqs = 0;	/* total number of requests for type */
    int dg_totalreqs = 0;	/* total number of requests for group */
    int sr_totalreqs = 0;	/* total number of requests for server */


    total_reqs = global.usr_succ + global.usr_unsucc;

    printf ("\nRtcopy Statistics for the period : ");
    print_time_interval (starttime, endtime);

/* print global statistics */
    
    if (!Sflag) {
      if (!gflag) {
	if (!wflag)
    	    printf ("\n\nSummarising for allservers :");
	else
	    printf ("\n\nSummarising week %s for allservers :", week);
    	printf ("\n.................................");
    	printf ("\n\nUSER COMMANDS SUCCESSFUL : %d", global.usr_succ); 
    	printf ("\nDEV COMMANDS SUCCESSFUL : %d", global.dev_succ);
    	printf ("\nUSER COMMANDS FAILED : %d", global.usr_unsucc);
    	printf ("\nDEV COMMANDS FAILED : %d", global.dev_unsucc);
	if (global.usr_unsucc > 0) {
		printf ("\n\nANALYSED FAILURES : %5.1f %% - %d", 
			((float)(global.usr_unsucc) / (float)(total_reqs)) * 100.0, 
			global.usr_unsucc);
		printf ("\nUSER ERRORS : %5.1f %% - %d", 
			((float)(global.usr_errors) /(float)( total_reqs)) * 100.0, 
			global.usr_errors);
		printf ("\nTAPE ERRORS : %5.1f %% - %d", 
			((float)(global.tape_errors) / (float)(total_reqs)) * 100.0, 
			global.tape_errors);
		printf ("\nSHIFT ERRORS : %5.1f %% - %d",  
			((float)(global.shift_errors) / (float)(total_reqs)) * 100.0, 
			global.shift_errors);
		printf ("\nROBOT ERRORS : %5.1f %% - %d", 
			((float)(global.robot_errors) / (float)(total_reqs)) * 100.0, 
			global.robot_errors);
		printf ("\nNETWORK ERRORS : %5.1f %% - %d", 
			((float)(global.network_errors) / (float)(total_reqs)) * 100.0, 
			global.network_errors);
		printf ("\nOTHER ERRORS : %5.1f %% - %d", 	
			((float)(global.other_errors) / (float)(total_reqs)) * 100.0, 
			global.other_errors);
		printf ("\nUNKNOWN ERRORS : %5.1f %% - %d", 
			((float)(global.unknown_errors) / (float)(total_reqs)) * 100.0, 
			global.unknown_errors);
		printf ("\nNO MESSAGE : %5.1f %% - %d",
		       ((float)(global.no_mess) / (float)(total_reqs)) * 100.0,
			global.no_mess);

		printf ("\n");

/* print errors and num Occs */

		for (i = 0; i < num_lines; i++) {
		    if (emp[i].error_code == 1) {
			if (set == 0 && global.usr_errors > 0) {
			    printf ("\n\nUSER ERRORS (DETAIL)");
			    set = 1;
			}
		    }
		    else if (emp[i].error_code == 2 && global.tape_errors > 0) {
			if (set != 2) {
			    printf ("\n\nTAPE ERRORS (DETAIL)");
			    set = 2;
			}
		    }
		    else if (emp[i].error_code == 3 && global.shift_errors > 0) {
			if (set != 3) {
			    printf ("\n\nSHIFT ERRORS (DETAIL)");
			    set = 3;
			}
		    }
		    else if (emp[i].error_code == 4 && global.robot_errors > 0) {
			if (set != 4) {
			    printf ("\n\nROBOT ERRORS (DETAIL)");
			    set = 4;
			}
		    }
		    else if (emp[i].error_code == 5 && global.network_errors > 0) {
			if (set != 5) {
			    printf ("\n\nNETWORK ERRORS (DETAIL)");
			    set = 5;
			}
		    }
		    else if (emp[i].error_code == 6 && global.other_errors > 0) {
			if (set != 6) {
			    printf ("\n\nOTHER ERRORS (DETAIL)");
			    set = 6;
			}
		    }
		    if (emp[i].num_occs > 0)  
			printf ("\n%s  -> %5.1f %% - %d", emp[i].message, 
				((float)(emp[i].num_occs) / (float)(total_reqs)) * 100.0, 
				emp[i].num_occs);
		}
		printf ("\n");

/* print unknown errors */

		if (unknown_first != NULL) {
		    printf ("\nUNKNOWN ERRORS FOUND");
		    uk = unknown_first;
		    while (uk != NULL) {
			printf ("\n%s\t:%s  -> %5.1f %% - %d", uk->server_name, uk->message, 
			 ((float)(uk->num_occs) / (float)(total_reqs)) * 100.0, uk->num_occs);
			uk = uk->next;
		    }
		    printf ("\n");
		}
		if (global.no_mess > 0) {
		    printf ("\nNumber of failed requests with no matching entry in rtcopy.err file");
		    printf (" = %d", global.no_mess);
		    printf ("\nHost Name\tJob ID\tRequest Complete Time");
		    fd = fail_first;
		    while (fd != NULL) {
			printf ("\n%s\t\t%d\t", fd->server, fd->jid);
			print_time_val (fd->complete_time);
			fd = fd->next;
		    }
		    printf ("\n");
		}
		printf ("\n");
	}
      }

/* print device type statistics */

        if (!gflag) {
    	    ty = type_first;
    	    while (ty) {
	        ty_totalreqs = ty->stat.usr_succ + ty->stat.usr_unsucc;
		if (!wflag)
		    printf ("\n\nSummarising for %s servers", ty->type);
		else
		    printf ("\n\nSummarising week %s for %s servers", week, ty->type);
	    	printf ("\n...............................");
 	    	printf ("\n\nUSER COMMANDS SUCCESSFUL : %d", ty->stat.usr_succ);
    	    	printf ("\nDEV COMMANDS SUCCESSFUL : %d", ty->stat.dev_succ);
    	    	printf ("\nUSER COMMANDS FAILED : %d", ty->stat.usr_unsucc);
    	    	printf ("\nDEV COMMANDS FAILED : %d", ty->stat.dev_unsucc);
		if (ty->stat.usr_unsucc > 0) {
    	    	    printf ("\n\nANALYSED FAILURES : %5.1f %% - %d",
                        ((float)(ty->stat.usr_unsucc) / (float)(ty_totalreqs)) * 100.0, 
		 	ty->stat.usr_unsucc);
    	    	    printf ("\nUSER ERRORS : %5.1f %% - %d",
                 	((float)(ty->stat.usr_errors) /(float)(ty_totalreqs)) * 100.0, 
		 	ty->stat.usr_errors);
    	    	    printf ("\nTAPE ERRORS : %5.1f %% - %d",
                 	((float)(ty->stat.tape_errors) / (float)(ty_totalreqs)) * 100.0, 
		 	ty->stat.tape_errors);
     	    	    printf ("\nSHIFT ERRORS : %5.1f %% - %d",
                 	((float)(ty->stat.shift_errors) / (float)(ty_totalreqs)) * 100.0, 
		 	ty->stat.shift_errors);
    	    	    printf ("\nROBOT ERRORS : %5.1f %% - %d",
                  	((float)(ty->stat.robot_errors) / (float)(ty_totalreqs)) * 100.0, 
		 	ty->stat.robot_errors);
    	    	    printf ("\nNETWORK ERRORS : %5.1f %% - %d",
                 	((float)(ty->stat.network_errors) / (float)(ty_totalreqs)) * 100.0, 
		 	ty->stat.network_errors);
    	    	    printf ("\nOTHER ERRORS : %5.1f %% - %d",
                 	((float)(ty->stat.other_errors) / (float)(ty_totalreqs)) * 100.0, 
		 	ty->stat.other_errors);
     	    	    printf ("\nUNKNOWN ERRORS : %5.1f %% - %d",
                  	((float)(ty->stat.unknown_errors) / (float)(ty_totalreqs)) * 100.0, 
		 	ty->stat.unknown_errors);
            	    printf ("\nNO MESSAGE : %5.1f %% - %d",
                 	((float)(ty->stat.no_mess) / (float)(ty_totalreqs)) * 100.0,
                 	ty->stat.no_mess);
     	     	    printf ("\n");
	    	    set = 0;

/* print errors and num Occs */

	            for (i = 0; i < num_lines; i++) {
        	    	if (emp[i].error_code == 1) {
	      	    	    if (set == 0 && ty->stat.usr_errors > 0) {
                    	    	printf ("\n\nUSER ERRORS (DETAIL)");
                    	    	set = 1;
                    	    }
  		    	}
        	    	else if (emp[i].error_code == 2 && ty->stat.tape_errors > 0) {
            	    	    if (set != 2) {
                	    	printf ("\n\nTAPE ERRORS (DETAIL)");
                	    	set = 2;
              	            }
        	    	}
        	    	else if (emp[i].error_code == 3 && ty->stat.shift_errors > 0) {
            	    	    if (set != 3) {
                	    	printf ("\n\nSHIFT ERRORS (DETAIL)");
                	    	set = 3;
            	     	    }
        	    	}
        	    	else if (emp[i].error_code == 4 && ty->stat.robot_errors > 0) {
            	    	    if (set != 4) {
                	    	printf ("\n\nROBOT ERRORS (DETAIL)");
                	    	set = 4;
            	    	    }
        	    	}
        	        else if (emp[i].error_code == 5 && ty->stat.network_errors > 0) {
            	    	    if (set != 5) {
                	    	printf ("\n\nNETWORK ERRORS (DETAIL)");
                	    	set = 5;
     	  	    	    }
        	    	}
        	    	else if (emp[i].error_code == 6 && ty->stat.other_errors > 0) {
            	    	    if (set != 6) {
                	    	printf ("\n\nOTHER ERRORS (DETAIL)");
                	    	set = 6;
            	    	    }
        	    	}
        	    	if (ty->err_messages[i] > 0)
        	    	    printf ("\n%s  -> %5.1f %% - %d", emp[i].message,
                	     ((float)(ty->err_messages[i]) / (float)(ty_totalreqs)) * 100.0, 
			     ty->err_messages[i]);
    	            }
	            printf ("\n");

/* print unknown errors */

        	    if (unknown_first != NULL && ty->stat.unknown_errors > 0) {
            		printf ("\nUNKNOWN ERRORS FOUND");
            		uk = unknown_first;
            		while (uk != NULL) {
			    if (strcmp (uk->type, ty->type) == 0) 
                	        printf ("\n%s\t:%s  -> %5.1f %% - %d", uk->server_name, uk->message,
                 	    	   ((float)(uk->num_occs) / (float)(total_reqs)) * 100.0, uk->num_occs);
                	    uk = uk->next;
            		}
            		printf ("\n");
        	    }
        	    if (ty->stat.no_mess > 0) {
            		printf ("\nNumber of failed requests with no matching entry in rtcopy.err file");
            		printf (" = %d", ty->stat.no_mess);
            		printf ("\nHost Name\tJob ID\tRequest Complete Time");
            		fd = fail_first;
            		while (fd != NULL) {
			    if (strcmp (ty->type, fd->type) == 0) {
                		printf ("\n%s\t\t%d\t", fd->server, fd->jid);
                		print_time_val (fd->complete_time);
			    }
                	    fd = fd->next;
            		}
            		printf ("\n");
        	    }
        	    printf ("\n");
	 	}		     
	        ty = ty->next; 
            }
        }
        else {

	/* print device group statistics */
 	
	    dg = dev_first;
            while (dg) {
                dg_totalreqs = dg->stat.usr_succ + dg->stat.usr_unsucc;
		if (!wflag)
		    printf ("\n\nSummarising for %s servers", dg->dgn);
		else
		    printf ("\n\nSummarising week %s for %s servers", week, dg->dgn);
            	printf ("\n...............................");
            	printf ("\n\nUSER COMMANDS SUCCESSFUL : %d", dg->stat.usr_succ);
            	printf ("\nDEV COMMANDS SUCCESSFUL : %d", dg->stat.dev_succ);
            	printf ("\nUSER COMMANDS FAILED : %d", dg->stat.usr_unsucc);
            	printf ("\nDEV COMMANDS FAILED : %d", dg->stat.dev_unsucc);
		if (dg->stat.usr_unsucc > 0) {
            	    printf ("\n\nANALYSED FAILURES : %5.1f %% - %d",
                	((float)(dg->stat.usr_unsucc) / (float)(dg_totalreqs)) * 100.0, 
			dg->stat.usr_unsucc);
            	    printf ("\nUSER ERRORS : %5.1f %% - %d",
                	((float)(dg->stat.usr_errors) /(float)(dg_totalreqs)) * 100.0, 
			dg->stat.usr_errors);
            	    printf ("\nTAPE ERRORS : %5.1f %% - %d",
                	((float)(dg->stat.tape_errors) / (float)(dg_totalreqs)) * 100.0, 
			dg->stat.tape_errors);
            	    printf ("\nSHIFT ERRORS : %5.1f %% - %d",
                	((float)(dg->stat.shift_errors) / (float)(dg_totalreqs)) * 100.0, 
			dg->stat.shift_errors);
            	    printf ("\nROBOT ERRORS : %5.1f %% - %d",
                	((float)(dg->stat.robot_errors) / (float)(dg_totalreqs)) * 100.0, 
			dg->stat.robot_errors);
            	    printf ("\nNETWORK ERRORS : %5.1f %% - %d",
                	((float)(dg->stat.network_errors) / (float)(dg_totalreqs)) * 100.0, 
			dg->stat.network_errors);
            	    printf ("\nOTHER ERRORS : %5.1f %% - %d",
                	((float)(dg->stat.other_errors) / (float)(dg_totalreqs)) * 100.0, 
			dg->stat.other_errors);
            	    printf ("\nUNKNOWN ERRORS : %5.1f %% - %d",
                	((float)(dg->stat.unknown_errors) / (float)(dg_totalreqs)) * 100.0, 
			dg->stat.unknown_errors);
                    printf ("\nNO MESSAGE : %5.1f %% - %d",
                 	((float)(dg->stat.no_mess) / (float)(dg_totalreqs)) * 100.0,
                  	dg->stat.no_mess);
            	    printf ("\n");
            	    set = 0;

/* print errors and num Occs */

            	    for (i = 0; i < num_lines; i++) {
                    	if (emp[i].error_code == 1) {
                    	    if (set == 0 && dg->stat.usr_errors > 0) {
                            	printf ("\n\nUSER ERRORS (DETAIL)");
                            	set = 1;
                    	    }
                    	}
                        else if (emp[i].error_code == 2 && dg->stat.tape_errors > 0) {
                    	    if (set != 2) {
                            	printf ("\n\nTAPE ERRORS (DETAIL)");
                            	set = 2;
                    	    }
                    	}
                        else if (emp[i].error_code == 3 && dg->stat.shift_errors > 0) {
                    	    if (set != 3) {
                            	printf ("\n\nSHIFT ERRORS (DETAIL)");
                            	set = 3;
                    	    }
                        }
                        else if (emp[i].error_code == 4 && dg->stat.robot_errors > 0) {
                    	    if (set != 4) {
                            	printf ("\n\nROBOT ERRORS (DETAIL)");
                            	set = 4;
                    	    }
                        }
 		        else if (emp[i].error_code == 5 && dg->stat.network_errors > 0) {
                            if (set != 5) {
                                printf ("\n\nNETWORK ERRORS (DETAIL)");
                            	set = 5;
                    	    }
                    	}
                        else if (emp[i].error_code == 6 && dg->stat.other_errors > 0) {
                    	    if (set != 6) {
                                printf ("\n\nOTHER ERRORS (DETAIL)");
                                set = 6;
                    	    }
                    	}
                    	if (dg->err_messages[i] > 0)
                    	    printf ("\n%s  -> %5.1f %% - %d", emp[i].message,
                            ((float)(dg->err_messages[i]) / (float)(dg_totalreqs)) * 100.0, 
			    dg->err_messages[i]);
                    }
                    printf ("\n");
		}
                dg = dg->next;
            }
        }
    }

/* print server statistics */
        
    sr = serv_first;
    while (sr) {
        if (sr->failed != 1) {
	    sr_totalreqs = sr->stat.usr_succ + sr->stat.usr_unsucc;
	    if (!wflag)
		printf ("\n\nSummarising for %s servers", sr->server_name);
	    else
		printf ("\n\nSummarising week %s for %s servers", week, sr->server_name);
            printf ("\n...............................");
            printf ("\n\nUSER COMMANDS SUCCESSFUL : %d", sr->stat.usr_succ);
            printf ("\nDEV COMMANDS SUCCESSFUL : %d", sr->stat.dev_succ);
            printf ("\nUSER COMMANDS FAILED : %d", sr->stat.usr_unsucc);
            printf ("\nDEV COMMANDS FAILED : %d", sr->stat.dev_unsucc);
	    if (sr->stat.usr_unsucc > 0) {
            	printf ("\n\nANALYSED FAILURES : %5.1f %% - %d",
                    ((float)(sr->stat.usr_unsucc) / (float)(sr_totalreqs)) * 100.0, 
		    sr->stat.usr_unsucc);
            	printf ("\nUSER ERRORS : %5.1f %% - %d",
                    ((float)(sr->stat.usr_errors) /(float)(sr_totalreqs)) * 100.0, 
		    sr->stat.usr_errors);
            	printf ("\nTAPE ERRORS : %5.1f %% - %d",
                    ((float)(sr->stat.tape_errors) / (float)(sr_totalreqs)) * 100.0, 
		    sr->stat.tape_errors);
            	printf ("\nSHIFT ERRORS : %5.1f %% - %d",
                    ((float)(sr->stat.shift_errors) / (float)(sr_totalreqs)) * 100.0, 
		    sr->stat.shift_errors);
             	printf ("\nROBOT ERRORS : %5.1f %% - %d",
                    ((float)(sr->stat.robot_errors) / (float)(sr_totalreqs)) * 100.0, 
		    sr->stat.robot_errors);
            	printf ("\nNETWORK ERRORS : %5.1f %% - %d",
                    ((float)(sr->stat.network_errors) / (float)(sr_totalreqs)) * 100.0, 
		    sr->stat.network_errors);
            	printf ("\nOTHER ERRORS : %5.1f %% - %d",
                    ((float)(sr->stat.other_errors) / (float)(sr_totalreqs)) * 100.0, 
		    sr->stat.other_errors);
            	printf ("\nUNKNOWN ERRORS : %5.1f %% - %d",
                    ((float)(sr->stat.unknown_errors) / (float)(sr_totalreqs)) * 100.0, 
		    sr->stat.unknown_errors);
	    	printf ("\nNO MESSAGE : %5.1f %% - %d",
		    ((float)(sr->stat.no_mess) / (float)(sr_totalreqs)) * 100.0,
		    sr->stat.no_mess);
            	printf ("\n");
                set = 0;

/* print errors and num Occs */

    	        for (i = 0; i < num_lines; i++) {
                    if (emp[i].error_code == 1) {
                    	if (set == 0 && sr->stat.usr_errors > 0) {
                            printf ("\n\nUSER ERRORS (DETAIL)");
                            set = 1;
                    	}
                    }
                    else if (emp[i].error_code == 2 && sr->stat.tape_errors > 0) {
                    	if (set != 2) {
                            printf ("\n\nTAPE ERRORS (DETAIL)");
                            set = 2;
                    	}
                    }
                    else if (emp[i].error_code == 3 && sr->stat.shift_errors > 0) {
                    	if (set != 3) {
                            printf ("\n\nSHIFT ERRORS (DETAIL)");
                            set = 3;
                    	}
                    }
                    else if (emp[i].error_code == 4 && sr->stat.robot_errors > 0) {
                    	if (set != 4) {
                            printf ("\n\nROBOT ERRORS (DETAIL)");
                            set = 4;
                    	}
                    }
                    else if (emp[i].error_code == 5 && sr->stat.network_errors > 0) {
                    	if (set != 5) {
                            printf ("\n\nNETWORK ERRORS (DETAIL)");
                            set = 5;
                    	}
                    }
                    else if (emp[i].error_code == 6 && sr->stat.other_errors > 0) {
                        if (set != 6) {
                            printf ("\n\nOTHER ERRORS (DETAIL)");
                            set = 6;
                    	}
                    }
                    if (sr->err_messages[i] > 0)
                        printf ("\n%s  -> %5.1f %% - %d", emp[i].message,
                          ((float)(sr->err_messages[i]) / (float)(sr_totalreqs)) * 100.0, 
			   sr->err_messages[i]);
                }
                printf ("\n");
 /* print unknown errors */
 
	        if (unknown_first != NULL && sr->stat.unknown_errors > 0) {
            	    printf ("\nUNKNOWN ERRORS FOUND");
            	    uk = unknown_first;
            	    while (uk != NULL) {
		    	if (strcmp (uk->server_name, sr->server_name) == 0)
                   	    printf ("\n%s\t:%s  -> %5.1f %% - %d", uk->server_name, 
				uk->message, ((float)(uk->num_occs) / (float)(total_reqs)) 
				* 100.0, uk->num_occs);
                        uk = uk->next;
            	    }
            	    printf ("\n");
	        }
                if (sr->stat.no_mess > 0) {
                    printf ("\nNumber of failed requests with no matching entry ");
		    printf ("in rtcopy.err file");
                    printf (" = %d", sr->stat.no_mess);
                    printf ("\nHost Name\tJob ID\tRequest Complete Time");
                    fd = fail_first;
                	while (fd != NULL) {
		        if (strcmp (sr->server_name, fd->server) == 0) {		 
                  	    printf ("\n%s\t\t%d\t", fd->server, fd->jid);
                  	    print_time_val (fd->complete_time);
		    	}
                        fd = fd->next;
                    }
                    printf ("\n");
                }
	    }
	    else
                printf ("\n");
        }
        sr = sr->next;
    }
}

/************************************************/
/* Function to print out details of tape errors */
/************************************************/

print_tape_errs (tp)
struct tp_sort *tp;
{
    int i = 0;		/* counter */

    printf ("\nReq Time           Server    Jid  VID     UNM      #Occ  Error Message\n");
    for (i = 0; i < num_tperrs; i++) {
	print_time_val (tp[i].first_time);
	printf ("  %-7s %5d  %-7s %-9s %3d  %s\n", tp[i].server, tp[i].first_jid,
		tp[i].vid, tp[i].unm, tp[i].num_occs, emp[tp[i].mess_line].message);
	if (tp[i].num_occs > 1) {
	    print_time_val (tp[i].last_time);
            printf ("          %5d  %-7s %-9s      %s\n", tp[i].last_jid, tp[i].vid,
		tp[i].unm, emp[tp[i].mess_line].message);
	}
    }
}
	
/**************************/
/* Function to print time */
/**************************/

print_time_val (stime)
time_t stime;
{
   struct tm *tm;       /* pointer to tm record */

   tm = localtime (&stime);
   printf ("%02d/%02d/%04d %02d:%02d:%02d",
	tm->tm_mday, tm->tm_mon+1, tm->tm_year+1900, tm->tm_hour, 
	tm->tm_min, tm->tm_sec);
}

/***************************************/
/* Function to print out time interval */
/***************************************/

print_time_interval (starttime, endtime)
time_t starttime;
time_t endtime;
{
   printf (" (");
   print_time_val (starttime);
   printf ("  -  ");
   print_time_val (endtime);
   printf (")");
}

/***************************************************************************/
/* Function to read error messages from error file and match with requests */
/***************************************************************************/

read_errfile (server, vflag, wflag, week)
struct server *server;
int vflag;
int wflag;
char week[80];
{
    char errfile[80];		/* error file path and name */
    char buf[500];		/* buffer */
    int c;			/* return from read function */
    int failed = 0;		/* set if open fails */
    int found = 0;		/* flag */
    int found2 = 0;
    int fp;			/* file descriptor */
    int i;			/* counter */
    int match = 0;		/* flag set if matching error record found */
    char *message;		/* pointer to message read from file */
    char *p;			/* char pointer */
    char *q;			/* char pointer */
    struct requests *rq;	/* requests record */
    struct tape_errs *tp;	/* tape errors record */
    char savebuf[500];		/* buffer */
    int saveflag = 0;		/* flag set if part line in buffer */
    int stop = 0;		/* flag set when no need to continue reading file */

    message = NULL;
    errfile[0] = '\0';
    strcpy (errfile, server->server_name);
    strcat (errfile, ":");
    strcat (errfile, MSGLOG);
    if (wflag) {
	strcat (errfile, ".");
	strcat (errfile, week);
    }
    if (vflag)
	fprintf (stderr, "\nopening file : %s", errfile);
    if ((fp = rfio_open (errfile, O_RDONLY)) < 0) {
	fprintf (stderr, "%s : open error : %s\n", errfile, rfio_serror());
	failed = 1;
    }

    if (! failed) {
	check_retry (fp);
	rfio_lseek (fp, 0L, 0);	
	
	rq = req_first;
	while (rq != NULL) {
	    if (rq->complete_time <= 0) {
	    	rq = rq->next;
		continue;
	    }
	    while ((c = rfio_read (fp, buf, sizeof (buf) - 1)) > 0) {
		buf[c] = 0;
		p = buf;
		if (saveflag) {
		    q = strchr (buf, '\n');
		    if (!q) {
			strcat (savebuf, p);
			continue;
		    }
		    *q = '\0';
		    strcat (savebuf, p);
		    if (*savebuf != '\0' && *savebuf != ' ') 
			match = match_errors (savebuf, rq, &message, &stop);
		    if (match || stop) break;
		    saveflag = 0;
		    p = q + 1;	
		}
		while (q = strchr (p, '\n')) {
		    *q = '\0';
		    if (*p == '\0' || *p == ' ') {
			p = q + 1;
			continue;
		    }
		    match = match_errors (p, rq, &message, &stop);
		    if (match || stop) break;
		    p = q + 1;
		}
		if (match || stop) break;
		if (strlen (p)) {
		    strcpy (savebuf, p);
		    saveflag = 1;
		}
	    }
	    if (saveflag && (*p != '\0' || *p != ' ') && !match && !stop)
	        match = match_errors (savebuf, rq, &message, &stop);

	    if (match) {
		found = 0;
		for (i = 0; i < num_lines; i++) {
		    if (strstr (message, emp[i].message) != NULL) {
			rq->exitcode = emp[i].error_code;
			emp[i].num_occs++;
	 	 	rq->err_lineno = i;
			found = 1;
			if (rq->exitcode == 2) {
     			    tp = tape_first;
     			    if (tp == NULL) {
         			tp = (struct tape_errs *) calloc (1, sizeof (struct tape_errs));
         			tp->first_time = rq->req_time;
         			strcpy (tp->unm, rq->unm);
         			strcpy (tp->vid, rq->vid);
				tp->first_jid = rq->jid;
				strcpy (tp->server, server->server_name);
         			tp->mess_line = i;
         			tp->num_occs++;
         			tp->next = NULL;
         			tape_first = tp;
         			tape_prev = tp;
         			num_tperrs++;
     			    }
     			    else {
         			while (tp != NULL) {
             			    if ((strcmp (tp->unm, rq->unm) == 0) && 
					(strcmp (tp->vid, rq->vid) == 0) &&
                 			(strcmp (tp->server, server->server_name) == 0) &&
					tp->mess_line == i) {
                 			tp->last_time = rq->req_time;
					tp->last_jid = rq->jid;
                 			tp->num_occs++;
                 			found2 = 1;
                 			break;
             			    }
             			    tp = tp->next;
         		        }
         		        if (!found2) {
             			    tp = (struct tape_errs *) calloc (1, sizeof (struct tape_errs));
             			    tp->first_time = rq->req_time;
             			    strcpy (tp->unm, rq->unm);
             			    strcpy (tp->vid, rq->vid);
				    tp->first_jid = rq->jid;
         			    tp->mess_line = i;
             			    strcpy (tp->server, server->server_name);
             			    tp->num_occs++;
             			    tp->next = NULL;
             			    tape_prev->next = tp;
             			    tape_prev = tp;
             			    num_tperrs++;
         		        }
			    }
			    found2 = 0;
     			}
			break;
		    }
		}
  	    	if (!found) {
		    create_unknown (rq, message, server);
		    rq->exitcode = 10;
	 	}
		free (message);
		message = NULL;
	    }

	    rfio_lseek (fp, 0L, 0);	
	    saveflag = 0;
	    savebuf[0] = '\0';
	    rq = rq->next;
	    stop = 0;
	}

	rfio_close (fp);
     	rq = req_first;
	while (rq) {
	    if (rq->complete_time > 0) {
	        if (rq->exitcode == -1) {
		    create_failed (rq, server);
		    rq->exitcode = 11;
	    	}
	        insert_stats (rq, server, 1);
	    }
	    rq = rq->next;
	}

	while (req_first != NULL) 
	    delete_reqrecord (req_first);
    }
    else
    	printf ("\nServer %s rtcopy.err read failed\n",server->server_name);
}	
			

/***********************************************/
/* Function to read error messages into memory */
/***********************************************/

read_mess2mem()
{
    char buf[256];		/* buffer */
    struct err_mess *emp1;	/* error messages pointer */
    FILE *fp;			/* file descriptor */
    int len = 0;		/* length of error message */
    char *p1;			/* character pointer */

/* read in the error messages from "err_messages" file into memory */

    if ((fp = fopen (RTCOPYERRLIST, "r")) == NULL) {
        printf ("%s :open error\n", RTCOPYERRLIST);
        exit (1);
    }
    else {
        while (fgets (buf, sizeof (buf), fp))
            num_lines++;
    }
    rewind (fp);

    emp = (struct err_mess *) malloc (num_lines * sizeof (struct err_mess));
    emp1 = emp;
    while (fgets (buf, sizeof (buf), fp)) {
        p1 = strchr (buf + 1, '"');
        len = p1 - buf;
        emp1->message = malloc (len);
        *p1 = '\0';
        strncpy (emp1->message, buf + 1, len);
        emp1->error_code = atoi (p1 + 2);
        emp1->num_occs = 0;
        emp1++;
    }

}

/*********************************************************************/
/* Function to read the TPCONFIG files to extract device group types */
/*********************************************************************/

read_tpfile()
{

    char buf[256];		/* buffer */
    int c;			/* number of items read */
    int fp;			/* file identifier */
    char *p;			/* pointer to buffer */
    char *q;			/* pointer for strchr function */
    char savebuf[256];		/* save partial line */
    int saveflag = 0;		/* flag set if save required */
    int server_failed = 0; 	/* set if server open failed */
    struct server *serv_list;	/* pointer to list of all servers to open */
    char tpconfile[80];		/* TPCONFIG file path and name */

    tpconfile[0] = '\0';

/* for each server in the server list open the TPCONFIG file and read in data */

    serv_list = serv_first;
    while (serv_list) {

/* copy the name and path of the TPCCONFIG file into tpconfile */

    	strcpy (tpconfile, serv_list->server_name);
        strcat (tpconfile, ":");
        strcat (tpconfile, TPCONFIG);

        if ((fp = rfio_open (tpconfile, O_RDONLY)) < 0) {
            fprintf (stderr, "%s : open error : %s\n", tpconfile, rfio_serror());
            server_failed = 1;
	    serv_list->failed = 1;
        }

/* read in data, for each full line read in that is not a comment line  or a blank */
/* line insert_type. if an incomplete line is read in save it in savebuf and continue */
/* If at end of file saveflag is still set insert type from savebuf */

	if (!server_failed) {
	    while ((c = rfio_read (fp, buf, sizeof (buf) - 1)) > 0) {
	    	buf[c] = 0;
		p = buf;
	    	if (saveflag) {
		    q = strchr (buf, '\n');	
		    if (!q) {	
		    	strcat (savebuf, p);	
		    	continue;
		    } 
		    *q = '\0';			
		    strcat (savebuf, p);	
		    if (*savebuf != '#' && *savebuf != '\0') 
	    	        insert_type(savebuf, serv_list);
		    saveflag = 0;		
		    p = q + 1;			
	    	}
	    	while (q = strchr (p, '\n')) {
		    *q = '\0';			
		    if (*p == '#' || *p == '\0') {
			p = q + 1;		
		  	continue;
		    }
		    insert_type(p, serv_list);	
		    p = q + 1;
	    	}
	        if (strlen (p)) {		
		    strcpy (savebuf, p);	
		    saveflag = 1;	
	    	}
	    }

	    if (saveflag && *savebuf != '#' && *savebuf != '\0')
		insert_type(savebuf, serv_list);

            rfio_close(fp);
   	}
	server_failed = 0;
        serv_list = serv_list->next;
    }
}

/************************************************/
/* Function to copy info into new server record */
/************************************************/
sort_fill(serv_rec)
struct server *serv_rec;
{
    strcpy (serv_rec->server_name, serv_first->server_name);
    strcpy (serv_rec->type, serv_first->type);
    serv_rec->err_messages = (int *) calloc (num_lines, sizeof (int));
    serv_rec->next = NULL;
}
     
/****************************************************/
/* Function to sort the server list into type order */
/****************************************************/
sort_servlist()
{
    int c = 0;			/* return value from strcmp */
    int d = 0;			/* return value from strcmp */
    struct server *new_first; 	/* pointer to new server list */
    struct server *prev;
    struct server *serv_new;   	/* pointer to server list */
    struct server *st;		/* pointer to server structure */
    struct server *temp;

    new_first = (struct server*) calloc (1, sizeof (struct server));
    sort_fill (new_first);

    while (serv_first) {
	serv_new = new_first;
	while (serv_new) {
    	    c = strcmp (serv_first->type, serv_new->type);
	    d = strcmp (serv_first->server_name, serv_new->server_name);

 	    if (c == 0 && d == 0) break; 

	    st = (struct server*) calloc (1, sizeof (struct server));
	    sort_fill (st);

	    if (c > 0) {
		if (serv_new->next == NULL) {
		    serv_new->next = st;
		    break;
		}
		else 
		    prev = serv_new;		     
	    }
	    if (c < 0) {
		if (serv_new == new_first) {
		    st->next = serv_new;
		    new_first = st;
		    break;
		}
		else {
		    prev->next = st;
		    st->next = serv_new;
		    break;
		}
	    }
	    if (c == 0) {
		if (d < 0) {
		    if (serv_new == new_first) {
			st->next = serv_new;
			new_first = st;
			break;
		    }
		    else {
	 	        prev->next = st;
		        st->next = serv_new;
		        break;
		    }
		}
		else if (d > 0) {
		    if (serv_new->next == NULL) {
			serv_new->next = st;
			break;
		    }
		    else 
			prev = serv_new; 
		}
	    }
	    serv_new = serv_new->next;	
	}
	temp = serv_first;
	serv_first = serv_first->next;
	free (temp);
    }
    serv_first = new_first;
}

/***********************************************/
/* Function to sort tape errors by unm and vid */
/***********************************************/

sort_tperrs()
{
    struct tape_errs *tp;
    struct tape_errs *tp2;
    struct tp_sort *tps;
    int comp();
    int comp2();

    tperr_sort = (struct tp_sort *) malloc (num_tperrs * sizeof (struct tp_sort));
    tps = tperr_sort;

    tp = tape_first;
    while (tp != NULL) {
        tps->first_time = tp->first_time;	
 	tps->last_time = tp->last_time;
	strcpy (tps->unm, tp->unm);
	strcpy (tps->vid, tp->vid);
	tps->first_jid = tp->first_jid;
	tps->last_jid = tp->last_jid;
	tps->mess_line = tp->mess_line;
	tps->num_occs = tp->num_occs;
	strcpy (tps->server, tp->server);
	tps++;
	tp2 = tp;
	tp = tp->next;
	free (tp2);
    }
 
    qsort (tperr_sort, num_tperrs, sizeof (struct tp_sort), comp);
    printf ("\nDETAILS OF TAPE ERRORS - SORTED BY VID");
    print_tape_errs (tperr_sort);
    qsort (tperr_sort, num_tperrs, sizeof (struct tp_sort), comp2);
    printf ("\nDETAILS OF TAPE ERRORS - SORTED BY UNM");
    print_tape_errs (tperr_sort);
}        
/********************************************************/
/* Function to extract the device group for the request */
/********************************************************/
store_dgn (rp)
struct accttape *rp;
{
    struct store_dgn *sd;       /* pointer to stored dgn record */
    int found = 0;              /* flag set if record found */

/* store the jid and dgn for ACCTTAPE records */
/* if not already stored create a record */

    sd = store_first;
    if (sd == NULL) {
        sd = (struct store_dgn *) calloc (1, sizeof (struct store_dgn));
        sd->next = NULL;
        sd->prev = NULL;
        strcpy (sd->dgn, rp->dgn);
  	strcpy (sd->unm, rp->unm);
  	strcpy (sd->vid, rp->vid);
        sd->jid = rp->jid;
        store_first = sd;
        store_prev = sd;
    }
    else {
        while (sd) {
            if (sd->jid == rp->jid) {
		if (strcmp (sd->vid,  "") == 0)
		    strcpy (sd->vid, rp->vid);
		if (strcmp (sd->unm, "") == 0)
		    strcpy (sd->unm, rp->unm);
                found = 1;
                break;
            }
            sd = sd->next;
        }
        if (! found) {
            sd = (struct store_dgn *) calloc (1, sizeof (struct store_dgn));
            sd->next = NULL;
            strcpy (sd->dgn, rp->dgn);
  	    strcpy (sd->unm, rp->unm);
  	    strcpy (sd->vid, rp->vid);
            sd->jid = rp->jid;
            sd->prev = store_prev;
            store_prev->next = sd;
            store_prev = sd;
        }
    }
}

/*****************************************/
/* Function to swap byte order of fields */
/*****************************************/

swap_fields (rp)
struct accttape *rp;
{
   swap_it (rp->subtype);
   swap_it (rp->uid);
   swap_it (rp->gid);
   swap_it (rp->jid);
   swap_it (rp->fseq);
   swap_it (rp->reason);
}

swap_fields2 (rtcp)
struct acctrtcp *rtcp;
{
    swap_it (rtcp->subtype);
    swap_it (rtcp->uid);
    swap_it (rtcp->gid);
    swap_it (rtcp->jid);
    swap_it (rtcp->stgreqid);
    swap_it (rtcp->size);
    swap_it (rtcp->retryn);
    swap_it (rtcp->exitcode);
}

/*********************************/
/* Functions to sort tape errors */
/*********************************/
int comp (a, b)
struct tp_sort *a;
struct tp_sort *b;
{
    if (strcmp (a->vid, b->vid) == 0) return (0);
    else if (strcmp (a->vid, b->vid) < 0) return (-1);
    else return (1);
}

int comp2 (a, b)
struct tp_sort *a;
struct tp_sort *b;
{
    if (strcmp (a->unm, b->unm) == 0) return (0);
    else if (strcmp (a->unm, b->unm) < 0) return (-1);
    else return (1);
}

/*****************************************/
/* Function to give correct usage syntax */
/*****************************************/

usage (cmd)
char *cmd;
{
    fprintf (stderr, "usage: %s ", cmd);
    fprintf (stderr, "%s", "[-e end_time][-f accounting_file][-g device_group][-S server_name][-s start_time][-v][-w week_number]\n");
}		

