/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pinot.plugin.filesystem;

import com.google.api.gax.paging.Page;
import com.google.auth.oauth2.GoogleCredentials;
import com.google.cloud.storage.Blob;
import com.google.cloud.storage.BlobId;
import com.google.cloud.storage.BlobInfo;
import com.google.cloud.storage.Bucket;
import com.google.cloud.storage.CopyWriter;
import com.google.cloud.storage.Storage;
import com.google.cloud.storage.StorageBatch;
import com.google.cloud.storage.StorageBatchResult;
import com.google.cloud.storage.StorageException;
import com.google.cloud.storage.StorageOptions;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.net.URI;
import java.nio.channels.Channels;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Base64;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.pinot.plugin.filesystem.GcsUri;
import org.apache.pinot.shaded.com.google.common.base.Preconditions;
import org.apache.pinot.shaded.com.google.common.base.Strings;
import org.apache.pinot.shaded.com.google.common.collect.ImmutableCollection;
import org.apache.pinot.shaded.com.google.common.collect.ImmutableList;
import org.apache.pinot.spi.env.PinotConfiguration;
import org.apache.pinot.spi.filesystem.BasePinotFS;
import org.apache.pinot.spi.filesystem.FileMetadata;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GcsPinotFS
extends BasePinotFS {
    public static final String PROJECT_ID = "projectId";
    public static final String GCP_KEY = "gcpKey";
    public static final String JSON_KEY = "jsonKey";
    private static final Logger LOGGER = LoggerFactory.getLogger(GcsPinotFS.class);
    private static final int DELETE_BATCH_LIMIT = 100;
    private Storage _storage;

    public void init(PinotConfiguration config) {
        GoogleCredentials credentials = null;
        try {
            StorageOptions.Builder storageBuilder = StorageOptions.newBuilder();
            if (!Strings.isNullOrEmpty(config.getProperty(PROJECT_ID))) {
                LOGGER.info("Configs are: {}, {}", (Object)PROJECT_ID, (Object)config.getProperty(PROJECT_ID));
                String projectId = config.getProperty(PROJECT_ID);
                storageBuilder.setProjectId(projectId);
                if (!Strings.isNullOrEmpty(config.getProperty(GCP_KEY))) {
                    String gcpKey = config.getProperty(GCP_KEY);
                    credentials = GoogleCredentials.fromStream(Files.newInputStream(Paths.get(gcpKey, new String[0]), new OpenOption[0]));
                } else if (!Strings.isNullOrEmpty(config.getProperty(JSON_KEY))) {
                    String decodedJsonKey = config.getProperty(JSON_KEY);
                    Base64.Decoder decoder = Base64.getDecoder();
                    try {
                        byte[] decodedBytes = decoder.decode(decodedJsonKey);
                        decodedJsonKey = new String(decodedBytes);
                    }
                    catch (IllegalArgumentException e) {
                        LOGGER.info("Failed to decode jsonKey, using as is");
                    }
                    credentials = GoogleCredentials.fromStream(IOUtils.toInputStream((String)decodedJsonKey, (Charset)StandardCharsets.UTF_8));
                }
            }
            if (credentials == null) {
                LOGGER.info("Configs using default credential");
                credentials = GoogleCredentials.getApplicationDefault();
            }
            this._storage = (Storage)((StorageOptions.Builder)storageBuilder.setCredentials(credentials)).build().getService();
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    public final boolean mkdir(URI uri) throws IOException {
        LOGGER.info("mkdir {}", (Object)uri);
        try {
            GcsUri gcsUri = new GcsUri(uri);
            String directoryPath = gcsUri.getPrefix();
            if (directoryPath.equals("/")) {
                return true;
            }
            if (this.existsDirectoryOrBucket(gcsUri)) {
                return true;
            }
            Blob blob = this._storage.create(BlobInfo.newBuilder(BlobId.of(gcsUri.getBucketName(), directoryPath)).build(), new byte[0], new Storage.BlobTargetOption[0]);
            return blob.exists(new Blob.BlobSourceOption[0]);
        }
        catch (Exception e) {
            throw new IOException(e);
        }
    }

    public boolean delete(URI segmentUri, boolean forceDelete) throws IOException {
        LOGGER.info("Deleting uri {} force {}", (Object)segmentUri, (Object)forceDelete);
        return this.delete(new GcsUri(segmentUri), forceDelete);
    }

    public boolean deleteBatch(List<URI> segmentUris, boolean forceDelete) throws IOException {
        boolean result = true;
        ArrayList<BlobId> blobIds = new ArrayList<BlobId>();
        Iterator<URI> iterator = segmentUris.iterator();
        while (iterator.hasNext()) {
            GcsUri gcsUri = new GcsUri(iterator.next());
            if (this.existsDirectoryOrBucket(gcsUri)) {
                result &= this.delete(gcsUri, forceDelete);
                continue;
            }
            blobIds.add(this.getBlob(gcsUri).getBlobId());
            if (blobIds.size() < 100 && iterator.hasNext()) continue;
            List<Boolean> deleted = this._storage.delete(blobIds);
            result &= deleted.stream().allMatch(Boolean::booleanValue);
            blobIds.clear();
        }
        return result;
    }

    public boolean doMove(URI srcUri, URI dstUri) throws IOException {
        GcsUri srcGcsUri = new GcsUri(srcUri);
        GcsUri dstGcsUri = new GcsUri(dstUri);
        if (this.copy(srcGcsUri, dstGcsUri)) {
            return this.delete(srcGcsUri, true);
        }
        return false;
    }

    public boolean copyDir(URI srcUri, URI dstUri) throws IOException {
        LOGGER.info("Copying uri {} to uri {}", (Object)srcUri, (Object)dstUri);
        return this.copy(new GcsUri(srcUri), new GcsUri(dstUri));
    }

    public boolean exists(URI fileUri) throws IOException {
        if (fileUri == null) {
            return false;
        }
        return this.exists(new GcsUri(fileUri));
    }

    public long length(URI fileUri) throws IOException {
        try {
            GcsUri gcsUri = new GcsUri(fileUri);
            Preconditions.checkState(!this.isPathTerminatedByDelimiter(gcsUri), "URI is a directory");
            Blob blob = this.getBlob(gcsUri);
            Preconditions.checkState(this.existsBlob(blob), "File '%s' does not exist", (Object)fileUri);
            return blob.getSize();
        }
        catch (Exception t2) {
            throw new IOException(t2);
        }
    }

    public String[] listFiles(URI fileUri, boolean recursive) throws IOException {
        return this.listFilesFromGcsUri(new GcsUri(fileUri), recursive);
    }

    private String[] listFilesFromGcsUri(GcsUri gcsFileUri, boolean recursive) throws IOException {
        ImmutableList.Builder builder = ImmutableList.builder();
        String prefix = gcsFileUri.getPrefix();
        String bucketName = gcsFileUri.getBucketName();
        this.visitFiles(gcsFileUri, recursive, blob -> {
            if (!blob.getName().equals(prefix)) {
                builder.add(GcsUri.createGcsUri(bucketName, blob.getName()).toString());
            }
        });
        String[] listedFiles = builder.build().toArray(new String[0]);
        LOGGER.info("Listed {} files from URI: {}, is recursive: {}", listedFiles.length, gcsFileUri, recursive);
        return listedFiles;
    }

    public List<FileMetadata> listFilesWithMetadata(URI fileUri, boolean recursive) throws IOException {
        ImmutableList.Builder listBuilder = ImmutableList.builder();
        GcsUri gcsFileUri = new GcsUri(fileUri);
        String prefix = gcsFileUri.getPrefix();
        String bucketName = gcsFileUri.getBucketName();
        this.visitFiles(gcsFileUri, recursive, blob -> {
            if (!blob.getName().equals(prefix)) {
                boolean isDirectory = blob.getName().endsWith("/");
                FileMetadata.Builder fileBuilder = new FileMetadata.Builder().setFilePath(GcsUri.createGcsUri(bucketName, blob.getName()).toString()).setLength(blob.getSize().longValue()).setIsDirectory(isDirectory);
                if (!isDirectory) {
                    fileBuilder.setLastModifiedTime(blob.getUpdateTime().longValue());
                }
                listBuilder.add(fileBuilder.build());
            }
        });
        ImmutableCollection listedFiles = listBuilder.build();
        LOGGER.info("Listed {} files from URI: {}, is recursive: {}", listedFiles.size(), gcsFileUri, recursive);
        return listedFiles;
    }

    public void copyToLocalFile(URI srcUri, File dstFile) throws Exception {
        LOGGER.info("Copy {} to local {}", (Object)srcUri, (Object)dstFile.getAbsolutePath());
        Preconditions.checkState(!dstFile.isDirectory(), "File '%s' must not be a directory", (Object)dstFile);
        FileUtils.forceMkdir((File)dstFile.getParentFile());
        Blob blob = this.getBlob(new GcsUri(srcUri));
        Preconditions.checkState(this.existsBlob(blob), "File '%s' does not exists", (Object)srcUri);
        blob.downloadTo(dstFile.toPath());
    }

    public void copyFromLocalFile(File srcFile, URI dstUri) throws Exception {
        LOGGER.info("Copying file {} to uri {}", (Object)srcFile.getAbsolutePath(), (Object)dstUri);
        GcsUri dstGcsUri = new GcsUri(dstUri);
        Preconditions.checkState(!this.isPathTerminatedByDelimiter(dstGcsUri), "Path '%s' must be a filename", (Object)dstGcsUri);
        BlobInfo blobInfo = BlobInfo.newBuilder(BlobId.of(dstGcsUri.getBucketName(), dstGcsUri.getPath())).build();
        this._storage.createFrom(blobInfo, Files.newInputStream(srcFile.toPath(), new OpenOption[0]), new Storage.BlobWriteOption[0]);
    }

    public boolean isDirectory(URI uri) throws IOException {
        return this.existsDirectoryOrBucket(new GcsUri(uri));
    }

    public long lastModified(URI uri) throws IOException {
        return this.getBlob(new GcsUri(uri)).getUpdateTime();
    }

    public boolean touch(URI uri) throws IOException {
        try {
            LOGGER.info("touch {}", (Object)uri);
            GcsUri gcsUri = new GcsUri(uri);
            Blob blob = this.getBlob(gcsUri);
            long updateTime = blob.getUpdateTime();
            this._storage.update((BlobInfo)((Blob.Builder)blob.toBuilder().setMetadata((Map)blob.getMetadata())).build());
            long newUpdateTime = this.getBlob(gcsUri).getUpdateTime();
            return newUpdateTime > updateTime;
        }
        catch (StorageException e) {
            throw new IOException(e);
        }
    }

    public InputStream open(URI uri) throws IOException {
        try {
            Blob blob = this.getBlob(new GcsUri(uri));
            return Channels.newInputStream(blob.reader(new Blob.BlobSourceOption[0]));
        }
        catch (StorageException e) {
            throw new IOException(e);
        }
    }

    private Bucket getBucket(GcsUri gcsUri) {
        return this._storage.get(gcsUri.getBucketName(), new Storage.BucketGetOption[0]);
    }

    private Blob getBlob(GcsUri gcsUri) throws IOException {
        try {
            return this._storage.get(BlobId.of(gcsUri.getBucketName(), gcsUri.getPath()));
        }
        catch (StorageException e) {
            throw new IOException(e);
        }
    }

    private boolean existsBlob(Blob blob) {
        return blob != null && blob.exists(new Blob.BlobSourceOption[0]);
    }

    private boolean existsFile(GcsUri gcsUri) throws IOException {
        return this.existsBlob(this.getBlob(gcsUri));
    }

    private boolean isPathTerminatedByDelimiter(GcsUri gcsUri) {
        return gcsUri.getPath().endsWith("/");
    }

    private boolean existsDirectoryOrBucket(GcsUri gcsUri) throws IOException {
        String prefix = gcsUri.getPrefix();
        if (prefix.isEmpty()) {
            return true;
        }
        Blob blob = this._storage.get(BlobId.of(gcsUri.getBucketName(), prefix));
        if (this.existsBlob(blob)) {
            return true;
        }
        try {
            return this._storage.list(gcsUri.getBucketName(), Storage.BlobListOption.prefix(prefix)).iterateAll().iterator().hasNext();
        }
        catch (Exception t2) {
            throw new IOException(t2);
        }
    }

    private boolean isEmptyDirectory(GcsUri gcsUri) throws IOException {
        if (!this.existsDirectoryOrBucket(gcsUri)) {
            return false;
        }
        String prefix = gcsUri.getPrefix();
        boolean isEmpty = true;
        Page<Blob> page = prefix.equals("/") ? this.getBucket(gcsUri).list(new Storage.BlobListOption[0]) : this._storage.list(gcsUri.getBucketName(), Storage.BlobListOption.prefix(prefix));
        for (Blob blob : page.iterateAll()) {
            if (blob.getName().equals(prefix)) continue;
            isEmpty = false;
            break;
        }
        return isEmpty;
    }

    private void visitFiles(GcsUri fileUri, boolean recursive, Consumer<Blob> visitor) throws IOException {
        try {
            String prefix = fileUri.getPrefix();
            Page<Blob> page = recursive ? this._storage.list(fileUri.getBucketName(), Storage.BlobListOption.prefix(prefix)) : this._storage.list(fileUri.getBucketName(), Storage.BlobListOption.prefix(prefix), Storage.BlobListOption.currentDirectory());
            page.iterateAll().forEach(visitor);
        }
        catch (Exception t2) {
            throw new IOException(t2);
        }
    }

    private boolean exists(GcsUri gcsUri) throws IOException {
        if (this.existsDirectoryOrBucket(gcsUri)) {
            return true;
        }
        if (this.isPathTerminatedByDelimiter(gcsUri)) {
            return false;
        }
        return this.existsFile(gcsUri);
    }

    private boolean delete(GcsUri segmentUri, boolean forceDelete) throws IOException {
        try {
            if (!this.exists(segmentUri)) {
                return false;
            }
            if (this.existsDirectoryOrBucket(segmentUri)) {
                if (!forceDelete && !this.isEmptyDirectory(segmentUri)) {
                    return false;
                }
                String prefix = segmentUri.getPrefix();
                Page<Blob> page = prefix.equals("/") ? this.getBucket(segmentUri).list(new Storage.BlobListOption[0]) : this._storage.list(segmentUri.getBucketName(), Storage.BlobListOption.prefix(prefix));
                return this.batchDelete(page);
            }
            Blob blob = this.getBlob(segmentUri);
            return blob != null && blob.delete(new Blob.BlobSourceOption[0]);
        }
        catch (IOException e) {
            throw e;
        }
        catch (Exception t2) {
            throw new IOException(t2);
        }
    }

    private boolean batchDelete(Page<Blob> page) {
        boolean deleteSucceeded = true;
        StorageBatch batch = this._storage.batch();
        int batchSize = 0;
        ArrayList<StorageBatchResult<Boolean>> results = new ArrayList<StorageBatchResult<Boolean>>();
        for (Blob blob : page.iterateAll()) {
            results.add(batch.delete(blob.getBlobId(), new Storage.BlobSourceOption[0]));
            if (++batchSize < 100) continue;
            batch.submit();
            deleteSucceeded &= results.stream().allMatch(r -> r != null && (Boolean)r.get() != false);
            results = new ArrayList();
            batchSize = 0;
        }
        if (batchSize > 0) {
            batch.submit();
            deleteSucceeded &= results.stream().allMatch(r -> r != null && (Boolean)r.get() != false);
        }
        return deleteSucceeded;
    }

    private boolean copyFile(GcsUri srcUri, GcsUri dstUri) throws IOException {
        Blob blob = this.getBlob(srcUri);
        Blob newBlob = this._storage.create(BlobInfo.newBuilder(BlobId.of(dstUri.getBucketName(), dstUri.getPath())).build(), new byte[0], new Storage.BlobTargetOption[0]);
        CopyWriter copyWriter = blob.copyTo(newBlob.getBlobId(), new Blob.BlobSourceOption[0]);
        copyWriter.getResult();
        return copyWriter.isDone() && blob.exists(new Blob.BlobSourceOption[0]);
    }

    private boolean copy(GcsUri srcUri, GcsUri dstUri) throws IOException {
        if (!this.exists(srcUri)) {
            throw new IOException(String.format("Source URI '%s' does not exist", srcUri));
        }
        if (srcUri.equals(dstUri)) {
            return true;
        }
        if (!this.existsDirectoryOrBucket(srcUri)) {
            return this.copyFile(srcUri, dstUri);
        }
        if (srcUri.hasSubpath(dstUri) || dstUri.hasSubpath(srcUri)) {
            throw new IOException(String.format("Cannot copy from or to a subdirectory: '%s' -> '%s'", srcUri, dstUri));
        }
        if (!this.existsDirectoryOrBucket(dstUri)) {
            this.mkdir(dstUri.getUri());
        }
        boolean copySucceeded = true;
        for (String directoryEntry : this.listFilesFromGcsUri(srcUri, true)) {
            GcsUri srcFile = new GcsUri(URI.create(directoryEntry));
            String relativeSrcPath = srcUri.relativize(srcFile);
            GcsUri dstFile = dstUri.resolve(relativeSrcPath);
            if (this.isPathTerminatedByDelimiter(srcFile)) {
                copySucceeded &= this.mkdir(dstFile.getUri());
                continue;
            }
            copySucceeded &= this.copyFile(srcFile, dstFile);
        }
        return copySucceeded;
    }
}

